%feature("docstring") OT::LatentVariableModel
R"RAW(Latent variable covariance function.

Parameters
----------
nLevels : int
    Number of levels :math:`\ell` characterizing the categorical variable.
latentDim : int
    The dimension :math:`d_\ell` of the latent space onto which the categorical variable levels are projected.

Notes
-----
The *Latent variable covariance function* is a covariance model allowing to compute the covariance between 
different unordered values (or levels) of a categorical variable :math:`z`.

The underlying idea is that each categorical level is mapped onto a distinct point in a :math:`d_l`-dimensional latent space. 
The covariance between the various levels is then computed as the :class:`~openturns.SquaredExponential` covariance between 
the mappings in the latent space.
Let :math:`\phi(\cdot):\mathcal{Z}\rightarrow\mathbb{R}^{d_l}` be the mapping function, the covariance function between
two discrete values :math:`z_i` and :math:`z_j` is computed as:

.. math::
    C(z_i, z_j) = e^{-\frac{1}{2}  \left\| \vect{s}- \vect{t} \right\|_{2}^{2}}, \quad \forall (\phi(z_i), \phi(z_j)) \in \cD

The coordinates of the mapping points are part of the covariance model parameters, together with the latent squared 
exponential model scale and amplitude. It is important to note that in order to compensate for possible rotations 
and translations of the mapping points, the coordinates of the first level mapping are fixed to the latent space 
origin, whereas all of the coordinates of the second level mapping are fixed to :math:`0`, except for the first one. 
As a result, the number of active latent variable coordinates is equal to:

.. math::
    d_\ell (\ell - 2) + 1.

In practice, the class distinguishes between the `fullLatentVariables` attribute, which contains the actual latent variables
coordinates, and the `activeLatentVariables` attribute, which contains only the coordinates that can be modified.
Additional information can be found in [zhang2020]_.

Is is important to note that for the sake of simplicity, the categorical variable levels **must be** represented as integers, 
ranging from :math:`0` to :math:`l-1`. However, this representation is purely practical, and the actual values
assigned to each level have no practical meaning or effect: only the latent variables coordinates have an effect on the
covariance value. Moreover, these categorical variables, which are encoded using numerical values, can be of a non-numerical
nature (e.g., types of material, architectural choices, colors, etc.).

Finally, for a similar reason, when using this type of kernel when defining a Gaussian process, it is suggested to 
rely on a constant functional basis: please use the :class:`~openturns.ConstantBasisFactory` class.


See Also
--------
CovarianceModel

Examples
--------
Create a latent model covariance function with a latent space of dimension 2, for a categorical variable 
characterized by 3 levels:

>>> import openturns as ot
>>> covModel = ot.LatentVariableModel(3, 2)
>>> activeCoordinates = [0.1, 0.3, -0.4]
>>> covModel.setLatentVariables(activeCoordinates)
>>> print(covModel(1, 2))
[[ 0.904837 ]]
>>> print(covModel(0, 2))
[[ 0.882497 ]]
>>> print(covModel(1, 1))
[[ 1 ]])RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::computeAsScalar
"Compute the covariance function for scalar model.

Parameters
----------
s, t : floats
	Must have integer values between :math:`0` and :math:`l-1`

Returns
-------
covariance : float
    Covariance.

Notes
-----
The method makes sense only if the dimension of the process is :math:`d=1`.
It evaluates :math:`C(s, t)`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getActiveLatentVariables
"Active latent variables accessor.

Parameters
----------
activeLatentVariables : :class:`~openturns.Point`
    Active coordinates of the categorical levels in the latent space. 
    The inactive coordinates are set to 0 (i.e., the first latent 
    variable is projected onto the Euclidean space origin, and the second
    latent variable can only be located along the first axis)."

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getFullLatentVariables
"Latent variables accessor.

Returns
-------
activeLatentVariables : :class:`~openturns.Point`
    Coordinates of the categorical levels in the latent space."

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getLatentDimension
"Latent dimension accessor.

Returns
-------
latentDimension : int
    Dimension of the latent space."

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getNLevels
"Number of levels accessor.

Returns
-------
nLevels : int
    Number of levels characterizing the categorical variable."

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::setFullParameter
R"RAW(Set the full parameters of the covariance function.

Parameters
----------
parameter : :class:`~openturns.Point`
    List of the full parameter of the covariance function i.e.
    the scale parameter :math:`\vect{\theta} \in \Rset`, the
    the amplitude parameter :math:`\vect{\sigma} \in \Rset`, and
    the latent variables coordinates,  :math:`\vect{x}_{lat} \in \Rset^{latentDim * nLevels - 2 * latentDim + 1}`)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getFullParameter
R"RAW(Get the full parameters of the covariance function.

Returns
-------
parameter : :class:`~openturns.Point`
    List of the full parameter of the covariance function i.e.
    the scale parameter :math:`\vect{\theta} \in \Rset`, the
    the amplitude parameter :math:`\vect{\sigma} \in \Rset`, and
    the latent variables coordinates,  :math:`\vect{x}_{lat} \in \Rset^{latentDim * nLevels - 2 * latentDim + 1}`)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::setParameter
R"RAW(Set the active parameters values of the covariance function.

Parameters
----------
parameter : :class:`~openturns.Point`
    List of the parameter of the covariance function, among which there are
    the scale parameter :math:`\vect{\theta} \in \Rset`, the
    the amplitude parameter :math:`\vect{\sigma} \in \Rset`, and
    the latent variables coordinates,  :math:`\vect{x}_{lat} \in \Rset^{latentDim * nLevels - 2 * latentDim + 1}`)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getParameter
R"RAW(Get the active parameters values of the covariance function.

Returns
-------
parameter : :class:`~openturns.Point`
    List of the full parameter of the covariance function, among which there are
    the scale parameter :math:`\vect{\theta} \in \Rset`, the
    the amplitude parameter :math:`\vect{\sigma} \in \Rset`, and
    the latent variables coordinates,  :math:`\vect{x}_{lat} \in \Rset^{latentDim * nLevels - 2 * latentDim + 1}`)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::setLatentVariables
"Number of levels accessor.

Parameters
----------
LatentVariables : sequence of float 
    Active coordinates of the categorical levels in the latent space."

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::draw
"Not yet implemented"

// ---------------------------------------------------------------------

%feature("docstring") OT::LatentVariableModel::getLevelNumber
R"RAW(Level number accessor.

Returns
-------
nLevels : int
    Number of levels :math:`\ell` characterizing the categorical variable.
)RAW"

