/*
* @license Apache-2.0
*
* Copyright (c) 2019 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/**
* Evaluates the natural logarithm of the probability density function (logPDF) for a Student's t distribution.
*
* @param x - input value
* @returns evaluated logPDF
*/
type Unary = ( x: number ) => number;

/**
* Interface for the natural logarithm of the probability density function (logPDF) of a Student's t distribution.
*/
interface LogPDF {
	/**
	* Evaluates the natural logarithm of the probability density function (PDF) for a Student's t distribution with degrees of freedom `v` at a value `x`.
	*
	* ## Notes
	*
	* -   If provided a non-positive value for `v`, the function returns `NaN`.
	*
	* @param x - input value
	* @param v - degrees of freedom
	* @returns evaluated logPDF
	*
	* @example
	* var y = logpdf( 0.3, 4.0 );
	* // returns ~-1.036
	*
	* @example
	* var y = logpdf( 2.0, 0.7 );
	* // returns ~-2.841
	*
	* @example
	* var y = logpdf( -1.0, 0.5 );
	* // returns ~-2.134
	*
	* @example
	* var y = logpdf( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = logpdf( NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var y = logpdf( 2.0, -1.0 );
	* // returns NaN
	*/
	( x: number, v: number ): number;

	/**
	* Returns a function for evaluating the natural logarithm of the probability density function (PDF) for a Student's t distribution with `v` degrees of freedom.
	*
	* @param v - degrees of freedom
	* @returns logPDF
	*
	* @example
	* var mylogPDF = logpdf.factory( 1.0 );
	* var y = mylogPDF( 3.0 );
	* // returns ~-3.447
	*
	* y = mylogPDF( 1.0 );
	* // returns ~-1.838
	*/
	factory( v: number ): Unary;
}

/**
* Student's t distribution natural logarithm of probability density function (logPDF).
*
* @param x - input value
* @param v - degrees of freedom
* @returns evaluated logPDF
*
* @example
* var y = logpdf( 3.0, 1.0 );
* // returns ~-3.442
*
* var mylogPDF = logpdf.factory( 3.0 );
* y = mylogPDF( 1.0 );
* // returns ~-1.575
*/
declare var logPDF: LogPDF;


// EXPORTS //

export = logPDF;
