/* tcl_ws-ftypes.c

   Implementation of Tcl object types needed to comfortably and efficiently
   support libwireshark field types.

   Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013 Eloy Paris

   Some code here has been borrowed from the Wireshark source code, and
   is copyright Gerald Combs and others.

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

Discussion:

The basic Tcl integer types are insufficient to represent libwireshark
field types. There are a couple of reasons for this: first, the basic
Tcl integer types are always signed integers, and lack unsigned types.
Second, while the size in bits of the basic Tcl integer types is great
(32, 64 and multi-precision integers), it is not easy to control
portability. Finally, while not integer types, some other key basic
types, like Ethernet, IPX, IPv4, IPv6, etc. are not provided by basic
Tcl types.

Testing on 32-bit Linux reveals the following ranges:

int: -2147483648 (INT32_MIN) to 2147483647 (INT32_MAX)
wide int: (-9223372036854775808 to -2147483649) && (2147483648 to 9223372036854775807 (INT64_MAX) )
bignum: <= -9223372036854775809 && >= 9223372036854775808

Tcl_IntObj(3tcl) states:

"The C integral types for which Tcl provides value exchange routines
are int, long int, Tcl_WideInt, and mp_int. The int and long int types
are provided by the C language standard. The Tcl_WideInt type is a
typedef defined to be whatever signed integral type covers at least the
64-bit integer range (-9223372036854775808 to 9223372036854775807).
Depending on the platform and the C compiler, the actual type might
be long int, long long int, int64, or something else. The mp_int
type is a multiple-precision integer type defined by the LibTomMath
multiple-precision integer library."

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <inttypes.h> /* For the PRIxxx macros and intxx_t/uintxx_t types. */
#include <string.h>
#include <tcl.h>
#include <dnet.h>

#include "util.h"
#include "missing.h"


/********************************************************************
 *		     The "uint32" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void uint32_update_string(Tcl_Obj *);
static void uint32_free_int_rep(Tcl_Obj *);
static void uint32_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclUInt32Type = {
    .name = "uint32",
    .freeIntRepProc = &uint32_free_int_rep,
    .dupIntRepProc = &uint32_dup_int_rep,
    .updateStringProc = &uint32_update_string,
    .setFromAnyProc = NULL /* Tcl_ObjType(3tcl) says that setFromAnyProc
			      may be set to NULL. */
};

static void
uint32_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    uint32_t *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (uint32_t *) ckalloc(sizeof(*new) );

    *new = *old;

    dupPtr->typePtr = &tclUInt32Type;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
uint32_update_string(Tcl_Obj *uint32_obj)
{
    size_t len;
    char buf[16]; /* Max. number of chars. will be for "4294967295" */

    snprintf(buf, sizeof(buf), "%" PRIu32,
	     *(uint32_t *) uint32_obj->internalRep.otherValuePtr);

    len = strlen(buf);

    uint32_obj->bytes = ckalloc(len + 1);
    if (!uint32_obj->bytes)
	return;

    strlcpy(uint32_obj->bytes, buf, len + 1);
    uint32_obj->length = len;
}

static void
uint32_free_int_rep(Tcl_Obj *obj)
{
    ckfree(obj->internalRep.otherValuePtr);

    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewUInt32Obj(uint32_t u32)
{
    Tcl_Obj *obj;
    uint32_t *u32ptr;

    obj = Tcl_NewObj();

    u32ptr = (uint32_t *) ckalloc(sizeof(*u32ptr) );
    *u32ptr = u32;

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclUInt32Type;
    obj->internalRep.otherValuePtr = u32ptr;

    return obj;
}

int
Tcl_GetUInt8FromObj(Tcl_Obj *obj, uint8_t *u8ptr)
{
    uint32_t u32;

    if (obj->typePtr != &tclUInt32Type)
	return TCL_ERROR;

    u32 = *(uint32_t *) obj->internalRep.otherValuePtr;
    *u8ptr = u32;

    return TCL_OK;
}

int
Tcl_GetUInt16FromObj(Tcl_Obj *obj, uint16_t *u16ptr)
{
    uint32_t u32;

    /* I am not sure that we need this check... with it, if obj is a list,
     * we fail the check and return without storing a number in u16ptr,
     * which then causes everything else that depends on correct reading
     * of the value to fail, e.g. isanswer, etc. Without the check things
     * seem to be working fine. Not sure what the right thing to do is.
     * Leaving this commented out for now but the other types still have
     * the check.
     *
     * The problem was uncovered when libwireshark switched icmp.ident
     * from a single value to a list (for big endian and little endian
     * representations) and Network Expect's send_expect command stopped
     * working as a result. This apparently happened in Wireshark 1.8.
     *
     * peloy@nexexpect.org
     * Mon Jul 15 23:40:31 EDT 2013
     */
#if 0
    if (obj->typePtr != &tclUInt32Type)
	return TCL_ERROR;
#endif

    u32 = *(uint32_t *) obj->internalRep.otherValuePtr;
    *u16ptr = u32;

    return TCL_OK;
}

int
Tcl_GetUInt32FromObj(Tcl_Obj *obj, uint32_t *u32ptr)
{
    if (obj->typePtr != &tclUInt32Type)
	return TCL_ERROR;

    *u32ptr = *(uint32_t *) obj->internalRep.otherValuePtr;

    return TCL_OK;
}
/********************************************************************
 *		      The "int32" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void int32_update_string(Tcl_Obj *);
static void int32_free_int_rep(Tcl_Obj *);
static void int32_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclInt32Type = {
    .name = "int32",
    .freeIntRepProc = &int32_free_int_rep,
    .dupIntRepProc = &int32_dup_int_rep,
    .updateStringProc = &int32_update_string,
    .setFromAnyProc = NULL /* Tcl_ObjType(3tcl) says that setFromAnyProc
			      may be set to NULL. */
};

static void
int32_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    int32_t *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (int32_t *) ckalloc(sizeof(*new) );

    *new = *old;

    dupPtr->typePtr = &tclInt32Type;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
int32_update_string(Tcl_Obj *int32_obj)
{
    size_t len;
    char buf[16]; /* Max. number of chars. will be for "-2147483648" */

    snprintf(buf, sizeof(buf), "%" PRId32,
	     *(int32_t *) int32_obj->internalRep.otherValuePtr);

    len = strlen(buf);

    int32_obj->bytes = ckalloc(len + 1);
    if (!int32_obj->bytes)
	return;

    strlcpy(int32_obj->bytes, buf, len + 1);
    int32_obj->length = len;
}

static void
int32_free_int_rep(Tcl_Obj *obj)
{
    ckfree(obj->internalRep.otherValuePtr);

    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewInt32Obj(int32_t i32)
{
    Tcl_Obj *obj;
    int32_t *i32ptr;

    obj = Tcl_NewObj();

    i32ptr = (int32_t *) ckalloc(sizeof(*i32ptr) );
    *i32ptr = i32;

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclInt32Type;
    obj->internalRep.otherValuePtr = i32ptr;

    return obj;
}

int
Tcl_GetInt32FromObj(Tcl_Obj *obj, int32_t *i32ptr)
{
    if (obj->typePtr != &tclInt32Type)
	return TCL_ERROR;

    *i32ptr = *(int32_t *) obj->internalRep.otherValuePtr;

    return TCL_OK;
}

/********************************************************************
 *		     The "uint64" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void uint64_update_string(Tcl_Obj *);
static void uint64_free_int_rep(Tcl_Obj *);
static void uint64_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclUInt64Type = {
    .name = "uint64",
    .freeIntRepProc = &uint64_free_int_rep,
    .dupIntRepProc = &uint64_dup_int_rep,
    .updateStringProc = &uint64_update_string,
    .setFromAnyProc = NULL /* Tcl_ObjType(3tcl) says that setFromAnyProc
			      may be set to NULL. */
};

static void
uint64_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    uint64_t *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (uint64_t *) ckalloc(sizeof(*new) );

    *new = *old;

    dupPtr->typePtr = &tclUInt64Type;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
uint64_update_string(Tcl_Obj *uint64_obj)
{
    size_t len;
    char buf[24]; /* Max. number of chars. will be for "18446744073709551615" */

    snprintf(buf, sizeof(buf), "%" PRIu64,
	     *(uint64_t *) uint64_obj->internalRep.otherValuePtr);

    len = strlen(buf);

    uint64_obj->bytes = ckalloc(len + 1);
    if (!uint64_obj->bytes)
	return;

    strlcpy(uint64_obj->bytes, buf, len + 1);
    uint64_obj->length = len;
}

static void
uint64_free_int_rep(Tcl_Obj *obj)
{
    ckfree(obj->internalRep.otherValuePtr);

    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewUInt64Obj(uint64_t u64)
{
    Tcl_Obj *obj;
    uint64_t *u64ptr;

    obj = Tcl_NewObj();

    u64ptr = (uint64_t *) ckalloc(sizeof(*u64ptr) );
    *u64ptr = u64;

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclUInt64Type;
    obj->internalRep.otherValuePtr = u64ptr;

    return obj;
}

int
Tcl_GetUInt64FromObj(Tcl_Obj *obj, uint64_t *u64ptr)
{
    if (obj->typePtr != &tclUInt64Type)
	return TCL_ERROR;

    *u64ptr = *(uint64_t *) obj->internalRep.otherValuePtr;

    return TCL_OK;
}

/********************************************************************
 *		      The "int64" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void int64_update_string(Tcl_Obj *);
static void int64_free_int_rep(Tcl_Obj *);
static void int64_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclInt64Type = {
    .name = "int64",
    .freeIntRepProc = &int64_free_int_rep,
    .dupIntRepProc = &int64_dup_int_rep,
    .updateStringProc = &int64_update_string,
    .setFromAnyProc = NULL /* Tcl_ObjType(3tcl) says that setFromAnyProc
			      may be set to NULL. */
};

static void
int64_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    int64_t *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (int64_t *) ckalloc(sizeof(*new) );

    *new = *old;

    dupPtr->typePtr = &tclInt64Type;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
int64_update_string(Tcl_Obj *int64_obj)
{
    size_t len;
    char buf[24]; /* Max. number of chars. will be for "-9223372036854775808" */

    snprintf(buf, sizeof(buf), "%" PRId64,
	     *(int64_t *) int64_obj->internalRep.otherValuePtr);

    len = strlen(buf);

    int64_obj->bytes = ckalloc(len + 1);
    if (!int64_obj->bytes)
	return;

    strlcpy(int64_obj->bytes, buf, len + 1);
    int64_obj->length = len;
}

static void
int64_free_int_rep(Tcl_Obj *obj)
{
    ckfree(obj->internalRep.otherValuePtr);

    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewInt64Obj(int64_t i64)
{
    Tcl_Obj *obj;
    int64_t *i64ptr;

    obj = Tcl_NewObj();

    i64ptr = (int64_t *) ckalloc(sizeof(*i64ptr) );
    *i64ptr = i64;

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclInt64Type;
    obj->internalRep.otherValuePtr = i64ptr;

    return obj;
}

int
Tcl_GetInt64FromObj(Tcl_Obj *obj, int64_t *i64ptr)
{
    if (obj->typePtr != &tclInt64Type)
	return TCL_ERROR;

    *i64ptr = *(int64_t *) obj->internalRep.otherValuePtr;

    return TCL_OK;
}

/********************************************************************
 *		      The "addr" Tcl object type                  *
 ********************************************************************/

/* Forward declarations */
static void addr_update_string(Tcl_Obj *);
static void addr_free_int_rep(Tcl_Obj *);
static void addr_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclAddrType = {
    .name = "addr",
    .freeIntRepProc = &addr_free_int_rep,
    .dupIntRepProc = &addr_dup_int_rep,
    .updateStringProc = &addr_update_string,
    .setFromAnyProc = NULL /* Tcl_ObjType(3tcl) says that setFromAnyProc
			      may be set to NULL. */
};

static void
addr_dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    struct addr *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (struct addr *) ckalloc(sizeof(*new) );

    *new = *old;

    dupPtr->typePtr = &tclAddrType;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
addr_update_string(Tcl_Obj *addr_obj)
{
    size_t len;
    struct addr *a;
    char *s;

    a = addr_obj->internalRep.otherValuePtr;

    switch (a->addr_type) {
    case ADDR_TYPE_ETH:
	s = mac_to_ascii(&a->addr_eth);
	break;
    case ADDR_TYPE_IP:
	s = ipaddr2str(a->addr_ip);
	break;
    case ADDR_TYPE_IP6:
	s = ip6addr2str(a->addr_ip6);
	break;
    }

    len = strlen(s);

    addr_obj->bytes = ckalloc(len + 1);
    if (!addr_obj->bytes)
	return;

    strlcpy(addr_obj->bytes, s, len + 1);
    addr_obj->length = len;
}

static void
addr_free_int_rep(Tcl_Obj *obj)
{
    ckfree(obj->internalRep.otherValuePtr);

    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewAddrObj(struct addr *a)
{
    Tcl_Obj *obj;
    struct addr *p;

    obj = Tcl_NewObj();

    p = (struct addr *) ckalloc(sizeof(*p) );
    *p = *a;

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclAddrType;
    obj->internalRep.otherValuePtr = p;

    return obj;
}

int
Tcl_GetAddrFromObj(Tcl_Obj *obj, struct addr *addr)
{
    if (obj->typePtr != &tclAddrType)
	return TCL_ERROR;

    *addr = *(struct addr *) obj->internalRep.otherValuePtr;

    return TCL_OK;
}

int
Tcl_GetIPAddrFromObj(Tcl_Obj *obj, ip_addr_t *ipaddr)
{
    if (obj->typePtr != &tclAddrType)
	return TCL_ERROR;

    *ipaddr = ( (struct addr *) obj->internalRep.otherValuePtr)->addr_ip;

    return TCL_OK;
}

/********************************************************************/

void
nexp_register_tcl_ws_ftypes(void)
{
    Tcl_RegisterObjType(&tclUInt32Type);
    Tcl_RegisterObjType(&tclInt32Type);

    Tcl_RegisterObjType(&tclUInt64Type);
    Tcl_RegisterObjType(&tclInt64Type);

    Tcl_RegisterObjType(&tclAddrType);
}
