/* pdu-arp.c
  
   PDU builder for ARP requests

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"

static void
builder(const GNode *pdu, void *dest)
{
    struct arp_hdr *arp;
    struct arp_ethip *ethip;
    ip_addr_t tmp;
    uint16_t operation;
    const char *pdu_name;
    void *field;

    arp = dest;
    ethip = (struct arp_ethip *) ( (uint8_t *) arp + sizeof(struct arp_hdr) );

    SSVAL(arp, offsetof(struct arp_hdr, ar_hrd), htons(ARP_HRD_ETH) );
    SSVAL(arp, offsetof(struct arp_hdr, ar_pro), htons(ETH_TYPE_IP) );
    arp->ar_hln = ETH_ADDR_LEN;
    arp->ar_pln = sizeof(ip_addr_t);

    pdu_name = pb_getname(pdu);
    if (!strcmp(pdu_name, "arp") || !strcmp(pdu_name, "rarp") )
	operation = num_next(_pb_pdata(pdu, "op") );
    else if (!strcmp(pdu_name, "arp-reply") )
	operation = ARP_OP_REPLY;
    else if (!strcmp(pdu_name, "rarp-reply") )
	operation = ARP_OP_REVREPLY;
    else if (!strcmp(pdu_name, "arp-request") )
	operation = ARP_OP_REQUEST;
    else if (!strcmp(pdu_name, "rarp-request") )
	operation = ARP_OP_REVREQUEST;
    SSVAL(arp, offsetof(struct arp_hdr, ar_op), htons(operation) );

    /*
     * XXX it'd be nice to have this default to the MAC address
     * of some interface (but which one?)
     */
    if ( (field = _pb_pdata(pdu, "sha") ) )
	memcpy(ethip->ar_sha, field, ETH_ADDR_LEN);
    else
	memset(ethip->ar_sha, 0, ETH_ADDR_LEN);

    tmp = ip_next(_pb_pdata(pdu, "sip") );
    SIVAL(ethip->ar_spa, 0, tmp);

    if ( (field = _pb_pdata(pdu, "tha") ) )
	memcpy(ethip->ar_tha, field, ETH_ADDR_LEN);
    else
	memset(ethip->ar_tha, 0, ETH_ADDR_LEN);

    tmp = ip_next(_pb_pdata(pdu, "tip") );
    SIVAL(ethip->ar_tpa, 0, tmp);
}

static pdu_t pdu_arp = {
    .name = "arp",
    .description = "ARP packet",
    .documented_in = "RFC 826",
    .len = sizeof(struct arp_hdr) + sizeof(struct arp_ethip),
    .fields = (field_t []) {
	{.name = "sha", .type = PDU_FTYPE_MACADDR},
	{.name = "sip", .type = PDU_FTYPE_IP},
	{.name = "tha", .type = PDU_FTYPE_MACADDR},
	{.name = "tip", .type = PDU_FTYPE_IP},
	{.name = "op", .type = PDU_FTYPE_UINT16},
	{.name = NULL}
    },
    .build = &builder
};

static pdu_t pdu_rarp = {
    .name = "rarp",
    .description = "Reverse ARP packet",
    .documented_in = "RFC 826",
    .len = sizeof(struct arp_hdr) + sizeof(struct arp_ethip),
    .fields = (field_t []) {
	{.name = "sha", .type = PDU_FTYPE_MACADDR},
	{.name = "sip", .type = PDU_FTYPE_IP},
	{.name = "tha", .type = PDU_FTYPE_MACADDR},
	{.name = "tip", .type = PDU_FTYPE_IP},
	{.name = "op", .type = PDU_FTYPE_UINT16},
	{.name = NULL}
    },
    .build = &builder
};

static pdu_t pdu_arp_request = {
    .name = "arp-request",
    .description = "ARP request packet",
    .documented_in = "RFC 826",
    .len = sizeof(struct arp_hdr) + sizeof(struct arp_ethip),
    .fields = (field_t []) {
	{.name = "sha", .type = PDU_FTYPE_MACADDR},
	{.name = "sip", .type = PDU_FTYPE_IP},
	{.name = "tha", .type = PDU_FTYPE_MACADDR},
	{.name = "tip", .type = PDU_FTYPE_IP},
	{.name = NULL}
    },
    .build = &builder
};

static pdu_t pdu_rarp_request = {
    .name = "rarp-request",
    .description = "Reverse ARP request packet",
    .documented_in = "RFC 826",
    .len = sizeof(struct arp_hdr) + sizeof(struct arp_ethip),
    .fields = (field_t []) {
	{.name = "sha", .type = PDU_FTYPE_MACADDR},
	{.name = "sip", .type = PDU_FTYPE_IP},
	{.name = "tha", .type = PDU_FTYPE_MACADDR},
	{.name = "tip", .type = PDU_FTYPE_IP},
	{.name = NULL}
    },
    .build = &builder
};

static pdu_t pdu_arp_reply = {
    .name = "arp-reply",
    .description = "ARP reply packet",
    .documented_in = "RFC 826",
    .len = sizeof(struct arp_hdr) + sizeof(struct arp_ethip),
    .fields = (field_t []) {
	{.name = "sha", .type = PDU_FTYPE_MACADDR},
	{.name = "sip", .type = PDU_FTYPE_IP},
	{.name = "tha", .type = PDU_FTYPE_MACADDR},
	{.name = "tip", .type = PDU_FTYPE_IP},
	{.name = NULL}
    },
    .build = &builder
};

static pdu_t pdu_rarp_reply = {
    .name = "rarp-reply",
    .description = "Reverse ARP reply packet",
    .documented_in = "RFC 826",
    .len = sizeof(struct arp_hdr) + sizeof(struct arp_ethip),
    .fields = (field_t []) {
	{.name = "sha", .type = PDU_FTYPE_MACADDR},
	{.name = "sip", .type = PDU_FTYPE_IP},
	{.name = "tha", .type = PDU_FTYPE_MACADDR},
	{.name = "tip", .type = PDU_FTYPE_IP},
	{.name = NULL}
    },
    .build = &builder
};

void
_pb_register_arp(void)
{
    _pb_register_protocol(&pdu_arp);
    _pb_register_protocol(&pdu_rarp);
    _pb_register_protocol(&pdu_arp_request);
    _pb_register_protocol(&pdu_arp_reply);
    _pb_register_protocol(&pdu_rarp_request);
    _pb_register_protocol(&pdu_rarp_reply);
}
