/*
  Copyright (C) 2008 Steffen Basting, André Gaul, Jan Friederich

        This file is part of mmpong.

        mmpong is free software: you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation, either version 3 of the License, or
        (at your option) any later version.

        mmpong is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef WIN32
# ifndef _WIN32_WINNT
#  define _WIN32_WINNT 0x0501 //assume windows xp or above
# endif
# include <winsock2.h>
# include <ws2tcpip.h>
#endif

#ifdef __APPLE__
#include <CoreFoundation/CoreFoundation.h>
#endif

#include <signal.h>
#include <getopt.h>
#include <errno.h>
#include <SDL_mixer.h>
#include <iostream>

#include "clientstate.h"
#include "config.h"
#include "gui.h"
#include "netgame.h"
#include "renderer.h"
#include "resource.h"

using namespace std;

#define DEFAULT_WINDOW_TITLE "mmpong-gl"
#define DEFAULT_WINDOW_WIDTH 800
#define DEFAULT_WINDOW_HEIGHT 600
#define DEFAULT_FRAMERATE 30

#define SIGN(sgn_u)  ((sgn_u < 0) ? -1 : 1)


// Audio stuff. TODO: do it nicely ;-)
static Mix_Chunk *mx_pong1, *mx_pong2, *mx_score;



// signal handler for POSIX signals
#ifndef WIN32
ClientState *globalstate;
void sigterm_handler(int signal)
{
	if (signal == SIGPIPE)
		printf("Caught SIGPIPE.\n");
	if (signal == SIGURG)
		printf("Caught SIGURG.\n");
	if (signal == SIGTERM)
		globalstate->signal_exiting= 1;
}
#endif


// command line options parser
static int parse_options(int argc, char **argv, ClientState *state) {
	const char *short_options= "szh";
	const struct option long_options[]= {
		{ "sexy", no_argument, NULL, 's' },
		{ "nosound", no_argument, NULL, 'z' },
		{ "help", no_argument, NULL, 'h' },
		{ NULL, 0, NULL, '\0' }
	};
	const char *options_desc[]= {
		"Enable graphics effects (fast hardware recommended)",
		"Refrain from playing annoying sounds (your co-workers will appreciate it)",
		"Display this message and exit",
		NULL
	};

	extern int optind;
	int opt, helpscreen= 0;
	while ((opt = getopt_long(argc, argv, short_options, long_options, NULL))) {
		if (opt == (-1)) break;

		switch(opt) {
		case 's':
			state->sexy= true;
			break;
		case 'z':
			state->sound= false;
			break;
		case 'h':
			helpscreen= true;
			break;
		case '?':
			exit(1);
			break; // :-)
		default:
			exit(1);
			break; // :-)
		}
	}

	if (/*optind >= argc ||*/ argc - optind > 2 || helpscreen) {
		printf("mmpong-gl client %d.%02d (r%s.c%d.l%d)\nReleased under the terms of the GPL (http://www.gnu.org/licenses/gpl.html).\n", VER_MAJ, VER_MIN, VER_SVN, (int)sizeof(struct gameplay), gameplay_getversion());
		printf("Usage: %s", argv[0]);
		for (int i = 0; long_options[i].name; ++i)
			printf(" [--%s]", long_options[i].name);
		printf(" [server [port]]\n");
		for (int i = 0; options_desc[i] && long_options[i].name; ++i) {
			printf("\t-%c, --%s:\t%s\n",
				long_options[i].val,
				long_options[i].name,
				options_desc[i]);
		}
		exit(1);
	}

	return optind;
}


int init_sound() {
	int audio_rate = 22050;
	Uint16 audio_format = AUDIO_S16;
	int audio_channels = 2;
	int audio_buffers = 512;

	if (Mix_OpenAudio(audio_rate, audio_format, audio_channels, audio_buffers))
	{
		fprintf(stderr, "SDL_mixer: Unable to use Mix_OpenAudio!\n");
		//return PONG_FAILURE;
	}

	mx_pong1 = Mix_LoadWAV(RESOURCE("sounds/pong_1.wav"));
	mx_pong2 = Mix_LoadWAV(RESOURCE("sounds/pong_2.wav"));
	mx_score = Mix_LoadWAV(RESOURCE("sounds/score.wav"));

	return PONG_SUCCESS;
}


int main(int argc, char *argv[])
{
	ClientState state;
# ifndef WIN32
		globalstate = &state;
# endif

	errno=0;
	int x, y;
	uint8_t *key_state;
	uint32_t mouse_state, score[2];

	float dir_old = 0.0;

#ifdef __APPLE__
	char respath[1024];
	CFBundleRef mainBundle = CFBundleGetMainBundle();
	assert(mainBundle);
	CFURLRef mainBundleURL = CFBundleCopyBundleURL(mainBundle);
	assert(mainBundleURL);
	CFStringRef cfStringRef = CFURLCopyFileSystemPath(mainBundleURL, kCFURLPOSIXPathStyle);
	assert(cfStringRef);
	CFStringGetCString(cfStringRef, respath, 1024, kCFStringEncodingASCII);
	CFRelease(mainBundleURL);
	CFRelease(cfStringRef);
	string respathpp(respath);
	respathpp += "/Contents/Resources/";
	
	RESOURCE(respathpp.c_str());
#endif

	int argopt = parse_options(argc, argv, &state);

#ifndef WIN32
	struct sigaction termact;
	termact.sa_handler= sigterm_handler;
	termact.sa_flags= SA_RESETHAND;
	sigemptyset(&termact.sa_mask);
	if ( (sigaction(SIGTERM, &termact, NULL)) ||
			(termact.sa_flags= SA_RESTART, sigaction(SIGPIPE, &termact, NULL)) ) {
		fprintf(stderr, "Error putting signal handlers in place.\n");
		exit(-1);
	}
#else
	//winsock init
	WSADATA wsaData;
	int wsaresult = WSAStartup(MAKEWORD(2,2), &wsaData);
	if (wsaresult != 0) {
		fprintf(stderr, "Error initializing Windows Sockets (code==%d)\n", wsaresult);
		exit(-1);
	}
#endif

	char *portname= (char *)"1212";
	if (argc - argopt > 1)
		portname= argv[argopt+1];
	if (argc - argopt > 0) {
		state.server = argv[argopt];
		state.port = portname;
	}
	
	Renderer renderer(DEFAULT_WINDOW_WIDTH, DEFAULT_WINDOW_HEIGHT, state.full, state.sexy, RESOURCE(""), DEFAULT_WINDOW_TITLE);
	state.renderer = &renderer;
	GUI gui(RESOURCE(""), renderer.get_width(), renderer.get_height(), true, &state);

	if (argc - argopt > 0) {
		// try to connect directly without showing a non-nerdy GUI
		gui.set_visible(false);
		try {
			state.game = new NetGame(state.server, state.port, 0);
		} catch (runtime_error &err) {
			cerr << err.what() << endl;
			gui.show_dialog("Error", err.what());
			gui.set_visible(true);
			delete state.game;
			state.game = NULL;
		}
	}

	float pos=0.5;
	int holdgrab= 0;

	// Save scores
	if (state.game)
		for (int idx=0;idx<=1;idx++) score[idx] = state.game->get_game()->pad_attr[idx].score;

	init_sound();
	

	const struct timeval time_wait = { 0, 1000 * 1000 / DEFAULT_FRAMERATE };
	struct timeval time_start, time_diff;
	memcpy(&time_diff, &time_wait, sizeof(struct timeval));

	while(!state.signal_exiting) {
		// check for time 'overflow'
		if (time_wait.tv_sec < time_diff.tv_sec || 
				(time_wait.tv_sec == time_diff.tv_sec && 
				 time_wait.tv_usec < time_diff.tv_usec) )
			// no more time left :( 
		{
			time_diff.tv_sec = 0;
			time_diff.tv_usec = 0;
		}
		else
			timeval_combine(-1, &time_diff, 1, &time_wait);

		if (state.game) {
			// update game (if available)
			try {
				state.game->update(&time_diff);
			} catch (runtime_error &err) {
				cerr << err.what() << endl;
				gui.show_dialog("Error", err.what());
				gui.set_visible(true);
				delete state.game;
				state.game = NULL;
			}
		}
		else
#ifdef WIN32
			Sleep( (time_wait.tv_sec * 1000) + (time_wait.tv_usec / 1000) );
#else
			usleep( ((long)time_wait.tv_sec) * 1000 * 1000 + time_wait.tv_usec );
#endif

		gettimeofday(&time_start, NULL);

		// get events
		SDL_Event event;
		while (SDL_PollEvent(&event)) {
			if (event.type == SDL_QUIT) 
				state.signal_exiting = 1;

			// handle 'important' events here

			if (event.type == SDL_KEYUP && event.key.keysym.sym == SDLK_ESCAPE)
				gui.set_visible(!gui.get_visible());

			if (event.type == SDL_VIDEORESIZE) {
				gui.resize_init();
				renderer.set_winsize(event.resize.w, event.resize.h);
				gui.resize_finish(event.resize.w, event.resize.h);
			}

			if (gui.get_visible()) {
				// gui handles events
				gui.handle_event(event);

			}	else {

				if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_SPACE) {
					holdgrab = !holdgrab;
					SDL_WM_GrabInput( holdgrab ? SDL_GRAB_ON : SDL_GRAB_OFF);
					SDL_WM_SetCaption( (holdgrab)? DEFAULT_WINDOW_TITLE" (Press space bar to release pointer)" : DEFAULT_WINDOW_TITLE, NULL);
				}
				
				if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_q) 
					state.signal_exiting = 1;
				
				if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_RETURN) 
					gui.set_full(!renderer.get_full());
			}
			
		}

		if (state.game) {
			// flush send buffer
			try {
				state.game->sendpos_flush();
				// process keyboard and mouse for position
				if (!gui.get_visible()) {
					// get position
					bool newpos=false;
					key_state = SDL_GetKeyState(NULL);
					mouse_state = SDL_GetRelativeMouseState(&x, &y);

					if (y != 0) {
						pos -= 2.0f*((float)(y)/600);
						newpos = true;
					}

					if (key_state[SDLK_UP]) {
						pos += 0.05f;//state.game->get_game()->pad[team].size / 2;
						newpos = true;
					}

					if (key_state[SDLK_DOWN]) {
						pos -= 0.05f;//state.game->get_game()->pad[team].size / 2;
						newpos = true;
					}

					if (pos > 1.0f - state.game->get_game()->pad[state.game->get_team()].size/2.0f)
						pos = 1.0f - state.game->get_game()->pad[state.game->get_team()].size/2.0f;
					
					if (pos < state.game->get_game()->pad[state.game->get_team()].size/2.0f)
						pos = state.game->get_game()->pad[state.game->get_team()].size/2.0f;

					// send position
					if (newpos) 
						state.game->sendpos(pos);
				}
			}
			catch (runtime_error &err) {
				cerr << err.what() << endl;
				gui.show_dialog("Error", err.what());
				gui.set_visible(true);
				delete state.game;
				state.game = NULL;
			}
			// render game (if available)
			// note: do not combine with the code above! (game maybe null now!)
			try {
				renderer.render(*state.game, pos);
			} catch (runtime_error &err) {
				// no way to go out smoothly if renderer is broken
				cerr << err.what() << endl;
				exit(-1);
			}
			// play score-sound
			for (int idx=0; state.sound && idx<=1;idx++)
				if (score[idx] != state.game->get_game()->pad_attr[idx].score) {
					Mix_PlayChannel(-1, mx_score, 0);
					score[idx] = state.game->get_game()->pad_attr[idx].score;
					break;
				}
			// play bounce-sound
			// Play annoying sounds when the ball bounces off the paddle
			if (state.sound && dir_old != 0.0 && (SIGN(dir_old) != SIGN(state.game->get_game()->ball.dir[0]))) {
				if (dir_old < 0.0) Mix_PlayChannel(-1, mx_pong1, 0);
				else Mix_PlayChannel(-1, mx_pong2, 0);
			}
			dir_old = state.game->get_game()->ball.dir[0];
		} else {
			// just make a blank screen with field texture
			// if no game is running
			renderer.render();
		}

		// this is kinda self-explanatory ;)
		gui.render();

		// show it
		SDL_GL_SwapBuffers();
		
		gettimeofday(&time_diff, NULL);
		timeval_combine(1, &time_diff, -1, &time_start);
	}

	Mix_CloseAudio();	// Close audio stuff
#ifdef WIN32
	WSACleanup();
#endif

	return 0;
}

