#include "protoApp.h"
#include "protoTimer.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

class TimerTestApp : public ProtoApp
{
    public:
      TimerTestApp();
      ~TimerTestApp();

      bool OnStartup(int argc, const char*const* argv);
      bool ProcessCommands(int argc, const char*const* argv);
      void OnShutdown();

    private:
      enum CmdType {CMD_INVALID, CMD_ARG, CMD_NOARG};
      static const char* const CMD_LIST[];
      static CmdType GetCmdType(const char* string);
      bool OnCommand(const char* cmd, const char* val);        
      void Usage();
      
      bool OnTimeout(ProtoTimer& theTimer);
      
      ProtoTimer    the_timer;
      
      bool          first_timeout;
      unsigned long timeout_count;
      double        ave_sum;
      double        squ_sum;
      double        delta_min;
      double        delta_max;
      double        elapsed_sec;
      
      struct timeval last_time;
      
      
      bool          use_nanosleep;
      

}; // end class TimerTestApp

void TimerTestApp::Usage()
{
    fprintf(stderr, "Usage: timerTest [help] interval <timerInterval>\n");
}

const char* const TimerTestApp::CMD_LIST[] =
{
    "-help",        // print help info an exit
    "+interval",    // <timerInterval>
    "-nanosleep",   
    "-priority", 
    "-precise",
    "+debug",       // <debugLevel>
    NULL
};

/**
 * This macro creates our ProtoApp derived application instance 
 */
PROTO_INSTANTIATE_APP(TimerTestApp) 

TimerTestApp::TimerTestApp()
    : first_timeout(true)
{
    the_timer.SetListener(this, &TimerTestApp::OnTimeout);
    the_timer.SetInterval(1.0);
    the_timer.SetRepeat(-1);
}

TimerTestApp::~TimerTestApp()
{
}

TimerTestApp::CmdType TimerTestApp::GetCmdType(const char* cmd)
{
    if (!cmd) return CMD_INVALID;
    unsigned int len = strlen(cmd);
    bool matched = false;
    CmdType type = CMD_INVALID;
    const char* const* nextCmd = CMD_LIST;
    while (*nextCmd)
    {
        if (!strncmp(cmd, *nextCmd+1, len))
        {
            if (matched)
            {
                // ambiguous command (command should match only once)
                return CMD_INVALID;
            }
            else
            {
                matched = true;   
                if ('+' == *nextCmd[0])
                    type = CMD_ARG;
                else
                    type = CMD_NOARG;
            }
        }
        nextCmd++;
    }
    return type; 
}  // end TimerTestApp::GetCmdType()


bool TimerTestApp::OnStartup(int argc, const char*const* argv)
{
    
    if (!ProcessCommands(argc, argv))
    {
        PLOG(PL_ERROR, "TimerTestApp::OnStartup() error processing command line options\n");
        return false;   
    }

#ifdef LINUX    
    if (use_nanosleep)
    {
        struct timespec tspec;
        double interval = the_timer.GetInterval();
        tspec.tv_sec = (unsigned long)interval;
        tspec.tv_nsec = (unsigned long)(1.0e+09 * (interval - (double)tspec.tv_sec));
        while (1)
        {
            OnTimeout(the_timer);
            clock_nanosleep(CLOCK_MONOTONIC, 0, &tspec, NULL);
        }
        return false;
            
    }
    else
#endif // LINUX
    {
        ActivateTimer(the_timer);
    }
    
    return true;
}  // end TimerTestApp::OnStartup()

void TimerTestApp::OnShutdown()
{
   PLOG(PL_ERROR, "timerTest: Done.\n"); 
}  // end TimerTestApp::OnShutdown()

bool TimerTestApp::ProcessCommands(int argc, const char*const* argv)
{
    // Dispatch command-line commands to our OnCommand() method
    int i = 1;
    while ( i < argc)
    {
        // Is it a timerTest command?
        switch (GetCmdType(argv[i]))
        {
            case CMD_INVALID:
            {
                PLOG(PL_ERROR, "TimerTestApp::ProcessCommands() Invalid command:%s\n", 
                        argv[i]);
                Usage();
                return false;
            }
            case CMD_NOARG:
                if (!OnCommand(argv[i], NULL))
                {
                    PLOG(PL_ERROR, "TimerTestApp::ProcessCommands() ProcessCommand(%s) error\n", 
                            argv[i]);
                    Usage();
                    return false;
                }
                i++;
                break;
            case CMD_ARG:
                if (!OnCommand(argv[i], argv[i+1]))
                {
                    PLOG(PL_ERROR, "TimerTestApp::ProcessCommands() ProcessCommand(%s, %s) error\n", 
                            argv[i], argv[i+1]);
                    Usage();
                    return false;
                }
                i += 2;
                break;
        }
    }
    return true;  
}  // end TimerTestApp::ProcessCommands()

bool TimerTestApp::OnCommand(const char* cmd, const char* val)
{
    // (TBD) move command processing into Mgen class ???
    CmdType type = GetCmdType(cmd);
    ASSERT(CMD_INVALID != type);
    size_t len = strlen(cmd);
    if ((CMD_ARG == type) && !val)
    {
        PLOG(PL_ERROR, "TimerTestApp::ProcessCommand(%s) missing argument\n", cmd);
        Usage();
        return false;
    }
    else if (!strncmp("help", cmd, len))
    {
        Usage();
        exit(0);
    }
    else if (!strncmp("nanosleep", cmd, len))
    {
        use_nanosleep = true;
    }
    else if (!strncmp("priority", cmd, len))
    {
        dispatcher.SetPriorityBoost(true);
        dispatcher.BoostPriority();
    }
    else if (!strncmp("precise", cmd, len))
    {
        dispatcher.SetPreciseTiming(true);
    }
    else if (!strncmp("interval", cmd, len))
    {
        float timerInterval;
        if (1 != sscanf(val, "%e", &timerInterval))
        {
            PLOG(PL_ERROR, "TimerTestApp::OnCommand(interval) error: invalid argument\n");
            return false;
        }
        the_timer.SetInterval((double)timerInterval);
    }
    else if (!strncmp("debug", cmd, len))
    {
        SetDebugLevel(atoi(val));
    }
    else
    {
        PLOG(PL_ERROR, "timerTest error: invalid command\n");
        Usage();
        return false;
    }
    return true;
}  // end TimerTestApp::OnCommand()


bool TimerTestApp::OnTimeout(ProtoTimer& /*theTimer*/)
{
    struct timeval currentTime;
    ProtoSystemTime(currentTime);
    if (first_timeout)
    {
        elapsed_sec = 0.0;
        first_timeout = false;
        ave_sum = 0.0;
        squ_sum = 0.0;
        timeout_count = 0;
        delta_min = 1000.0;
        delta_max = 0.0;
    }
    else
    {
        double delta = currentTime.tv_sec - last_time.tv_sec;
        delta += 1.0e-06 * (currentTime.tv_usec - last_time.tv_usec);
        
        if (delta > delta_max)
            delta_max = delta;
        if (delta < delta_min)
            delta_min = delta;
        
        elapsed_sec += delta;
        timeout_count++;
        
        ave_sum += delta;
        squ_sum += (delta *delta);
        
        if (elapsed_sec > 5.0)
        {
            double ave = ave_sum / timeout_count;
            double var = (squ_sum - (ave_sum * ave)) / timeout_count;
            
            TRACE("timer interval: ave>%lf min>%lf max>%lf var>%lf\n",
                    ave, delta_min, delta_max, var);
            
            elapsed_sec = 0.0;
        }
    }
    last_time = currentTime;
    return true;   
}  // end TimerTestApp::OnTimeout()


