# Copyright (C) 2005-2010 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

"""Minimal Test

This is a basic test to verify that mago-ng is working as expected and can
be run with various testing framework

To run it with:
$ mago-ng <path_to_this_file>

You can code ldtp directly in there or an external module
The only mandatory element is 'launcher' (and window_name for now)

set setupOnce to False to launch/close the app for each test

The purpose of this example is to test the management of the authentication
dialog
"""

from mago import TestCase
import unittest
import ldtp

class TestAuthenticate(TestCase):
    """Test the authentication dialog
    """
    launcher = 'time-admin'
    window_name = 'dlgTimeandDateSettings'

    def setUp(self):
        """setUp method. Click on btn2 to popup the auth dialog """
        super(TestAuthenticate, self).setUp()
        ldtp.click(self.window_name, 'btn2')

    def test_password(self):
        """Test authentication with a valid password

        This test enters a valid password and check the authenticate returned
        True, that the dialog is not there anymore and that the calendar is
        enabled (meaning that the authentication succeeded)
        """
        if not self.testConfig.has_option('auth', 'password'):
            raise self.failureException,"Password is mandatory. Set it in the configuration file"

        password = self.testConfig.get('auth', 'password')
        rc = self.application.authenticate(password)
        self.assertTrue(rc)
        self.assertFalse(ldtp.guiexist('dlgAuthenticate'))
        self.assertTrue(ldtp.stateenabled(self.window_name, 'calDate'))

    def test_cancel(self):
        """Test the Cancel action of the authentication dialog

        Open the authentication dialog and cancel, then check the return status
        and that the dialog is closed
        """
        rc = self.application.authenticate(cancel = True)

        self.assertTrue(rc)
        self.assertFalse(ldtp.guiexist('dlgAuthenticate'))


    def test_wrongpassword(self):
        """Test authentication with a wrong password

        The test enters a wrong password and check that authenticate returns
        False, that the dialog is still there.

        Then call authenticate again with cancel and check that the return
        status is True and the dialog vanished
        """
        # Set an invalid password
        rc = self.application.authenticate(password = "this password is invalid")
        self.assertFalse(rc)
        self.assertTrue(ldtp.guiexist('dlgAuthenticate'))

        # Cancel the auth. dialog
        rc = self.application.authenticate(cancel = True)
        self.assertTrue(rc)
        self.assertFalse(ldtp.guiexist('dlgAuthenticate'))


if __name__ == "__main__":
    unittest.main()
