/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */



/*!
    \page lcmaps_verify_proxy.mod Verification plugin plugin
 
    \section posixsynopsis SYNOPSIS
    \b lcmaps_verify_proxy.mod
 
    \section posixdesc DESCRIPTION
 
The Verify Proxy module will verify the validity of the proxy (with its chain) or any other X.509 certificate chain that has been used in the LCMAPS framework.
The module will report to the LCMAPS framework the validity of the chain.

 
*/



#ifndef LCMAPS_PROXYLIFETIME_H
#define LCMAPS_PROXYLIFETIME_H


/*****************************************************************************
                            Include header files
******************************************************************************/
#include <openssl/x509.h>
#include <time.h>

#include <lcmaps/lcmaps_types.h>


/******************************************************************************
                                Definitions
******************************************************************************/
#define LEAF_PROXY      2000	    /* Last proxy in chain */
#define INNER_PROXY     3000	    /* Middle proxy */
#define MYPROXY_PROXY   4000	    /* First proxy (after EEC) */
#define UNDEFINED_PROXY 4000


/**
 * type containing (proxy)level, its TTL and a link to the next entry in the
 * linked list */
struct TProxyLevelTTL
{
   int    level;
   time_t ttl;
   struct TProxyLevelTTL *next;
};

 
/******************************************************************************
                          Public functions
******************************************************************************/

/**
 * Verifies whether proxy lifetimes in given chain of length depth, comply with
 * the policy at each level, as given by plt. 
 * Returns: 1 on succes, 0 failure
 */
int lcmaps_lifetime_verifyProxyLifeTime (struct TProxyLevelTTL * plt, STACK_OF(X509) *chain, int depth);


/**
 * Verifies whether the VOMS AC lifetimes in given lcmaps_vomsdata comply with
 * the policy for the VOMS Attributes' Life Time (only one policy).
 * Returns 1 on success, 0 on failure
 */
int lcmaps_lifetime_verifyVOMSLifeTime (struct TProxyLevelTTL * plt, lcmaps_vomsdata_t * lcmaps_vomsdata);


/**
 * Pushes a new lifetime struct element with given level and TTL to the list of
 * lifetime structs. Returns 0 on success, -1 on error.
 */
int lcmaps_lifetime_Push_New_Entry (struct TProxyLevelTTL ** p, int level, time_t ttl);


/**
 * Prints TTL for each Level.
 */
void lcmaps_lifetime_Print_TTL_By_Level (struct TProxyLevelTTL * p);


/**
 * Frees the linked list of lifetime structs.
 */
void lcmaps_lifetime_FreeProxyLevelTTL (struct TProxyLevelTTL ** p);


/**
 * Converts a string of a special time format to a time_t
 * parameter datetime: string in the format:  2d-13:37 which means a time period
 * of 2 days, 13 hours and 37 minutes
 * Returns: number of seconds on success or -1 on failure.
 */
time_t lcmaps_lifetime_ttl_char2time_t (char * datetime);

#endif /* LCMAPS_PROXYLIFETIME_H */
