/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_log.c
    \brief  Logging routines for LCAS
    \author Martijn Steenbakkers for the EU DataGrid.
*/


/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <syslog.h>
#include <time.h>
#include <ctype.h>
#include "_lcas_log.h"

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
#ifndef DEBUG_LEVEL
#define DEBUG_LEVEL 0 /*!< default debugging level */
#endif /* DEBUG_LEVEL */

/******************************************************************************
                       Define module specific variables
******************************************************************************/
static FILE *  lcas_logfp=NULL; /*!< stream associated with logfile \internal */
static int     logging_usrlog=0; /*!< flag to do user logging \internal */
static int     logging_syslog=0; /*!< flag to use syslog \internal */
static int     debug_level=0; /*!< debugging level \internal */
static char *  extra_logstr = NULL; /*!< string to be included in every log statement \internal */
static int     should_close_lcas_logfp = 0; /*!< Flag to check if the log stream should be closed \internal */ 

/******************************************************************************
Function:       lcas_log_open()
Description:    Start logging
Parameters:
                path:    path of logfile
                fp:      file pointer to already opened file (or NULL)
                logtype: DO_USRLOG, DO_SYSLOG
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_open(char * path, FILE * fp, unsigned short logtype )
{
    char * debug_env = NULL;
    char * logstr_env = NULL;

    if ((logtype & DO_SYSLOG) == DO_SYSLOG)
    {
//        fprintf(stderr,"lcas_log_open() error: attempt to do syslogging,");
//        fprintf(stderr," not supported yet\n");
        logging_syslog=1;
#if 0
        /* Not yet applicable, wait for LCAS to become daemon */
        openlog("EDG LCAS", LOG_PID, LOG_DAEMON);
#endif
    }
    if ((logtype & DO_USRLOG) == DO_USRLOG)
    {
        logging_usrlog=1;
        if (fp != NULL)
        {
            /* File already opened */
            lcas_logfp=fp;
            should_close_lcas_logfp = 0;
        }
        else if (path != NULL)
        {
            /* Try to append to file */
            if ((lcas_logfp = fopen(path, "a")) == NULL)
            {
                fprintf(stderr, "lcas_log_open(): Cannot open logfile %s: %s\n",
                        path, strerror(errno));
                if (logging_syslog)
                {
                    syslog(LOG_ERR, "lcas_log_open(): Cannot open logfile %s\n", path);
                }
                return 1;
            }
        }
        else
        {
            fprintf(stderr, "lcas_log_open(): Please specify either (open) file descriptor");
            fprintf(stderr, " or name of logfile\n");
            return 1;
        }
    }
    /*
     * Set the debugging level:
     *    1. Try if DEBUG_LEVEL > 0
     *    2. Try if LCAS_DEBUG_LEVEL is set and if it is an integer
     *    3. set debug_level = 0;
     */
    if ( (int)(DEBUG_LEVEL) > 0 )
    {
        debug_level = (int)(DEBUG_LEVEL);
    }
    else if ( (debug_env = getenv("LCAS_DEBUG_LEVEL")) )
    {
        /* convert into integer */
        int j = 0;

        for (j = 0; j < strlen(debug_env); j++)
        {
            if (!isdigit((debug_env)[j]))
            {
                fprintf(stderr,"lcas_log_open(): found non-digit in environment variable in \"LCAS_DEBUG_LEVEL\" = %s\n", debug_env);
                return 1;
            }
        }
        debug_level = atoi(debug_env);
        if (debug_level < 0)
        {
            fprintf(stderr,"lcas_log_open(): environment variable in \"LCAS_DEBUG_LEVEL\" should be >= 0\n");
            return 1;
        }
    }
    else
    {
        debug_level = 0;
    }

    if (debug_level > 0)
    {
        lcas_log(0,"lcas_log_open(): setting debugging level to %d\n", debug_level);
    }

    /*
     * Check if there is an extra log string
     * These environment variables are checked: JOB_REPOSITORY_ID and GATEKEEPER_JM_ID
     */
    if ( (logstr_env = getenv("LCAS_LOG_STRING")) != NULL )
    {
        extra_logstr = strdup(logstr_env);
    }
    else if ( (logstr_env = getenv("JOB_REPOSITORY_ID")) != NULL )
    {
        extra_logstr = strdup(logstr_env);
    }
    else if ( (logstr_env = getenv("GATEKEEPER_JM_ID")) != NULL )
    {
        extra_logstr = strdup(logstr_env);
    }

    return 0;
}

/******************************************************************************
Function:       lcas_log()
Description:    Log information to file and or syslog
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log(int prty, char * fmt, ...)
{
    va_list pvar;
    char    buf[MAX_LOG_BUFFER_SIZE];
    int     res;

    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    if ( (res >= MAX_LOG_BUFFER_SIZE) || (res < 0) )
    {
        fprintf(stderr,"lcas_log(): log string too long (> %d)\n",
                MAX_LOG_BUFFER_SIZE);
    }
    if (logging_usrlog)
    {
        if (lcas_logfp == NULL)
        {
            fprintf(stderr,"lcas_log() error: cannot open file descriptor\n");
            return 1;
        }
        if (extra_logstr == NULL)
        {
            fprintf(lcas_logfp,"LCAS   %d: %s", prty, buf);
        }
        else
        {
            fprintf(lcas_logfp,"LCAS   %d: %s : %s", prty, extra_logstr, buf);
        }
        fflush(lcas_logfp);
    }
    if (logging_syslog && prty)
    {
        syslog(prty, "%s", buf);
    }

    /* 
     * If logging to Syslog is selected, then the priority of 0 would broadcast the message in the system by the syslogd
     * This must be prevented, hence the following hack to map the priority 0 message to 1
     * Without this hack some messages would not be submitted to Syslog, because they are masked for a good reason.
     */
    if (logging_syslog && (prty == 0))
    {
        syslog(1, "%s", buf);
    }

    return 0;
}

/******************************************************************************
Function:       lcas_log_a_string()
Description:    Log a string according to the passed format to file and or syslog
Parameters:
                prty:       syslog priority (if 0 don't syslog)
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_a_string(
        int prty,
        char * fmt,
        char * the_string
        )
    \brief log information
    
    This function logs information for LCAS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.

    \param prty         syslog priority (if 0 don't syslog).
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcas_log_a_string(int prty, char * fmt, char * the_string)
{
    char    buf[MAX_LOG_BUFFER_SIZE];
    int     res;

    res = snprintf(buf, MAX_LOG_BUFFER_SIZE, fmt, the_string);
    if ( (res >= MAX_LOG_BUFFER_SIZE) || (res < 0) )
    {
        fprintf(stderr,"lcas_log_a_string(): log string too long (> %d)\n",
                MAX_LOG_BUFFER_SIZE);
    }
    if (logging_usrlog)
    {
        if (lcas_logfp == NULL)
        {
            fprintf(stderr,"lcas_log() error: cannot open file descriptor\n");
            return 1;
        }
        if (extra_logstr == NULL)
        {
            fprintf(lcas_logfp,"LCAS %d: %s", prty, buf);
        }
        else
        {
            fprintf(lcas_logfp,"LCAS %d: %s : %s", prty, extra_logstr, buf);
        }
        fflush(lcas_logfp);
    }
    if (logging_syslog && prty)
    {
        syslog(prty, "%s", buf);
    }

    /* 
     * If logging to Syslog is selected, then the priority of 0 would broadcast the message in the system by the syslogd
     * This must be prevented, hence the following hack to map the priority 0 message to 1
     * Without this hack some messages would not be submitted to Syslog, because they are masked for a good reason.
     */
    if (logging_syslog && (prty == 0))
    {
        syslog(1, "%s", buf);
    }

    return 0;
}

/******************************************************************************
Function:       lcas_log_debug()
Description:    Print debugging information
Parameters:
                debug_lvl: debugging level
                fmt:       string format
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_debug(int debug_lvl, char * fmt, ...)
{
    va_list pvar;
    char    buf[MAX_LOG_BUFFER_SIZE];
    int     res;

    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    if ( (res >= MAX_LOG_BUFFER_SIZE) || (res < 0) )
    {
        fprintf(stderr,"lcas_log(): log string too long (> %d)\n",
                MAX_LOG_BUFFER_SIZE);
    }
    if (debug_lvl <= debug_level)
    {
        /* 
         * If logging to Syslog is selected, then the priority of 0 would broadcast the message in the system by the syslogd
         * This must be prevented, hence the following hack to map the priority 0 message to 1
         * Without this hack some messages would not be submitted to Syslog, because they are masked for a good reason.
         */

        /* lcas_log(0,buf); */
        lcas_log(1,buf);
        return 0;
    }
    return 1;
}

/******************************************************************************
Function:       lcas_log_a_string_debug()
Description:    Print debugging information
Parameters:
                debug_lvl:  debugging level
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_a_string_debug(
        int debug_lvl,
        char * fmt,
        char * the_string
        )
    \brief Print debugging information

    This function prints debugging information (using lcas_log with priority 0)
    provided debug_lvl <= DEBUG_LEVEL (default is 0).

    \param debug_lvl    debugging level
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcas_log_a_string_debug(int debug_lvl, char * fmt, char * the_string)
{
    if (debug_lvl <= debug_level)
    {
        /* 
         * If logging to Syslog is selected, then the priority of 0 would broadcast the message in the system by the syslogd
         * This must be prevented, hence the following hack to map the priority 0 message to 1
         * Without this hack some messages would not be submitted to Syslog, because they are masked for a good reason.
         */

        /* lcas_log_a_string(0, fmt, the_string); */
        lcas_log_a_string(1, fmt, the_string);
        return 0;
    }
    return 1;
}

/******************************************************************************
Function:       lcas_log_close()
Description:    Stop logging
Parameters:
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_close()
{
    if (extra_logstr != NULL)
    {
        free(extra_logstr);
        extra_logstr = NULL;
    }

    if (should_close_lcas_logfp)
    {
        fclose(lcas_logfp);
        lcas_logfp=NULL;
    }

    return 0;
}

/******************************************************************************
Function:       lcas_log_time()
Description:    Log information to file and or syslog with a timestamp
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_time(int prty, char * fmt, ...)
{
    va_list     pvar;
    char        buf[MAX_LOG_BUFFER_SIZE];
    char *      datetime = NULL;
    char *      tmpbuf = NULL;
    int         res;
    time_t      mclock;
    struct tm * tmp = NULL;


    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    if ( (res >= MAX_LOG_BUFFER_SIZE) || (res < 0) )
    {
        fprintf(stderr,"lcas_log_time(): log string too long (> %d)\n",
                MAX_LOG_BUFFER_SIZE);
    }

    if (extra_logstr == NULL)
    {
        time(&mclock);
        /* tmp = localtime(&mclock); */
        tmp = gmtime(&mclock);

        datetime = malloc(sizeof(char) * 20);

        res=snprintf(datetime, 20, "%04d-%02d-%02d.%02d:%02d:%02d",
               tmp->tm_year + 1900, tmp->tm_mon + 1, tmp->tm_mday,
               tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
        if ( (res >= 20) || (res < 0) )
        {
            fprintf(stderr,"lcas_log_time(): date string too long (> %d)\n",
                    20);
        }

        tmpbuf = (char *) malloc ((strlen(datetime) + strlen(buf) + strlen(" : ")) * sizeof(char) + 1);
        strcpy(tmpbuf, datetime);
        strcat(tmpbuf, " : ");
        strcat(tmpbuf, buf);
    }
    else
    {
        tmpbuf = (char *) malloc ((strlen(extra_logstr) + strlen(buf) + strlen(" : ")) * sizeof(char) + 1);
        strcpy(tmpbuf, extra_logstr);
        strcat(tmpbuf, " : ");
        strcat(tmpbuf, buf);
    }


    if (logging_usrlog)
    {
        if (lcas_logfp == NULL)
        {
            fprintf(stderr,"lcas_log_time() error: cannot open file descriptor\n");
            return 1;
        }
        fprintf(lcas_logfp,"LCAS   %d: %s",prty,tmpbuf);
        fflush(lcas_logfp);
    }
    if (logging_syslog && prty)
    {
        syslog(prty, "%s", tmpbuf);
    }

    /* 
     * If logging to Syslog is selected, then the priority of 0 would broadcast the message in the system by the syslogd
     * This must be prevented, hence the following hack to map the priority 0 message to 1
     * Without this hack some messages would not be submitted to Syslog, because they are masked for a good reason.
     */
    if (logging_syslog && (prty == 0))
    {
        syslog(1, "%s", buf);
    }

    if (datetime != NULL) free(datetime);
    if (tmpbuf != NULL) free(tmpbuf);

    return 0;
}


/******************************************************************************
Function:       lcas_get_debug_level()
Description:    Retrieve the debug_level
Parameters:
Returns:        the debug_level
******************************************************************************/
int
lcas_get_debug_level()
{
    return debug_level;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/lcas_log.c,v $
    $Date: 2010-06-23 08:30:41 $
    $Revision: 2.17 $
    $Author: msalle $
******************************************************************************/
