package latexDraw.generators.svg;

import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.LaTeXDrawPolygon;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.*;
import latexDraw.parsers.svg.elements.path.SVGPathSegLineto;
import latexDraw.parsers.svg.elements.path.SVGPathSegList;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a polygon.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LPolygonSVGGenerator extends LShapeSVGGenerator
{

	public LPolygonSVGGenerator(LaTeXDrawPolygon f)
	{
		super(f);
	}

	
	
	public LPolygonSVGGenerator(SVGPathElement elt)
	{
		super(new LaTeXDrawPolygon(true));
		
		if(elt==null || !elt.isPolygon())
			throw new IllegalArgumentException();
		
		SVGPathSegList segs = elt.getSegList();
		LaTeXDrawPolygon pol = (LaTeXDrawPolygon)getShape();
		int i, size;
		
		for(i=0, size=segs.size()-1; i<size; i++)
			pol.addPoint(new LaTeXDrawPoint2D(((SVGPathSegLineto)segs.elementAt(i)).getPoint()));
		
		pol.updateShape();
		pol.updateBorders();
		setSVGParameters(elt);
		applyTransformations(elt);
	}
	
	
	
	/**
	 * Creates a polygon from an SVG polygon element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LPolygonSVGGenerator(SVGPolygonElement elt)
	{
		this(new LaTeXDrawPolygon(true));
		
		LaTeXDrawPolygon pol = (LaTeXDrawPolygon)getShape();
		
		setSVGPolygonParameters(elt);
		applyTransformations(elt);
		pol.updateShape();
		pol.updateBorders();
	}
	
	
	
	
	public LPolygonSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	
	/**
	 * Creates a polygon from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LPolygonSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new LaTeXDrawPolygon(true));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGPolygonElement))
			throw new IllegalArgumentException();
		
		SVGPolygonElement main = (SVGPolygonElement)elt2;
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		setSVGPolygonParameters(main);
		
		((LaTeXDrawPolygon)getShape()).updateBorders();
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	private void setSVGPolygonParameters(SVGPolygonElement e)
	{
		setSVGParameters(e);
		Vector<Point2D> ptsPol = e.getPoints2D();
		LaTeXDrawPolygon p = (LaTeXDrawPolygon)getShape();
		
		if(ptsPol==null)
			throw new IllegalArgumentException();
		
		for(Point2D pt : ptsPol)
			p.addPoint(new LaTeXDrawPoint2D(pt));
	}
	
	
	
	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null)
			throw new IllegalArgumentException();
		
		SVGElement root = new SVGGElement(doc);
		SVGPolygonElement elt;
		String points = "";//$NON-NLS-1$
		LaTeXDrawPolygon p = (LaTeXDrawPolygon)shape;
		
        root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_POLYGON);
        root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
        
        for(LaTeXDrawPoint2D pt : p.getPoints())
     	   points+=pt.x + "," + pt.y + " ";//$NON-NLS-1$//$NON-NLS-2$
        
        if(shape.hasShadow())
        {
        	SVGPolygonElement shad = new SVGPolygonElement(doc);
         	
        	shad.setAttribute(SVGAttributes.SVG_POINTS, points);
        	setSVGShadowAttributes(shad, true);
        	root.appendChild(shad);
        }
        
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
        	elt = new SVGPolygonElement(doc);
        	elt.setAttribute(SVGAttributes.SVG_POINTS, points);
        	setSVGBorderBackground(elt, root);
        }
        
        elt = new SVGPolygonElement(doc);
        elt.setAttribute(SVGAttributes.SVG_POINTS, points);
        root.appendChild(elt);
        setSVGAttributes(doc, elt, true);
        setSVGRotationAttribute(root);
        
        if(shape.hasDoubleBoundary())
        {
        	SVGPolygonElement dblBord = new SVGPolygonElement(doc);
         	
        	dblBord.setAttribute(SVGAttributes.SVG_POINTS, points);
        	setSVGDoubleBordersAttributes(dblBord);
        	root.appendChild(dblBord);
        }
        
        return root;
	}
}
