{-|
Copyright   :  (C) 2022     , Google Inc.
License     :  BSD2 (see the file LICENSE)
Maintainer  :  QBayLogic B.V. <devops@qbaylogic.com>

This module provides a way to access static files that are useful when working
with Clash designs.
-}

module Clash.DataFiles where

import System.FilePath ((</>))

import Paths_clash_lib (getDataFileName)

{- | The Tcl Connector: a Tcl script that can parse Clash output and emit the
  correct commands for loading the design into Vivado (Quartus support will be
  added later).

  Apart from parsing the @clash-manifest.json@ files produced by Clash, the Tcl
  Connector also supports the so-called /Clash\<->Tcl API/. This functionality
  enables Clash primitives to pass complex instructions to the Tcl Connector.
  Current features are instantiating IP in Vivado and passing metadata along
  with Vivado XDC files.

  An example use of the Tcl Connector, demonstrating its basic features:

  > source -notrace clashConnector.tcl
  > # Pass it the path to "clash-manifest.json" of your top entity
  > clash::readMetadata vhdl/Design.topEntity
  > # Instantiate IP (no-op if no IP defined)
  > file mkdir ip
  > clash::createAndReadIp -dir ip
  > # Read all VHDL/Verilog/SystemVerilog files generated by Clash
  > clash::readHdl
  > # Handle XDC files, in correct order
  > clash::readXdc early
  > # A file containing PACKAGE_PIN and IOSTANDARD definitions (but not
  > # create_clock, clocks are part of the Clash-generated files)
  > read_xdc Arty-A7-35-Master.xdc
  > set_property USED_IN implementation [get_files Arty-A7-35-Master.xdc]
  > clash::readXdc {normal late}
  > synth_design -top $clash::topEntity -part xc7a35ticsg324-1L
  > opt_design
  > place_design
  > route_design
  > write_bitstream ${clash::topEntity}.bit

  "Clash.Xilinx.ClockGen" and @clash-cores:Clash.Cores.Xilinx@
  modules make use of the IP instantiating functionality; XDC metadata
  functionality is not currently used as the IP is already packaged with correct
  constraints by Vivado.

  More documentation about the Tcl Connector and the Clash\<->Tcl API will be
  made available later.

  In addition to this module, you can also write a copy of the Tcl script to a
  file by invoking

  > cabal run clash-lib:static-files -- --tcl-connector clashConnector.tcl
-}
tclConnector :: IO FilePath
tclConnector = getDataFileName $ "data-files" </> "tcl" </> "clashConnector.tcl"
