/*
** Copyright (c) 2018-2023 Valve Corporation
** Copyright (c) 2018-2023 LunarG, Inc.
** Copyright (c) 2022-2023 Advanced Micro Devices, Inc. All rights reserved.
**
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and associated documentation files (the "Software"),
** to deal in the Software without restriction, including without limitation
** the rights to use, copy, modify, merge, publish, distribute, sublicense,
** and/or sell copies of the Software, and to permit persons to whom the
** Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in
** all copies or substantial portions of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
** FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
*/

/*
** The IDs defined by this header should not be modified.  New API call
** IDs should always be appended to the existing API call ID list.
**
*/

#ifndef GFXRECON_FORMAT_API_CALL_ID_H
#define GFXRECON_FORMAT_API_CALL_ID_H

#include "util/defines.h"

#include <cstdint>

GFXRECON_BEGIN_NAMESPACE(gfxrecon)
GFXRECON_BEGIN_NAMESPACE(format)

constexpr uint32_t MakeApiCallId(uint16_t family, uint16_t api_call)
{
    return ((static_cast<uint32_t>(family) << 16) & 0xffff0000) | (static_cast<uint32_t>(api_call) & 0x0000ffff);
}

constexpr uint16_t GetApiCallFamily(uint32_t call_id)
{
    return static_cast<uint16_t>((call_id >> 16) & 0x0000ffff);
}

enum ApiFamilyId : uint16_t
{
    ApiFamily_None   = 0,
    ApiFamily_Vulkan = 1,
    ApiFamily_Dxgi   = 2,
    ApiFamily_D3D12  = 3,
    ApiFamily_AGS    = 4,
};

enum ApiCallId : uint32_t
{
    // clang-format off
    ApiCall_Unknown                                                                               = 0x0000,

    // Vulkan API
    ApiCall_vkCreateInstance                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1000),
    ApiCall_vkDestroyInstance                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1001),
    ApiCall_vkEnumeratePhysicalDevices                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1002),
    ApiCall_vkGetPhysicalDeviceFeatures                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1003),
    ApiCall_vkGetPhysicalDeviceFormatProperties                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1004),
    ApiCall_vkGetPhysicalDeviceImageFormatProperties                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1005),
    ApiCall_vkGetPhysicalDeviceProperties                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1006),
    ApiCall_vkGetPhysicalDeviceQueueFamilyProperties                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1007),
    ApiCall_vkGetPhysicalDeviceMemoryProperties                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1008),
    ApiCall_vkGetInstanceProcAddr                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1009),
    ApiCall_vkGetDeviceProcAddr                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x100a),
    ApiCall_vkCreateDevice                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x100b),
    ApiCall_vkDestroyDevice                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x100c),
    ApiCall_vkEnumerateInstanceExtensionProperties                                                = MakeApiCallId(ApiFamily_Vulkan, 0x100d),
    ApiCall_vkEnumerateDeviceExtensionProperties                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x100e),
    ApiCall_vkEnumerateInstanceLayerProperties                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x100f),
    ApiCall_vkEnumerateDeviceLayerProperties                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1010),
    ApiCall_vkGetDeviceQueue                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1011),
    ApiCall_vkQueueSubmit                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1012),
    ApiCall_vkQueueWaitIdle                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1013),
    ApiCall_vkDeviceWaitIdle                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1014),
    ApiCall_vkAllocateMemory                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1015),
    ApiCall_vkFreeMemory                                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1016),
    ApiCall_vkMapMemory                                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1017),
    ApiCall_vkUnmapMemory                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1018),
    ApiCall_vkFlushMappedMemoryRanges                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1019),
    ApiCall_vkInvalidateMappedMemoryRanges                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x101a),
    ApiCall_vkGetDeviceMemoryCommitment                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x101b),
    ApiCall_vkBindBufferMemory                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x101c),
    ApiCall_vkBindImageMemory                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x101d),
    ApiCall_vkGetBufferMemoryRequirements                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x101e),
    ApiCall_vkGetImageMemoryRequirements                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x101f),
    ApiCall_vkGetImageSparseMemoryRequirements                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1020),
    ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1021),
    ApiCall_vkQueueBindSparse                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1022),
    ApiCall_vkCreateFence                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1023),
    ApiCall_vkDestroyFence                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1024),
    ApiCall_vkResetFences                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1025),
    ApiCall_vkGetFenceStatus                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1026),
    ApiCall_vkWaitForFences                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1027),
    ApiCall_vkCreateSemaphore                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1028),
    ApiCall_vkDestroySemaphore                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1029),
    ApiCall_vkCreateEvent                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x102a),
    ApiCall_vkDestroyEvent                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x102b),
    ApiCall_vkGetEventStatus                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x102c),
    ApiCall_vkSetEvent                                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x102d),
    ApiCall_vkResetEvent                                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x102e),
    ApiCall_vkCreateQueryPool                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x102f),
    ApiCall_vkDestroyQueryPool                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1030),
    ApiCall_vkGetQueryPoolResults                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1031),
    ApiCall_vkCreateBuffer                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1032),
    ApiCall_vkDestroyBuffer                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1033),
    ApiCall_vkCreateBufferView                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1034),
    ApiCall_vkDestroyBufferView                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1035),
    ApiCall_vkCreateImage                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1036),
    ApiCall_vkDestroyImage                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1037),
    ApiCall_vkGetImageSubresourceLayout                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1038),
    ApiCall_vkCreateImageView                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1039),
    ApiCall_vkDestroyImageView                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x103a),
    ApiCall_vkCreateShaderModule                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x103b),
    ApiCall_vkDestroyShaderModule                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x103c),
    ApiCall_vkCreatePipelineCache                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x103d),
    ApiCall_vkDestroyPipelineCache                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x103e),
    ApiCall_vkGetPipelineCacheData                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x103f),
    ApiCall_vkMergePipelineCaches                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1040),
    ApiCall_vkCreateGraphicsPipelines                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1041),
    ApiCall_vkCreateComputePipelines                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1042),
    ApiCall_vkDestroyPipeline                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1043),
    ApiCall_vkCreatePipelineLayout                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1044),
    ApiCall_vkDestroyPipelineLayout                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1045),
    ApiCall_vkCreateSampler                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1046),
    ApiCall_vkDestroySampler                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1047),
    ApiCall_vkCreateDescriptorSetLayout                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1048),
    ApiCall_vkDestroyDescriptorSetLayout                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1049),
    ApiCall_vkCreateDescriptorPool                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x104a),
    ApiCall_vkDestroyDescriptorPool                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x104b),
    ApiCall_vkResetDescriptorPool                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x104c),
    ApiCall_vkAllocateDescriptorSets                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x104d),
    ApiCall_vkFreeDescriptorSets                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x104e),
    ApiCall_vkUpdateDescriptorSets                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x104f),
    ApiCall_vkCreateFramebuffer                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1050),
    ApiCall_vkDestroyFramebuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1051),
    ApiCall_vkCreateRenderPass                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1052),
    ApiCall_vkDestroyRenderPass                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1053),
    ApiCall_vkGetRenderAreaGranularity                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1054),
    ApiCall_vkCreateCommandPool                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1055),
    ApiCall_vkDestroyCommandPool                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1056),
    ApiCall_vkResetCommandPool                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1057),
    ApiCall_vkAllocateCommandBuffers                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1058),
    ApiCall_vkFreeCommandBuffers                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1059),
    ApiCall_vkBeginCommandBuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x105a),
    ApiCall_vkEndCommandBuffer                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x105b),
    ApiCall_vkResetCommandBuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x105c),
    ApiCall_vkCmdBindPipeline                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x105d),
    ApiCall_vkCmdSetViewport                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x105e),
    ApiCall_vkCmdSetScissor                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x105f),
    ApiCall_vkCmdSetLineWidth                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1060),
    ApiCall_vkCmdSetDepthBias                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1061),
    ApiCall_vkCmdSetBlendConstants                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1062),
    ApiCall_vkCmdSetDepthBounds                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1063),
    ApiCall_vkCmdSetStencilCompareMask                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1064),
    ApiCall_vkCmdSetStencilWriteMask                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1065),
    ApiCall_vkCmdSetStencilReference                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1066),
    ApiCall_vkCmdBindDescriptorSets                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1067),
    ApiCall_vkCmdBindIndexBuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1068),
    ApiCall_vkCmdBindVertexBuffers                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1069),
    ApiCall_vkCmdDraw                                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x106a),
    ApiCall_vkCmdDrawIndexed                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x106b),
    ApiCall_vkCmdDrawIndirect                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x106c),
    ApiCall_vkCmdDrawIndexedIndirect                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x106d),
    ApiCall_vkCmdDispatch                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x106e),
    ApiCall_vkCmdDispatchIndirect                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x106f),
    ApiCall_vkCmdCopyBuffer                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1070),
    ApiCall_vkCmdCopyImage                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1071),
    ApiCall_vkCmdBlitImage                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1072),
    ApiCall_vkCmdCopyBufferToImage                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1073),
    ApiCall_vkCmdCopyImageToBuffer                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1074),
    ApiCall_vkCmdUpdateBuffer                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1075),
    ApiCall_vkCmdFillBuffer                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1076),
    ApiCall_vkCmdClearColorImage                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1077),
    ApiCall_vkCmdClearDepthStencilImage                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1078),
    ApiCall_vkCmdClearAttachments                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1079),
    ApiCall_vkCmdResolveImage                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x107a),
    ApiCall_vkCmdSetEvent                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x107b),
    ApiCall_vkCmdResetEvent                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x107c),
    ApiCall_vkCmdWaitEvents                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x107d),
    ApiCall_vkCmdPipelineBarrier                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x107e),
    ApiCall_vkCmdBeginQuery                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x107f),
    ApiCall_vkCmdEndQuery                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1080),
    ApiCall_vkCmdResetQueryPool                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1081),
    ApiCall_vkCmdWriteTimestamp                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1082),
    ApiCall_vkCmdCopyQueryPoolResults                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1083),
    ApiCall_vkCmdPushConstants                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1084),
    ApiCall_vkCmdBeginRenderPass                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1085),
    ApiCall_vkCmdNextSubpass                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1086),
    ApiCall_vkCmdEndRenderPass                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1087),
    ApiCall_vkCmdExecuteCommands                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1088),
    ApiCall_vkEnumerateInstanceVersion                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1089),
    ApiCall_vkBindBufferMemory2                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x108a),
    ApiCall_vkBindImageMemory2                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x108b),
    ApiCall_vkGetDeviceGroupPeerMemoryFeatures                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x108c),
    ApiCall_vkCmdSetDeviceMask                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x108d),
    ApiCall_vkCmdDispatchBase                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x108e),
    ApiCall_vkEnumeratePhysicalDeviceGroups                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x108f),
    ApiCall_vkGetImageMemoryRequirements2                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1090),
    ApiCall_vkGetBufferMemoryRequirements2                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1091),
    ApiCall_vkGetImageSparseMemoryRequirements2                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1092),
    ApiCall_vkGetPhysicalDeviceFeatures2                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1093),
    ApiCall_vkGetPhysicalDeviceProperties2                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1094),
    ApiCall_vkGetPhysicalDeviceFormatProperties2                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1095),
    ApiCall_vkGetPhysicalDeviceImageFormatProperties2                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1096),
    ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1097),
    ApiCall_vkGetPhysicalDeviceMemoryProperties2                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1098),
    ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1099),
    ApiCall_vkTrimCommandPool                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x109a),
    ApiCall_vkGetDeviceQueue2                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x109b),
    ApiCall_vkCreateSamplerYcbcrConversion                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x109c),
    ApiCall_vkDestroySamplerYcbcrConversion                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x109d),
    ApiCall_vkCreateDescriptorUpdateTemplate                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x109e),
    ApiCall_vkDestroyDescriptorUpdateTemplate                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x109f),
    ApiCall_vkUpdateDescriptorSetWithTemplate                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10a0),
    ApiCall_vkGetPhysicalDeviceExternalBufferProperties                                           = MakeApiCallId(ApiFamily_Vulkan, 0x10a1),
    ApiCall_vkGetPhysicalDeviceExternalFenceProperties                                            = MakeApiCallId(ApiFamily_Vulkan, 0x10a2),
    ApiCall_vkGetPhysicalDeviceExternalSemaphoreProperties                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10a3),
    ApiCall_vkGetDescriptorSetLayoutSupport                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10a4),
    ApiCall_vkDestroySurfaceKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10a5),
    ApiCall_vkGetPhysicalDeviceSurfaceSupportKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10a6),
    ApiCall_vkGetPhysicalDeviceSurfaceCapabilitiesKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10a7),
    ApiCall_vkGetPhysicalDeviceSurfaceFormatsKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10a8),
    ApiCall_vkGetPhysicalDeviceSurfacePresentModesKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10a9),
    ApiCall_vkCreateSwapchainKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10aa),
    ApiCall_vkDestroySwapchainKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ab),
    ApiCall_vkGetSwapchainImagesKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10ac),
    ApiCall_vkAcquireNextImageKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ad),
    ApiCall_vkQueuePresentKHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10ae),
    ApiCall_vkGetDeviceGroupPresentCapabilitiesKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10af),
    ApiCall_vkGetDeviceGroupSurfacePresentModesKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10b0),
    ApiCall_vkGetPhysicalDevicePresentRectanglesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10b1),
    ApiCall_vkAcquireNextImage2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10b2),
    ApiCall_vkGetPhysicalDeviceDisplayPropertiesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10b3),
    ApiCall_vkGetPhysicalDeviceDisplayPlanePropertiesKHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10b4),
    ApiCall_vkGetDisplayPlaneSupportedDisplaysKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10b5),
    ApiCall_vkGetDisplayModePropertiesKHR                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10b6),
    ApiCall_vkCreateDisplayModeKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10b7),
    ApiCall_vkGetDisplayPlaneCapabilitiesKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10b8),
    ApiCall_vkCreateDisplayPlaneSurfaceKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10b9),
    ApiCall_vkCreateSharedSwapchainsKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x10ba),
    ApiCall_vkCreateXlibSurfaceKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10bb),
    ApiCall_vkGetPhysicalDeviceXlibPresentationSupportKHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10bc),
    ApiCall_vkCreateXcbSurfaceKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10bd),
    ApiCall_vkGetPhysicalDeviceXcbPresentationSupportKHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10be),
    ApiCall_vkCreateWaylandSurfaceKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10bf),
    ApiCall_vkGetPhysicalDeviceWaylandPresentationSupportKHR                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10c0),
    ApiCall_vkCreateMirSurfaceKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10c1),
    ApiCall_vkGetPhysicalDeviceMirPresentationSupportKHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10c2),
    ApiCall_vkCreateAndroidSurfaceKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10c3),
    ApiCall_vkCreateWin32SurfaceKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10c4),
    ApiCall_vkGetPhysicalDeviceWin32PresentationSupportKHR                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10c5),
    ApiCall_vkGetPhysicalDeviceFeatures2KHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10c6),
    ApiCall_vkGetPhysicalDeviceProperties2KHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10c7),
    ApiCall_vkGetPhysicalDeviceFormatProperties2KHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10c8),
    ApiCall_vkGetPhysicalDeviceImageFormatProperties2KHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10c9),
    ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2KHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10ca),
    ApiCall_vkGetPhysicalDeviceMemoryProperties2KHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10cb),
    ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2KHR                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10cc),
    ApiCall_vkGetDeviceGroupPeerMemoryFeaturesKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10cd),
    ApiCall_vkCmdSetDeviceMaskKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ce),
    ApiCall_vkCmdDispatchBaseKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10cf),
    ApiCall_vkTrimCommandPoolKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10d0),
    ApiCall_vkEnumeratePhysicalDeviceGroupsKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10d1),
    ApiCall_vkGetPhysicalDeviceExternalBufferPropertiesKHR                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10d2),
    ApiCall_vkGetMemoryWin32HandleKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10d3),
    ApiCall_vkGetMemoryWin32HandlePropertiesKHR                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10d4),
    ApiCall_vkGetMemoryFdKHR                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10d5),
    ApiCall_vkGetMemoryFdPropertiesKHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x10d6),
    ApiCall_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10d7),
    ApiCall_vkImportSemaphoreWin32HandleKHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10d8),
    ApiCall_vkGetSemaphoreWin32HandleKHR                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10d9),
    ApiCall_vkImportSemaphoreFdKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10da),
    ApiCall_vkGetSemaphoreFdKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10db),
    ApiCall_vkCmdPushDescriptorSetKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10dc),
    ApiCall_vkCmdPushDescriptorSetWithTemplateKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10dd),
    ApiCall_vkCreateDescriptorUpdateTemplateKHR                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10de),
    ApiCall_vkDestroyDescriptorUpdateTemplateKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10df),
    ApiCall_vkUpdateDescriptorSetWithTemplateKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10e0),
    ApiCall_vkCreateRenderPass2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10e1),
    ApiCall_vkCmdBeginRenderPass2KHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x10e2),
    ApiCall_vkCmdNextSubpass2KHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10e3),
    ApiCall_vkCmdEndRenderPass2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10e4),
    ApiCall_vkGetSwapchainStatusKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10e5),
    ApiCall_vkGetPhysicalDeviceExternalFencePropertiesKHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10e6),
    ApiCall_vkImportFenceWin32HandleKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x10e7),
    ApiCall_vkGetFenceWin32HandleKHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x10e8),
    ApiCall_vkImportFenceFdKHR                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10e9),
    ApiCall_vkGetFenceFdKHR                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10ea),
    ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2KHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x10eb),
    ApiCall_vkGetPhysicalDeviceSurfaceFormats2KHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ec),
    ApiCall_vkGetPhysicalDeviceDisplayProperties2KHR                                              = MakeApiCallId(ApiFamily_Vulkan, 0x10ed),
    ApiCall_vkGetPhysicalDeviceDisplayPlaneProperties2KHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10ee),
    ApiCall_vkGetDisplayModeProperties2KHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10ef),
    ApiCall_vkGetDisplayPlaneCapabilities2KHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10f0),
    ApiCall_vkGetImageMemoryRequirements2KHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10f1),
    ApiCall_vkGetBufferMemoryRequirements2KHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10f2),
    ApiCall_vkGetImageSparseMemoryRequirements2KHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10f3),
    ApiCall_vkCreateSamplerYcbcrConversionKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10f4),
    ApiCall_vkDestroySamplerYcbcrConversionKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10f5),
    ApiCall_vkBindBufferMemory2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10f6),
    ApiCall_vkBindImageMemory2KHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10f7),
    ApiCall_vkGetDescriptorSetLayoutSupportKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10f8),
    ApiCall_vkCmdDrawIndirectCountKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10f9),
    ApiCall_vkCmdDrawIndexedIndirectCountKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10fa),
    ApiCall_vkCreateDebugReportCallbackEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10fb),
    ApiCall_vkDestroyDebugReportCallbackEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10fc),
    ApiCall_vkDebugReportMessageEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10fd),
    ApiCall_vkDebugMarkerSetObjectTagEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10fe),
    ApiCall_vkDebugMarkerSetObjectNameEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10ff),
    ApiCall_vkCmdDebugMarkerBeginEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1100),
    ApiCall_vkCmdDebugMarkerEndEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1101),
    ApiCall_vkCmdDebugMarkerInsertEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1102),
    ApiCall_vkCmdDrawIndirectCountAMD                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1103),
    ApiCall_vkCmdDrawIndexedIndirectCountAMD                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1104),
    ApiCall_vkGetShaderInfoAMD                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1105),
    ApiCall_vkGetPhysicalDeviceExternalImageFormatPropertiesNV                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1106),
    ApiCall_vkGetMemoryWin32HandleNV                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1107),
    ApiCall_vkCreateViSurfaceNN                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1108),
    ApiCall_vkCmdBeginConditionalRenderingEXT                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1109),
    ApiCall_vkCmdEndConditionalRenderingEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x110a),
    ApiCall_vkCmdProcessCommandsNVX                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x110b),
    ApiCall_vkCmdReserveSpaceForCommandsNVX                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x110c),
    ApiCall_vkCreateIndirectCommandsLayoutNVX                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x110d),
    ApiCall_vkDestroyIndirectCommandsLayoutNVX                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x110e),
    ApiCall_vkCreateObjectTableNVX                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x110f),
    ApiCall_vkDestroyObjectTableNVX                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1110),
    ApiCall_vkRegisterObjectsNVX                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1111),
    ApiCall_vkUnregisterObjectsNVX                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1112),
    ApiCall_vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1113),
    ApiCall_vkCmdSetViewportWScalingNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1114),
    ApiCall_vkReleaseDisplayEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1115),
    ApiCall_vkAcquireXlibDisplayEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1116),
    ApiCall_vkGetRandROutputDisplayEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1117),
    ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2EXT                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1118),
    ApiCall_vkDisplayPowerControlEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1119),
    ApiCall_vkRegisterDeviceEventEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x111a),
    ApiCall_vkRegisterDisplayEventEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x111b),
    ApiCall_vkGetSwapchainCounterEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x111c),
    ApiCall_vkGetRefreshCycleDurationGOOGLE                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x111d),
    ApiCall_vkGetPastPresentationTimingGOOGLE                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x111e),
    ApiCall_vkCmdSetDiscardRectangleEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x111f),
    ApiCall_vkSetHdrMetadataEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1120),
    ApiCall_vkCreateIOSSurfaceMVK                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1121),
    ApiCall_vkCreateMacOSSurfaceMVK                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1122),
    ApiCall_vkSetDebugUtilsObjectNameEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1123),
    ApiCall_vkSetDebugUtilsObjectTagEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1124),
    ApiCall_vkQueueBeginDebugUtilsLabelEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1125),
    ApiCall_vkQueueEndDebugUtilsLabelEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1126),
    ApiCall_vkQueueInsertDebugUtilsLabelEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1127),
    ApiCall_vkCmdBeginDebugUtilsLabelEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1128),
    ApiCall_vkCmdEndDebugUtilsLabelEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1129),
    ApiCall_vkCmdInsertDebugUtilsLabelEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x112a),
    ApiCall_vkCreateDebugUtilsMessengerEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x112b),
    ApiCall_vkDestroyDebugUtilsMessengerEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x112c),
    ApiCall_vkSubmitDebugUtilsMessageEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x112d),
    ApiCall_vkGetAndroidHardwareBufferPropertiesANDROID                                           = MakeApiCallId(ApiFamily_Vulkan, 0x112e),
    ApiCall_vkGetMemoryAndroidHardwareBufferANDROID                                               = MakeApiCallId(ApiFamily_Vulkan, 0x112f),
    ApiCall_vkCmdSetSampleLocationsEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1130),
    ApiCall_vkGetPhysicalDeviceMultisamplePropertiesEXT                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1131),
    ApiCall_vkCreateValidationCacheEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1132),
    ApiCall_vkDestroyValidationCacheEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1133),
    ApiCall_vkMergeValidationCachesEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1134),
    ApiCall_vkGetValidationCacheDataEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1135),
    ApiCall_vkGetMemoryHostPointerPropertiesEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1136),
    ApiCall_vkCmdWriteBufferMarkerAMD                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1137),
    ApiCall_vkCmdBindShadingRateImageNV                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1138),
    ApiCall_vkCmdSetViewportShadingRatePaletteNV                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1139),
    ApiCall_vkCmdSetCoarseSampleOrderNV                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x113a),
    ApiCall_vkGetImageDrmFormatModifierPropertiesEXT                                              = MakeApiCallId(ApiFamily_Vulkan, 0x113b),
    ApiCall_vkCreateAccelerationStructureNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x113c),
    ApiCall_vkDestroyAccelerationStructureNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x113d),
    ApiCall_vkGetAccelerationStructureMemoryRequirementsNV                                        = MakeApiCallId(ApiFamily_Vulkan, 0x113e),
    ApiCall_vkBindAccelerationStructureMemoryNV                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x113f),
    ApiCall_vkCmdBuildAccelerationStructureNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1140),
    ApiCall_vkCmdCopyAccelerationStructureNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1141),
    ApiCall_vkCmdTraceRaysNV                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1142),
    ApiCall_vkCreateRayTracingPipelinesNV                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1143),
    ApiCall_vkGetRayTracingShaderGroupHandlesNV                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1144),
    ApiCall_vkGetAccelerationStructureHandleNV                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1145),
    ApiCall_vkCmdWriteAccelerationStructuresPropertiesNV                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1146),
    ApiCall_vkCompileDeferredNV                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1147),
    ApiCall_vkCmdDrawMeshTasksNV                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1148),
    ApiCall_vkCmdDrawMeshTasksIndirectNV                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1149),
    ApiCall_vkCmdDrawMeshTasksIndirectCountNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x114a),
    ApiCall_vkCmdSetExclusiveScissorNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x114b),
    ApiCall_vkCmdSetCheckpointNV                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x114c),
    ApiCall_vkGetQueueCheckpointDataNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x114d),
    ApiCall_vkCreateImagePipeSurfaceFUCHSIA                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x114e),
    ApiCall_vkCmdBindTransformFeedbackBuffersEXT                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x114f),
    ApiCall_vkCmdBeginTransformFeedbackEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1150),
    ApiCall_vkCmdEndTransformFeedbackEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1151),
    ApiCall_vkCmdBeginQueryIndexedEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1152),
    ApiCall_vkCmdEndQueryIndexedEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1153),
    ApiCall_vkCmdDrawIndirectByteCountEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1154),
    ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1155),
    ApiCall_vkGetCalibratedTimestampsEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1156),
    ApiCall_vkGetBufferDeviceAddressEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1157),
    ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesNV                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1158),
    ApiCall_vkGetImageViewHandleNVX                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1159),
    ApiCall_vkCreateMetalSurfaceEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x115a),
    ApiCall_vkCreateStreamDescriptorSurfaceGGP                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x115b),
    ApiCall_vkSetLocalDimmingAMD                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x115c),
    ApiCall_vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV                     = MakeApiCallId(ApiFamily_Vulkan, 0x115d),
    ApiCall_vkGetPhysicalDeviceSurfacePresentModes2EXT                                            = MakeApiCallId(ApiFamily_Vulkan, 0x115e),
    ApiCall_vkAcquireFullScreenExclusiveModeEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x115f),
    ApiCall_vkReleaseFullScreenExclusiveModeEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1160),
    ApiCall_vkGetDeviceGroupSurfacePresentModes2EXT                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1161),
    ApiCall_vkCreateHeadlessSurfaceEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1162),
    ApiCall_vkResetQueryPoolEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1163),
    ApiCall_vkGetPipelineExecutablePropertiesKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1164),
    ApiCall_vkGetPipelineExecutableStatisticsKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1165),
    ApiCall_vkGetPipelineExecutableInternalRepresentationsKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1166),
    ApiCall_vkInitializePerformanceApiINTEL                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1167),
    ApiCall_vkUninitializePerformanceApiINTEL                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1168),
    ApiCall_vkCmdSetPerformanceMarkerINTEL                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1169),
    ApiCall_vkCmdSetPerformanceStreamMarkerINTEL                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x116a),
    ApiCall_vkCmdSetPerformanceOverrideINTEL                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x116b),
    ApiCall_vkAcquirePerformanceConfigurationINTEL                                                = MakeApiCallId(ApiFamily_Vulkan, 0x116c),
    ApiCall_vkReleasePerformanceConfigurationINTEL                                                = MakeApiCallId(ApiFamily_Vulkan, 0x116d),
    ApiCall_vkQueueSetPerformanceConfigurationINTEL                                               = MakeApiCallId(ApiFamily_Vulkan, 0x116e),
    ApiCall_vkGetPerformanceParameterINTEL                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x116f),
    ApiCall_vkCmdSetLineStippleEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1170),
    ApiCall_vkGetSemaphoreCounterValueKHR                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1171),
    ApiCall_vkWaitSemaphoresKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1172),
    ApiCall_vkSignalSemaphoreKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1173),
    ApiCall_vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR                       = MakeApiCallId(ApiFamily_Vulkan, 0x1174),
    ApiCall_vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR                               = MakeApiCallId(ApiFamily_Vulkan, 0x1175),
    ApiCall_vkAcquireProfilingLockKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1176),
    ApiCall_vkReleaseProfilingLockKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1177),
    ApiCall_vkGetBufferDeviceAddressKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1178),
    ApiCall_vkGetBufferOpaqueCaptureAddressKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1179),
    ApiCall_vkGetDeviceMemoryOpaqueCaptureAddressKHR                                              = MakeApiCallId(ApiFamily_Vulkan, 0x117a),
    ApiCall_vkGetPhysicalDeviceToolPropertiesEXT                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x117b),
    ApiCall_vkCmdDrawIndirectCount                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x117c),
    ApiCall_vkCmdDrawIndexedIndirectCount                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x117d),
    ApiCall_vkCreateRenderPass2                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x117e),
    ApiCall_vkCmdBeginRenderPass2                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x117f),
    ApiCall_vkCmdNextSubpass2                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1180),
    ApiCall_vkCmdEndRenderPass2                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1181),
    ApiCall_vkResetQueryPool                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1182),
    ApiCall_vkGetSemaphoreCounterValue                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1183),
    ApiCall_vkWaitSemaphores                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1184),
    ApiCall_vkSignalSemaphore                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1185),
    ApiCall_vkGetBufferDeviceAddress                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1186),
    ApiCall_vkGetBufferOpaqueCaptureAddress                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1187),
    ApiCall_vkGetDeviceMemoryOpaqueCaptureAddress                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1188),
    ApiCall_vkCreateDeferredOperationKHR                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1189),
    ApiCall_vkDestroyDeferredOperationKHR                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x118a),
    ApiCall_vkGetDeferredOperationMaxConcurrencyKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x118b),
    ApiCall_vkGetDeferredOperationResultKHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x118c),
    ApiCall_vkDeferredOperationJoinKHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x118d),
    ApiCall_vkCreateAccelerationStructureKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x118e),
    ApiCall_vkDestroyAccelerationStructureKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x118f),
    ApiCall_vkGetAccelerationStructureMemoryRequirementsKHR                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1190),
    ApiCall_vkBindAccelerationStructureMemoryKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1191),
    ApiCall_vkCmdBuildAccelerationStructuresKHR                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1192),
    ApiCall_vkCmdBuildAccelerationStructuresIndirectKHR                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1193),
    ApiCall_vkBuildAccelerationStructuresKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1194),
    ApiCall_vkCopyAccelerationStructureKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1195),
    ApiCall_vkCopyAccelerationStructureToMemoryKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1196),
    ApiCall_vkCopyMemoryToAccelerationStructureKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1197),
    ApiCall_vkWriteAccelerationStructuresPropertiesKHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1198),
    ApiCall_vkCmdCopyAccelerationStructureKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1199),
    ApiCall_vkCmdCopyAccelerationStructureToMemoryKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x119a),
    ApiCall_vkCmdCopyMemoryToAccelerationStructureKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x119b),
    ApiCall_vkCmdTraceRaysKHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x119c),
    ApiCall_vkCreateRayTracingPipelinesKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x119d),
    ApiCall_vkGetRayTracingShaderGroupHandlesKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x119e),
    ApiCall_vkCmdWriteAccelerationStructuresPropertiesKHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x119f),
    ApiCall_vkGetAccelerationStructureDeviceAddressKHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1200),
    ApiCall_vkGetRayTracingCaptureReplayShaderGroupHandlesKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1201),
    ApiCall_vkCmdTraceRaysIndirectKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1202),
    ApiCall_vkGetDeviceAccelerationStructureCompatibilityKHR                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1203),
    ApiCall_vkGetGeneratedCommandsMemoryRequirementsNV                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1204),
    ApiCall_vkCmdPreprocessGeneratedCommandsNV                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1205),
    ApiCall_vkCmdExecuteGeneratedCommandsNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1206),
    ApiCall_vkCmdBindPipelineShaderGroupNV                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1207),
    ApiCall_vkCreateIndirectCommandsLayoutNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1208),
    ApiCall_vkDestroyIndirectCommandsLayoutNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1209),
    ApiCall_vkGetImageViewAddressNVX                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x120a),
    ApiCall_vkCreatePrivateDataSlotEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x120b),
    ApiCall_vkDestroyPrivateDataSlotEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x120c),
    ApiCall_vkSetPrivateDataEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x120d),
    ApiCall_vkGetPrivateDataEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x120e),
    ApiCall_vkCmdSetCullModeEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x120f),
    ApiCall_vkCmdSetFrontFaceEXT                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1210),
    ApiCall_vkCmdSetPrimitiveTopologyEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1211),
    ApiCall_vkCmdSetViewportWithCountEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1212),
    ApiCall_vkCmdSetScissorWithCountEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1213),
    ApiCall_vkCmdBindVertexBuffers2EXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1214),
    ApiCall_vkCmdSetDepthTestEnableEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1215),
    ApiCall_vkCmdSetDepthWriteEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1216),
    ApiCall_vkCmdSetDepthCompareOpEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1217),
    ApiCall_vkCmdSetDepthBoundsTestEnableEXT                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1218),
    ApiCall_vkCmdSetStencilTestEnableEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1219),
    ApiCall_vkCmdSetStencilOpEXT                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x121a),
    ApiCall_vkCreateDirectFBSurfaceEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x121b),
    ApiCall_vkGetPhysicalDeviceDirectFBPresentationSupportEXT                                     = MakeApiCallId(ApiFamily_Vulkan, 0x121c),
    ApiCall_vkCmdCopyBuffer2KHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x121d),
    ApiCall_vkCmdCopyImage2KHR                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x121e),
    ApiCall_vkCmdCopyBufferToImage2KHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x121f),
    ApiCall_vkCmdCopyImageToBuffer2KHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1220),
    ApiCall_vkCmdBlitImage2KHR                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1221),
    ApiCall_vkCmdResolveImage2KHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1222),
    ApiCall_vkGetAccelerationStructureBuildSizesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1223),
    ApiCall_vkGetRayTracingShaderGroupStackSizeKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1224),
    ApiCall_vkCmdSetRayTracingPipelineStackSizeKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1225),
    ApiCall_vkGetPhysicalDeviceFragmentShadingRatesKHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1226),
    ApiCall_vkCmdSetFragmentShadingRateKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1227),
    ApiCall_vkCmdSetFragmentShadingRateEnumNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1228),
    ApiCall_vkAcquireWinrtDisplayNV                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1229),
    ApiCall_vkGetWinrtDisplayNV                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x122a),
    ApiCall_vkCmdSetEvent2KHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x122b),
    ApiCall_vkCmdResetEvent2KHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x122c),
    ApiCall_vkCmdWaitEvents2KHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x122d),
    ApiCall_vkCmdPipelineBarrier2KHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x122e),
    ApiCall_vkCmdWriteTimestamp2KHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x122f),
    ApiCall_vkQueueSubmit2KHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1230),
    ApiCall_vkCmdWriteBufferMarker2AMD                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1231),
    ApiCall_vkGetQueueCheckpointData2NV                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1232),
    ApiCall_vkCmdSetVertexInputEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1233),
    ApiCall_vkGetMemoryZirconHandleFUCHSIA                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1234),
    ApiCall_vkGetMemoryZirconHandlePropertiesFUCHSIA                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1235),
    ApiCall_vkImportSemaphoreZirconHandleFUCHSIA                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1236),
    ApiCall_vkGetSemaphoreZirconHandleFUCHSIA                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1237),
    ApiCall_vkCmdSetPatchControlPointsEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1238),
    ApiCall_vkCmdSetRasterizerDiscardEnableEXT                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1239),
    ApiCall_vkCmdSetDepthBiasEnableEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x123a),
    ApiCall_vkCmdSetLogicOpEXT                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x123b),
    ApiCall_vkCmdSetPrimitiveRestartEnableEXT                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x123c),
    ApiCall_vkCreateScreenSurfaceQNX                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x123d),
    ApiCall_vkGetPhysicalDeviceScreenPresentationSupportQNX                                       = MakeApiCallId(ApiFamily_Vulkan, 0x123e),
    ApiCall_vkCmdSetColorWriteEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x123f),
    ApiCall_vkAcquireDrmDisplayEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1240),
    ApiCall_vkGetDrmDisplayEXT                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1241),
    ApiCall_vkCmdDrawMultiEXT                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1242),
    ApiCall_vkCmdDrawMultiIndexedEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1243),
    ApiCall_vkWaitForPresentKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1244),
    ApiCall_vkCmdBindInvocationMaskHUAWEI                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1245),
    ApiCall_vkGetMemoryRemoteAddressNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1246),
    ApiCall_vkGetDeviceBufferMemoryRequirementsKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1247),
    ApiCall_vkGetDeviceImageMemoryRequirementsKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1248),
    ApiCall_vkGetDeviceImageSparseMemoryRequirementsKHR                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1249),
    ApiCall_vkSetDeviceMemoryPriorityEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x124a),
    ApiCall_vkCmdBeginRenderingKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x124b),
    ApiCall_vkCmdEndRenderingKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x124c),
    ApiCall_vkGetPhysicalDeviceToolProperties                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x124d),
    ApiCall_vkCreatePrivateDataSlot                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x124e),
    ApiCall_vkDestroyPrivateDataSlot                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x124f),
    ApiCall_vkSetPrivateData                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1250),
    ApiCall_vkGetPrivateData                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1251),
    ApiCall_vkCmdSetEvent2                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1252),
    ApiCall_vkCmdResetEvent2                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1253),
    ApiCall_vkCmdWaitEvents2                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1254),
    ApiCall_vkCmdPipelineBarrier2                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1255),
    ApiCall_vkCmdWriteTimestamp2                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1256),
    ApiCall_vkQueueSubmit2                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1257),
    ApiCall_vkCmdCopyBuffer2                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1258),
    ApiCall_vkCmdCopyImage2                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1259),
    ApiCall_vkCmdCopyBufferToImage2                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x125a),
    ApiCall_vkCmdCopyImageToBuffer2                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x125b),
    ApiCall_vkCmdBlitImage2                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x125c),
    ApiCall_vkCmdResolveImage2                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x125d),
    ApiCall_vkCmdBeginRendering                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x125e),
    ApiCall_vkCmdEndRendering                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x125f),
    ApiCall_vkCmdSetCullMode                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1260),
    ApiCall_vkCmdSetFrontFace                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1261),
    ApiCall_vkCmdSetPrimitiveTopology                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1262),
    ApiCall_vkCmdSetViewportWithCount                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1263),
    ApiCall_vkCmdSetScissorWithCount                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1264),
    ApiCall_vkCmdBindVertexBuffers2                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1265),
    ApiCall_vkCmdSetDepthTestEnable                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1266),
    ApiCall_vkCmdSetDepthWriteEnable                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1267),
    ApiCall_vkCmdSetDepthCompareOp                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1268),
    ApiCall_vkCmdSetDepthBoundsTestEnable                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1269),
    ApiCall_vkCmdSetStencilTestEnable                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x126a),
    ApiCall_vkCmdSetStencilOp                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x126b),
    ApiCall_vkCmdSetRasterizerDiscardEnable                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x126c),
    ApiCall_vkCmdSetDepthBiasEnable                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x126d),
    ApiCall_vkCmdSetPrimitiveRestartEnable                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x126e),
    ApiCall_vkGetDeviceBufferMemoryRequirements                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x126f),
    ApiCall_vkGetDeviceImageMemoryRequirements                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1270),
    ApiCall_vkGetDeviceImageSparseMemoryRequirements                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1271),
    ApiCall_vkGetDescriptorSetLayoutHostMappingInfoVALVE                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1272),
    ApiCall_vkGetDescriptorSetHostMappingVALVE                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1273),
    ApiCall_vkCmdTraceRaysIndirect2KHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1274),
    ApiCall_vkGetImageSubresourceLayout2EXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1275),
    ApiCall_vkGetPipelinePropertiesEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1276),
    ApiCall_vkGetShaderModuleIdentifierEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1277),
    ApiCall_vkGetShaderModuleCreateInfoIdentifierEXT                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1278),
    ApiCall_vkGetFramebufferTilePropertiesQCOM                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1279),
    ApiCall_vkGetDynamicRenderingTilePropertiesQCOM                                               = MakeApiCallId(ApiFamily_Vulkan, 0x127a),
    ApiCall_vkGetDeviceFaultInfoEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x127b),
    ApiCall_vkCreateMicromapEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x127c),
    ApiCall_vkDestroyMicromapEXT                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x127d),
    ApiCall_vkCmdBuildMicromapsEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x127e),
    ApiCall_vkBuildMicromapsEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x127f),
    ApiCall_vkCopyMicromapEXT                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1280),
    ApiCall_vkCopyMicromapToMemoryEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1281),
    ApiCall_vkCopyMemoryToMicromapEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1282),
    ApiCall_vkWriteMicromapsPropertiesEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1283),
    ApiCall_vkCmdCopyMicromapEXT                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1284),
    ApiCall_vkCmdCopyMicromapToMemoryEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1285),
    ApiCall_vkCmdCopyMemoryToMicromapEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1286),
    ApiCall_vkCmdWriteMicromapsPropertiesEXT                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1287),
    ApiCall_vkGetDeviceMicromapCompatibilityEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1288),
    ApiCall_vkGetMicromapBuildSizesEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1289),
    ApiCall_vkCmdSetTessellationDomainOriginEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x128a),
    ApiCall_vkCmdSetDepthClampEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x128b),
    ApiCall_vkCmdSetPolygonModeEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x128c),
    ApiCall_vkCmdSetRasterizationSamplesEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x128d),
    ApiCall_vkCmdSetSampleMaskEXT                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x128e),
    ApiCall_vkCmdSetAlphaToCoverageEnableEXT                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x128f),
    ApiCall_vkCmdSetAlphaToOneEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1290),
    ApiCall_vkCmdSetLogicOpEnableEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1291),
    ApiCall_vkCmdSetColorBlendEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1292),
    ApiCall_vkCmdSetColorBlendEquationEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1293),
    ApiCall_vkCmdSetColorWriteMaskEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1294),
    ApiCall_vkCmdSetRasterizationStreamEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1295),
    ApiCall_vkCmdSetConservativeRasterizationModeEXT                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1296),
    ApiCall_vkCmdSetExtraPrimitiveOverestimationSizeEXT                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1297),
    ApiCall_vkCmdSetDepthClipEnableEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1298),
    ApiCall_vkCmdSetSampleLocationsEnableEXT                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1299),
    ApiCall_vkCmdSetColorBlendAdvancedEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x129a),
    ApiCall_vkCmdSetProvokingVertexModeEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x129b),
    ApiCall_vkCmdSetLineRasterizationModeEXT                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x129c),
    ApiCall_vkCmdSetLineStippleEnableEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x129d),
    ApiCall_vkCmdSetDepthClipNegativeOneToOneEXT                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x129e),
    ApiCall_vkCmdSetViewportWScalingEnableNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x129f),
    ApiCall_vkCmdSetViewportSwizzleNV                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x12a0),
    ApiCall_vkCmdSetCoverageToColorEnableNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x12a1),
    ApiCall_vkCmdSetCoverageToColorLocationNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12a2),
    ApiCall_vkCmdSetCoverageModulationModeNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x12a3),
    ApiCall_vkCmdSetCoverageModulationTableEnableNV                                               = MakeApiCallId(ApiFamily_Vulkan, 0x12a4),
    ApiCall_vkCmdSetCoverageModulationTableNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12a5),
    ApiCall_vkCmdSetShadingRateImageEnableNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x12a6),
    ApiCall_vkCmdSetRepresentativeFragmentTestEnableNV                                            = MakeApiCallId(ApiFamily_Vulkan, 0x12a7),
    ApiCall_vkCmdSetCoverageReductionModeNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x12a8),
    ApiCall_vkGetPhysicalDeviceOpticalFlowImageFormatsNV                                          = MakeApiCallId(ApiFamily_Vulkan, 0x12a9),
    ApiCall_vkCreateOpticalFlowSessionNV                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x12aa),
    ApiCall_vkDestroyOpticalFlowSessionNV                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x12ab),
    ApiCall_vkBindOpticalFlowSessionImageNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x12ac),
    ApiCall_vkCmdOpticalFlowExecuteNV                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x12ad),
    ApiCall_vkCmdDrawMeshTasksEXT                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x12ae),
    ApiCall_vkCmdDrawMeshTasksIndirectEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x12af),
    ApiCall_vkCmdDrawMeshTasksIndirectCountEXT                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x12b0),
    ApiCall_vkReleaseSwapchainImagesEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x12b1),
    ApiCall_vkCmdDrawClusterHUAWEI                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12b2),
    ApiCall_vkCmdDrawClusterIndirectHUAWEI                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x12b3),
    ApiCall_vkGetPhysicalDeviceVideoCapabilitiesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x12b4),
    ApiCall_vkGetPhysicalDeviceVideoFormatPropertiesKHR                                           = MakeApiCallId(ApiFamily_Vulkan, 0x12b5),
    ApiCall_vkCreateVideoSessionKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x12b6),
    ApiCall_vkDestroyVideoSessionKHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x12b7),
    ApiCall_vkGetVideoSessionMemoryRequirementsKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12b8),
    ApiCall_vkBindVideoSessionMemoryKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x12b9),
    ApiCall_vkCreateVideoSessionParametersKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12ba),
    ApiCall_vkUpdateVideoSessionParametersKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12bb),
    ApiCall_vkDestroyVideoSessionParametersKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x12bc),
    ApiCall_vkCmdBeginVideoCodingKHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x12bd),
    ApiCall_vkCmdEndVideoCodingKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12be),
    ApiCall_vkCmdControlVideoCodingKHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x12bf),
    ApiCall_vkCmdDecodeVideoKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x12c0),
    ApiCall_vkCmdEncodeVideoKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x12c1),
    ApiCall_vkCmdSetDiscardRectangleEnableEXT                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12c2),
    ApiCall_vkCmdSetDiscardRectangleModeEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x12c3),
    ApiCall_vkCmdSetExclusiveScissorEnableNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x12c4),
    ApiCall_vkMapMemory2KHR                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x12c5),
    ApiCall_vkUnmapMemory2KHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12c6),
    ApiCall_vkCreateShadersEXT                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x12c7),
    ApiCall_vkDestroyShaderEXT                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x12c8),
    ApiCall_vkGetShaderBinaryDataEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x12c9),
    ApiCall_vkCmdBindShadersEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x12ca),
    ApiCall_vkCmdSetAttachmentFeedbackLoopEnableEXT                                               = MakeApiCallId(ApiFamily_Vulkan, 0x12cb),
    ApiCall_vkGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR                               = MakeApiCallId(ApiFamily_Vulkan, 0x12cc),
    ApiCall_vkGetEncodedVideoSessionParametersKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x12cd),
    ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12ce),
    ApiCall_vkCmdSetDepthBias2EXT                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x12cf),
    ApiCall_vkCopyMemoryToImageEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12d0),
    ApiCall_vkCopyImageToMemoryEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12d1),
    ApiCall_vkCopyImageToImageEXT                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x12d2),
    ApiCall_vkTransitionImageLayoutEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x12d3),
    ApiCall_vkGetPipelineIndirectMemoryRequirementsNV                                             = MakeApiCallId(ApiFamily_Vulkan, 0x12d4),
    ApiCall_vkCmdUpdatePipelineIndirectBuffer                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x12d5),
    ApiCall_vkGetPipelineIndirectDeviceAddressNV                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x12d6),
    ApiCall_vkCmdUpdatePipelineIndirectBufferNV                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x12d7),
    ApiCall_vkCmdBindIndexBuffer2KHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x12d8),
    ApiCall_vkGetRenderingAreaGranularityKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x12d9),
    ApiCall_vkGetDeviceImageSubresourceLayoutKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x12da),
    ApiCall_vkGetImageSubresourceLayout2KHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x12db),
    ApiCall_vkFrameBoundaryANDROID                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12dc),
    ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsKHR                                        = MakeApiCallId(ApiFamily_Vulkan, 0x12dd),
    ApiCall_vkGetCalibratedTimestampsKHR                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x12de),
    ApiCall_vkCmdBindDescriptorSets2KHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x12df),
    ApiCall_vkCmdPushConstants2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12e0),
    ApiCall_vkCmdPushDescriptorSet2KHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x12e1),
    ApiCall_vkCmdPushDescriptorSetWithTemplate2KHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x12e2),
    ApiCall_vkCmdSetDescriptorBufferOffsets2EXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x12e3),
    ApiCall_vkCmdBindDescriptorBufferEmbeddedSamplers2EXT                                         = MakeApiCallId(ApiFamily_Vulkan, 0x12e4),

    ApiCall_VulkanLast,

    // DXGI API
    // dxgi.h
    ApiCall_CreateDXGIFactory                                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1000),
    ApiCall_CreateDXGIFactory1                                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1001),
    ApiCall_IDXGIObject_SetPrivateData                                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1002),
    ApiCall_IDXGIObject_SetPrivateDataInterface                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1003),
    ApiCall_IDXGIObject_GetPrivateData                                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1004),
    ApiCall_IDXGIObject_GetParent                                                                 = MakeApiCallId(ApiFamily_Dxgi, 0x1005),
    ApiCall_IDXGIDeviceSubObject_GetDevice                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1006),
    ApiCall_IDXGIResource_GetSharedHandle                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x1007),
    ApiCall_IDXGIResource_GetUsage                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1008),
    ApiCall_IDXGIResource_SetEvictionPriority                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1009),
    ApiCall_IDXGIResource_GetEvictionPriority                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x100a),
    ApiCall_IDXGIKeyedMutex_AcquireSync                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x100b),
    ApiCall_IDXGIKeyedMutex_ReleaseSync                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x100c),
    ApiCall_IDXGISurface_GetDesc                                                                  = MakeApiCallId(ApiFamily_Dxgi, 0x100d),
    ApiCall_IDXGISurface_Map                                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x100e),
    ApiCall_IDXGISurface_Unmap                                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x100f),
    ApiCall_IDXGISurface1_GetDC                                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1010),
    ApiCall_IDXGISurface1_ReleaseDC                                                               = MakeApiCallId(ApiFamily_Dxgi, 0x1011),
    ApiCall_IDXGIAdapter_EnumOutputs                                                              = MakeApiCallId(ApiFamily_Dxgi, 0x1012),
    ApiCall_IDXGIAdapter_GetDesc                                                                  = MakeApiCallId(ApiFamily_Dxgi, 0x1013),
    ApiCall_IDXGIAdapter_CheckInterfaceSupport                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1014),
    ApiCall_IDXGIOutput_GetDesc                                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1015),
    ApiCall_IDXGIOutput_GetDisplayModeList                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1016),
    ApiCall_IDXGIOutput_FindClosestMatchingMode                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1017),
    ApiCall_IDXGIOutput_WaitForVBlank                                                             = MakeApiCallId(ApiFamily_Dxgi, 0x1018),
    ApiCall_IDXGIOutput_TakeOwnership                                                             = MakeApiCallId(ApiFamily_Dxgi, 0x1019),
    ApiCall_IDXGIOutput_ReleaseOwnership                                                          = MakeApiCallId(ApiFamily_Dxgi, 0x101a),
    ApiCall_IDXGIOutput_GetGammaControlCapabilities                                               = MakeApiCallId(ApiFamily_Dxgi, 0x101b),
    ApiCall_IDXGIOutput_SetGammaControl                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x101c),
    ApiCall_IDXGIOutput_GetGammaControl                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x101d),
    ApiCall_IDXGIOutput_SetDisplaySurface                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x101e),
    ApiCall_IDXGIOutput_GetDisplaySurfaceData                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x101f),
    ApiCall_IDXGIOutput_GetFrameStatistics                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1020),
    ApiCall_IDXGISwapChain_Present                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1021),
    ApiCall_IDXGISwapChain_GetBuffer                                                              = MakeApiCallId(ApiFamily_Dxgi, 0x1022),
    ApiCall_IDXGISwapChain_SetFullscreenState                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1023),
    ApiCall_IDXGISwapChain_GetFullscreenState                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1024),
    ApiCall_IDXGISwapChain_GetDesc                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1025),
    ApiCall_IDXGISwapChain_ResizeBuffers                                                          = MakeApiCallId(ApiFamily_Dxgi, 0x1026),
    ApiCall_IDXGISwapChain_ResizeTarget                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x1027),
    ApiCall_IDXGISwapChain_GetContainingOutput                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1028),
    ApiCall_IDXGISwapChain_GetFrameStatistics                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1029),
    ApiCall_IDXGISwapChain_GetLastPresentCount                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x102a),
    ApiCall_IDXGIFactory_EnumAdapters                                                             = MakeApiCallId(ApiFamily_Dxgi, 0x102b),
    ApiCall_IDXGIFactory_MakeWindowAssociation                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x102c),
    ApiCall_IDXGIFactory_GetWindowAssociation                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x102d),
    ApiCall_IDXGIFactory_CreateSwapChain                                                          = MakeApiCallId(ApiFamily_Dxgi, 0x102e),
    ApiCall_IDXGIFactory_CreateSoftwareAdapter                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x102f),
    ApiCall_IDXGIDevice_GetAdapter                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1030),
    ApiCall_IDXGIDevice_CreateSurface                                                             = MakeApiCallId(ApiFamily_Dxgi, 0x1031),
    ApiCall_IDXGIDevice_QueryResourceResidency                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1032),
    ApiCall_IDXGIDevice_SetGPUThreadPriority                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x1033),
    ApiCall_IDXGIDevice_GetGPUThreadPriority                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x1034),
    ApiCall_IDXGIFactory1_EnumAdapters1                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x1035),
    ApiCall_IDXGIFactory1_IsCurrent                                                               = MakeApiCallId(ApiFamily_Dxgi, 0x1036),
    ApiCall_IDXGIAdapter1_GetDesc1                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1037),
    ApiCall_IDXGIDevice1_SetMaximumFrameLatency                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1038),
    ApiCall_IDXGIDevice1_GetMaximumFrameLatency                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1039),

    // dxgi1_2.h
    ApiCall_IDXGIDisplayControl_IsStereoEnabled                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x103a),
    ApiCall_IDXGIDisplayControl_SetStereoEnabled                                                  = MakeApiCallId(ApiFamily_Dxgi, 0x103b),
    ApiCall_IDXGIOutputDuplication_GetDesc                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x103c),
    ApiCall_IDXGIOutputDuplication_AcquireNextFrame                                               = MakeApiCallId(ApiFamily_Dxgi, 0x103d),
    ApiCall_IDXGIOutputDuplication_GetFrameDirtyRects                                             = MakeApiCallId(ApiFamily_Dxgi, 0x103e),
    ApiCall_IDXGIOutputDuplication_GetFrameMoveRects                                              = MakeApiCallId(ApiFamily_Dxgi, 0x103f),
    ApiCall_IDXGIOutputDuplication_GetFramePointerShape                                           = MakeApiCallId(ApiFamily_Dxgi, 0x1040),
    ApiCall_IDXGIOutputDuplication_MapDesktopSurface                                              = MakeApiCallId(ApiFamily_Dxgi, 0x1041),
    ApiCall_IDXGIOutputDuplication_UnMapDesktopSurface                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1042),
    ApiCall_IDXGIOutputDuplication_ReleaseFrame                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1043),
    ApiCall_IDXGISurface2_GetResource                                                             = MakeApiCallId(ApiFamily_Dxgi, 0x1044),
    ApiCall_IDXGIResource1_CreateSubresourceSurface                                               = MakeApiCallId(ApiFamily_Dxgi, 0x1045),
    ApiCall_IDXGIResource1_CreateSharedHandle                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1046),
    ApiCall_IDXGIDevice2_OfferResources                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x1047),
    ApiCall_IDXGIDevice2_ReclaimResources                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x1048),
    ApiCall_IDXGIDevice2_EnqueueSetEvent                                                          = MakeApiCallId(ApiFamily_Dxgi, 0x1049),
    ApiCall_IDXGISwapChain1_GetDesc1                                                              = MakeApiCallId(ApiFamily_Dxgi, 0x104a),
    ApiCall_IDXGISwapChain1_GetFullscreenDesc                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x104b),
    ApiCall_IDXGISwapChain1_GetHwnd                                                               = MakeApiCallId(ApiFamily_Dxgi, 0x104c),
    ApiCall_IDXGISwapChain1_GetCoreWindow                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x104d),
    ApiCall_IDXGISwapChain1_Present1                                                              = MakeApiCallId(ApiFamily_Dxgi, 0x104e),
    ApiCall_IDXGISwapChain1_IsTemporaryMonoSupported                                              = MakeApiCallId(ApiFamily_Dxgi, 0x104f),
    ApiCall_IDXGISwapChain1_GetRestrictToOutput                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1050),
    ApiCall_IDXGISwapChain1_SetBackgroundColor                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1051),
    ApiCall_IDXGISwapChain1_GetBackgroundColor                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1052),
    ApiCall_IDXGISwapChain1_SetRotation                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x1053),
    ApiCall_IDXGISwapChain1_GetRotation                                                           = MakeApiCallId(ApiFamily_Dxgi, 0x1054),
    ApiCall_IDXGIFactory2_IsWindowedStereoEnabled                                                 = MakeApiCallId(ApiFamily_Dxgi, 0x1055),
    ApiCall_IDXGIFactory2_CreateSwapChainForHwnd                                                  = MakeApiCallId(ApiFamily_Dxgi, 0x1056),
    ApiCall_IDXGIFactory2_CreateSwapChainForCoreWindow                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1057),
    ApiCall_IDXGIFactory2_GetSharedResourceAdapterLuid                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1058),
    ApiCall_IDXGIFactory2_RegisterStereoStatusWindow                                              = MakeApiCallId(ApiFamily_Dxgi, 0x1059),
    ApiCall_IDXGIFactory2_RegisterStereoStatusEvent                                               = MakeApiCallId(ApiFamily_Dxgi, 0x105a),
    ApiCall_IDXGIFactory2_UnregisterStereoStatus                                                  = MakeApiCallId(ApiFamily_Dxgi, 0x105b),
    ApiCall_IDXGIFactory2_RegisterOcclusionStatusWindow                                           = MakeApiCallId(ApiFamily_Dxgi, 0x105c),
    ApiCall_IDXGIFactory2_RegisterOcclusionStatusEvent                                            = MakeApiCallId(ApiFamily_Dxgi, 0x105d),
    ApiCall_IDXGIFactory2_UnregisterOcclusionStatus                                               = MakeApiCallId(ApiFamily_Dxgi, 0x105e),
    ApiCall_IDXGIFactory2_CreateSwapChainForComposition                                           = MakeApiCallId(ApiFamily_Dxgi, 0x105f),
    ApiCall_IDXGIAdapter2_GetDesc2                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1060),
    ApiCall_IDXGIOutput1_GetDisplayModeList1                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x1061),
    ApiCall_IDXGIOutput1_FindClosestMatchingMode1                                                 = MakeApiCallId(ApiFamily_Dxgi, 0x1062),
    ApiCall_IDXGIOutput1_GetDisplaySurfaceData1                                                   = MakeApiCallId(ApiFamily_Dxgi, 0x1063),
    ApiCall_IDXGIOutput1_DuplicateOutput                                                          = MakeApiCallId(ApiFamily_Dxgi, 0x1064),

    // dxgi1_3.h
    ApiCall_CreateDXGIFactory2                                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1065),
    ApiCall_DXGIGetDebugInterface1                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1066),
    ApiCall_IDXGIDevice3_Trim                                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1067),
    ApiCall_IDXGISwapChain2_SetSourceSize                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x1068),
    ApiCall_IDXGISwapChain2_GetSourceSize                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x1069),
    ApiCall_IDXGISwapChain2_SetMaximumFrameLatency                                                = MakeApiCallId(ApiFamily_Dxgi, 0x106a),
    ApiCall_IDXGISwapChain2_GetMaximumFrameLatency                                                = MakeApiCallId(ApiFamily_Dxgi, 0x106b),
    ApiCall_IDXGISwapChain2_GetFrameLatencyWaitableObject                                         = MakeApiCallId(ApiFamily_Dxgi, 0x106c),
    ApiCall_IDXGISwapChain2_SetMatrixTransform                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x106d),
    ApiCall_IDXGISwapChain2_GetMatrixTransform                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x106e),
    ApiCall_IDXGIOutput2_SupportsOverlays                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x106f),
    ApiCall_IDXGIFactory3_GetCreationFlags                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1070),
    ApiCall_IDXGIDecodeSwapChain_PresentBuffer                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1071),
    ApiCall_IDXGIDecodeSwapChain_SetSourceRect                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1072),
    ApiCall_IDXGIDecodeSwapChain_SetTargetRect                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1073),
    ApiCall_IDXGIDecodeSwapChain_SetDestSize                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x1074),
    ApiCall_IDXGIDecodeSwapChain_GetSourceRect                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1075),
    ApiCall_IDXGIDecodeSwapChain_GetTargetRect                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1076),
    ApiCall_IDXGIDecodeSwapChain_GetDestSize                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x1077),
    ApiCall_IDXGIDecodeSwapChain_SetColorSpace                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1078),
    ApiCall_IDXGIDecodeSwapChain_GetColorSpace                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1079),
    ApiCall_IDXGIFactoryMedia_CreateSwapChainForCompositionSurfaceHandle                          = MakeApiCallId(ApiFamily_Dxgi, 0x107a),
    ApiCall_IDXGIFactoryMedia_CreateDecodeSwapChainForCompositionSurfaceHandle                    = MakeApiCallId(ApiFamily_Dxgi, 0x107b),
    ApiCall_IDXGISwapChainMedia_GetFrameStatisticsMedia                                           = MakeApiCallId(ApiFamily_Dxgi, 0x107c),
    ApiCall_IDXGISwapChainMedia_SetPresentDuration                                                = MakeApiCallId(ApiFamily_Dxgi, 0x107d),
    ApiCall_IDXGISwapChainMedia_CheckPresentDurationSupport                                       = MakeApiCallId(ApiFamily_Dxgi, 0x107e),
    ApiCall_IDXGIOutput3_CheckOverlaySupport                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x107f),

    // dxgi1_4.h
    ApiCall_IDXGISwapChain3_GetCurrentBackBufferIndex                                             = MakeApiCallId(ApiFamily_Dxgi, 0x1080),
    ApiCall_IDXGISwapChain3_CheckColorSpaceSupport                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1081),
    ApiCall_IDXGISwapChain3_SetColorSpace1                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1082),
    ApiCall_IDXGISwapChain3_ResizeBuffers1                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1083),
    ApiCall_IDXGIOutput4_CheckOverlayColorSpaceSupport                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1084),
    ApiCall_IDXGIFactory4_EnumAdapterByLuid                                                       = MakeApiCallId(ApiFamily_Dxgi, 0x1085),
    ApiCall_IDXGIFactory4_EnumWarpAdapter                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x1086),
    ApiCall_IDXGIAdapter3_RegisterHardwareContentProtectionTeardownStatusEvent                    = MakeApiCallId(ApiFamily_Dxgi, 0x1087),
    ApiCall_IDXGIAdapter3_UnregisterHardwareContentProtectionTeardownStatus                       = MakeApiCallId(ApiFamily_Dxgi, 0x1088),
    ApiCall_IDXGIAdapter3_QueryVideoMemoryInfo                                                    = MakeApiCallId(ApiFamily_Dxgi, 0x1089),
    ApiCall_IDXGIAdapter3_SetVideoMemoryReservation                                               = MakeApiCallId(ApiFamily_Dxgi, 0x108a),
    ApiCall_IDXGIAdapter3_RegisterVideoMemoryBudgetChangeNotificationEvent                        = MakeApiCallId(ApiFamily_Dxgi, 0x108b),
    ApiCall_IDXGIAdapter3_UnregisterVideoMemoryBudgetChangeNotification                           = MakeApiCallId(ApiFamily_Dxgi, 0x108c),

    // dxgi1_5.h
    ApiCall_IDXGIOutput5_DuplicateOutput1                                                         = MakeApiCallId(ApiFamily_Dxgi, 0x108d),
    ApiCall_IDXGISwapChain4_SetHDRMetaData                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x108e),
    ApiCall_IDXGIDevice4_OfferResources1                                                          = MakeApiCallId(ApiFamily_Dxgi, 0x108f),
    ApiCall_IDXGIDevice4_ReclaimResources1                                                        = MakeApiCallId(ApiFamily_Dxgi, 0x1090),
    ApiCall_IDXGIFactory5_CheckFeatureSupport                                                     = MakeApiCallId(ApiFamily_Dxgi, 0x1091),

    // dxgi1_6.h
    ApiCall_DXGIDeclareAdapterRemovalSupport                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x1092),
    ApiCall_IDXGIAdapter4_GetDesc3                                                                = MakeApiCallId(ApiFamily_Dxgi, 0x1093),
    ApiCall_IDXGIOutput6_GetDesc1                                                                 = MakeApiCallId(ApiFamily_Dxgi, 0x1094),
    ApiCall_IDXGIOutput6_CheckHardwareCompositionSupport                                          = MakeApiCallId(ApiFamily_Dxgi, 0x1095),
    ApiCall_IDXGIFactory6_EnumAdapterByGpuPreference                                              = MakeApiCallId(ApiFamily_Dxgi, 0x1096),
    ApiCall_IDXGIFactory7_RegisterAdaptersChangedEvent                                            = MakeApiCallId(ApiFamily_Dxgi, 0x1097),
    ApiCall_IDXGIFactory7_UnregisterAdaptersChangedEvent                                          = MakeApiCallId(ApiFamily_Dxgi, 0x1098),

    // Unknwnbase.h
    ApiCall_IUnknown_QueryInterface                                                               = MakeApiCallId(ApiFamily_Dxgi, 0x1099),
    ApiCall_IUnknown_AddRef                                                                       = MakeApiCallId(ApiFamily_Dxgi, 0x109a),
    ApiCall_IUnknown_Release                                                                      = MakeApiCallId(ApiFamily_Dxgi, 0x109b),

    ApiCall_DXGILast,

    // D3D12 API
    // d3d12.h
    ApiCall_D3D12SerializeRootSignature                                                           = MakeApiCallId(ApiFamily_D3D12, 0x1000),
    ApiCall_D3D12CreateRootSignatureDeserializer                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1001),
    ApiCall_D3D12SerializeVersionedRootSignature                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1002),
    ApiCall_D3D12CreateVersionedRootSignatureDeserializer                                         = MakeApiCallId(ApiFamily_D3D12, 0x1003),
    ApiCall_D3D12CreateDevice                                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1004),
    ApiCall_D3D12GetDebugInterface                                                                = MakeApiCallId(ApiFamily_D3D12, 0x1005),
    ApiCall_D3D12EnableExperimentalFeatures                                                       = MakeApiCallId(ApiFamily_D3D12, 0x1006),
    ApiCall_ID3D12Object_GetPrivateData                                                           = MakeApiCallId(ApiFamily_D3D12, 0x1007),
    ApiCall_ID3D12Object_SetPrivateData                                                           = MakeApiCallId(ApiFamily_D3D12, 0x1008),
    ApiCall_ID3D12Object_SetPrivateDataInterface                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1009),
    ApiCall_ID3D12Object_SetName                                                                  = MakeApiCallId(ApiFamily_D3D12, 0x100a),
    ApiCall_ID3D12DeviceChild_GetDevice                                                           = MakeApiCallId(ApiFamily_D3D12, 0x100b),
    ApiCall_ID3D12RootSignatureDeserializer_GetRootSignatureDesc                                  = MakeApiCallId(ApiFamily_D3D12, 0x100c),
    ApiCall_ID3D12VersionedRootSignatureDeserializer_GetRootSignatureDescAtVersion                = MakeApiCallId(ApiFamily_D3D12, 0x100d),
    ApiCall_ID3D12VersionedRootSignatureDeserializer_GetUnconvertedRootSignatureDesc              = MakeApiCallId(ApiFamily_D3D12, 0x100e),
    ApiCall_ID3D12Heap_GetDesc                                                                    = MakeApiCallId(ApiFamily_D3D12, 0x100f),
    ApiCall_ID3D12Resource_Map                                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1010),
    ApiCall_ID3D12Resource_Unmap                                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1011),
    ApiCall_ID3D12Resource_GetDesc                                                                = MakeApiCallId(ApiFamily_D3D12, 0x1012),
    ApiCall_ID3D12Resource_GetGPUVirtualAddress                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1013),
    ApiCall_ID3D12Resource_WriteToSubresource                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1014),
    ApiCall_ID3D12Resource_ReadFromSubresource                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1015),
    ApiCall_ID3D12Resource_GetHeapProperties                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1016),
    ApiCall_ID3D12CommandAllocator_Reset                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1017),
    ApiCall_ID3D12Fence_GetCompletedValue                                                         = MakeApiCallId(ApiFamily_D3D12, 0x1018),
    ApiCall_ID3D12Fence_SetEventOnCompletion                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1019),
    ApiCall_ID3D12Fence_Signal                                                                    = MakeApiCallId(ApiFamily_D3D12, 0x101a),
    ApiCall_ID3D12Fence1_GetCreationFlags                                                         = MakeApiCallId(ApiFamily_D3D12, 0x101b),
    ApiCall_ID3D12PipelineState_GetCachedBlob                                                     = MakeApiCallId(ApiFamily_D3D12, 0x101c),
    ApiCall_ID3D12DescriptorHeap_GetDesc                                                          = MakeApiCallId(ApiFamily_D3D12, 0x101d),
    ApiCall_ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart                               = MakeApiCallId(ApiFamily_D3D12, 0x101e),
    ApiCall_ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart                               = MakeApiCallId(ApiFamily_D3D12, 0x101f),
    ApiCall_ID3D12CommandList_GetType                                                             = MakeApiCallId(ApiFamily_D3D12, 0x1020),
    ApiCall_ID3D12GraphicsCommandList_Close                                                       = MakeApiCallId(ApiFamily_D3D12, 0x1021),
    ApiCall_ID3D12GraphicsCommandList_Reset                                                       = MakeApiCallId(ApiFamily_D3D12, 0x1022),
    ApiCall_ID3D12GraphicsCommandList_ClearState                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1023),
    ApiCall_ID3D12GraphicsCommandList_DrawInstanced                                               = MakeApiCallId(ApiFamily_D3D12, 0x1024),
    ApiCall_ID3D12GraphicsCommandList_DrawIndexedInstanced                                        = MakeApiCallId(ApiFamily_D3D12, 0x1025),
    ApiCall_ID3D12GraphicsCommandList_Dispatch                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1026),
    ApiCall_ID3D12GraphicsCommandList_CopyBufferRegion                                            = MakeApiCallId(ApiFamily_D3D12, 0x1027),
    ApiCall_ID3D12GraphicsCommandList_CopyTextureRegion                                           = MakeApiCallId(ApiFamily_D3D12, 0x1028),
    ApiCall_ID3D12GraphicsCommandList_CopyResource                                                = MakeApiCallId(ApiFamily_D3D12, 0x1029),
    ApiCall_ID3D12GraphicsCommandList_CopyTiles                                                   = MakeApiCallId(ApiFamily_D3D12, 0x102a),
    ApiCall_ID3D12GraphicsCommandList_ResolveSubresource                                          = MakeApiCallId(ApiFamily_D3D12, 0x102b),
    ApiCall_ID3D12GraphicsCommandList_IASetPrimitiveTopology                                      = MakeApiCallId(ApiFamily_D3D12, 0x102c),
    ApiCall_ID3D12GraphicsCommandList_RSSetViewports                                              = MakeApiCallId(ApiFamily_D3D12, 0x102d),
    ApiCall_ID3D12GraphicsCommandList_RSSetScissorRects                                           = MakeApiCallId(ApiFamily_D3D12, 0x102e),
    ApiCall_ID3D12GraphicsCommandList_OMSetBlendFactor                                            = MakeApiCallId(ApiFamily_D3D12, 0x102f),
    ApiCall_ID3D12GraphicsCommandList_OMSetStencilRef                                             = MakeApiCallId(ApiFamily_D3D12, 0x1030),
    ApiCall_ID3D12GraphicsCommandList_SetPipelineState                                            = MakeApiCallId(ApiFamily_D3D12, 0x1031),
    ApiCall_ID3D12GraphicsCommandList_ResourceBarrier                                             = MakeApiCallId(ApiFamily_D3D12, 0x1032),
    ApiCall_ID3D12GraphicsCommandList_ExecuteBundle                                               = MakeApiCallId(ApiFamily_D3D12, 0x1033),
    ApiCall_ID3D12GraphicsCommandList_SetDescriptorHeaps                                          = MakeApiCallId(ApiFamily_D3D12, 0x1034),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRootSignature                                     = MakeApiCallId(ApiFamily_D3D12, 0x1035),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRootSignature                                    = MakeApiCallId(ApiFamily_D3D12, 0x1036),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRootDescriptorTable                               = MakeApiCallId(ApiFamily_D3D12, 0x1037),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRootDescriptorTable                              = MakeApiCallId(ApiFamily_D3D12, 0x1038),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRoot32BitConstant                                 = MakeApiCallId(ApiFamily_D3D12, 0x1039),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstant                                = MakeApiCallId(ApiFamily_D3D12, 0x103a),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRoot32BitConstants                                = MakeApiCallId(ApiFamily_D3D12, 0x103b),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants                               = MakeApiCallId(ApiFamily_D3D12, 0x103c),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRootConstantBufferView                            = MakeApiCallId(ApiFamily_D3D12, 0x103d),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRootConstantBufferView                           = MakeApiCallId(ApiFamily_D3D12, 0x103e),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRootShaderResourceView                            = MakeApiCallId(ApiFamily_D3D12, 0x103f),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRootShaderResourceView                           = MakeApiCallId(ApiFamily_D3D12, 0x1040),
    ApiCall_ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView                           = MakeApiCallId(ApiFamily_D3D12, 0x1041),
    ApiCall_ID3D12GraphicsCommandList_SetGraphicsRootUnorderedAccessView                          = MakeApiCallId(ApiFamily_D3D12, 0x1042),
    ApiCall_ID3D12GraphicsCommandList_IASetIndexBuffer                                            = MakeApiCallId(ApiFamily_D3D12, 0x1043),
    ApiCall_ID3D12GraphicsCommandList_IASetVertexBuffers                                          = MakeApiCallId(ApiFamily_D3D12, 0x1044),
    ApiCall_ID3D12GraphicsCommandList_SOSetTargets                                                = MakeApiCallId(ApiFamily_D3D12, 0x1045),
    ApiCall_ID3D12GraphicsCommandList_OMSetRenderTargets                                          = MakeApiCallId(ApiFamily_D3D12, 0x1046),
    ApiCall_ID3D12GraphicsCommandList_ClearDepthStencilView                                       = MakeApiCallId(ApiFamily_D3D12, 0x1047),
    ApiCall_ID3D12GraphicsCommandList_ClearRenderTargetView                                       = MakeApiCallId(ApiFamily_D3D12, 0x1048),
    ApiCall_ID3D12GraphicsCommandList_ClearUnorderedAccessViewUint                                = MakeApiCallId(ApiFamily_D3D12, 0x1049),
    ApiCall_ID3D12GraphicsCommandList_ClearUnorderedAccessViewFloat                               = MakeApiCallId(ApiFamily_D3D12, 0x104a),
    ApiCall_ID3D12GraphicsCommandList_DiscardResource                                             = MakeApiCallId(ApiFamily_D3D12, 0x104b),
    ApiCall_ID3D12GraphicsCommandList_BeginQuery                                                  = MakeApiCallId(ApiFamily_D3D12, 0x104c),
    ApiCall_ID3D12GraphicsCommandList_EndQuery                                                    = MakeApiCallId(ApiFamily_D3D12, 0x104d),
    ApiCall_ID3D12GraphicsCommandList_ResolveQueryData                                            = MakeApiCallId(ApiFamily_D3D12, 0x104e),
    ApiCall_ID3D12GraphicsCommandList_SetPredication                                              = MakeApiCallId(ApiFamily_D3D12, 0x104f),
    ApiCall_ID3D12GraphicsCommandList_SetMarker                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1050),
    ApiCall_ID3D12GraphicsCommandList_BeginEvent                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1051),
    ApiCall_ID3D12GraphicsCommandList_EndEvent                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1052),
    ApiCall_ID3D12GraphicsCommandList_ExecuteIndirect                                             = MakeApiCallId(ApiFamily_D3D12, 0x1053),
    ApiCall_ID3D12GraphicsCommandList1_AtomicCopyBufferUINT                                       = MakeApiCallId(ApiFamily_D3D12, 0x1054),
    ApiCall_ID3D12GraphicsCommandList1_AtomicCopyBufferUINT64                                     = MakeApiCallId(ApiFamily_D3D12, 0x1055),
    ApiCall_ID3D12GraphicsCommandList1_OMSetDepthBounds                                           = MakeApiCallId(ApiFamily_D3D12, 0x1056),
    ApiCall_ID3D12GraphicsCommandList1_SetSamplePositions                                         = MakeApiCallId(ApiFamily_D3D12, 0x1057),
    ApiCall_ID3D12GraphicsCommandList1_ResolveSubresourceRegion                                   = MakeApiCallId(ApiFamily_D3D12, 0x1058),
    ApiCall_ID3D12GraphicsCommandList1_SetViewInstanceMask                                        = MakeApiCallId(ApiFamily_D3D12, 0x1059),
    ApiCall_ID3D12GraphicsCommandList2_WriteBufferImmediate                                       = MakeApiCallId(ApiFamily_D3D12, 0x105a),
    ApiCall_ID3D12CommandQueue_UpdateTileMappings                                                 = MakeApiCallId(ApiFamily_D3D12, 0x105b),
    ApiCall_ID3D12CommandQueue_CopyTileMappings                                                   = MakeApiCallId(ApiFamily_D3D12, 0x105c),
    ApiCall_ID3D12CommandQueue_ExecuteCommandLists                                                = MakeApiCallId(ApiFamily_D3D12, 0x105d),
    ApiCall_ID3D12CommandQueue_SetMarker                                                          = MakeApiCallId(ApiFamily_D3D12, 0x105e),
    ApiCall_ID3D12CommandQueue_BeginEvent                                                         = MakeApiCallId(ApiFamily_D3D12, 0x105f),
    ApiCall_ID3D12CommandQueue_EndEvent                                                           = MakeApiCallId(ApiFamily_D3D12, 0x1060),
    ApiCall_ID3D12CommandQueue_Signal                                                             = MakeApiCallId(ApiFamily_D3D12, 0x1061),
    ApiCall_ID3D12CommandQueue_Wait                                                               = MakeApiCallId(ApiFamily_D3D12, 0x1062),
    ApiCall_ID3D12CommandQueue_GetTimestampFrequency                                              = MakeApiCallId(ApiFamily_D3D12, 0x1063),
    ApiCall_ID3D12CommandQueue_GetClockCalibration                                                = MakeApiCallId(ApiFamily_D3D12, 0x1064),
    ApiCall_ID3D12CommandQueue_GetDesc                                                            = MakeApiCallId(ApiFamily_D3D12, 0x1065),
    ApiCall_ID3D12Device_GetNodeCount                                                             = MakeApiCallId(ApiFamily_D3D12, 0x1066),
    ApiCall_ID3D12Device_CreateCommandQueue                                                       = MakeApiCallId(ApiFamily_D3D12, 0x1067),
    ApiCall_ID3D12Device_CreateCommandAllocator                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1068),
    ApiCall_ID3D12Device_CreateGraphicsPipelineState                                              = MakeApiCallId(ApiFamily_D3D12, 0x1069),
    ApiCall_ID3D12Device_CreateComputePipelineState                                               = MakeApiCallId(ApiFamily_D3D12, 0x106a),
    ApiCall_ID3D12Device_CreateCommandList                                                        = MakeApiCallId(ApiFamily_D3D12, 0x106b),
    ApiCall_ID3D12Device_CheckFeatureSupport                                                      = MakeApiCallId(ApiFamily_D3D12, 0x106c),
    ApiCall_ID3D12Device_CreateDescriptorHeap                                                     = MakeApiCallId(ApiFamily_D3D12, 0x106d),
    ApiCall_ID3D12Device_GetDescriptorHandleIncrementSize                                         = MakeApiCallId(ApiFamily_D3D12, 0x106e),
    ApiCall_ID3D12Device_CreateRootSignature                                                      = MakeApiCallId(ApiFamily_D3D12, 0x106f),
    ApiCall_ID3D12Device_CreateConstantBufferView                                                 = MakeApiCallId(ApiFamily_D3D12, 0x1070),
    ApiCall_ID3D12Device_CreateShaderResourceView                                                 = MakeApiCallId(ApiFamily_D3D12, 0x1071),
    ApiCall_ID3D12Device_CreateUnorderedAccessView                                                = MakeApiCallId(ApiFamily_D3D12, 0x1072),
    ApiCall_ID3D12Device_CreateRenderTargetView                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1073),
    ApiCall_ID3D12Device_CreateDepthStencilView                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1074),
    ApiCall_ID3D12Device_CreateSampler                                                            = MakeApiCallId(ApiFamily_D3D12, 0x1075),
    ApiCall_ID3D12Device_CopyDescriptors                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1076),
    ApiCall_ID3D12Device_CopyDescriptorsSimple                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1077),
    ApiCall_ID3D12Device_GetResourceAllocationInfo                                                = MakeApiCallId(ApiFamily_D3D12, 0x1078),
    ApiCall_ID3D12Device_GetCustomHeapProperties                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1079),
    ApiCall_ID3D12Device_CreateCommittedResource                                                  = MakeApiCallId(ApiFamily_D3D12, 0x107a),
    ApiCall_ID3D12Device_CreateHeap                                                               = MakeApiCallId(ApiFamily_D3D12, 0x107b),
    ApiCall_ID3D12Device_CreatePlacedResource                                                     = MakeApiCallId(ApiFamily_D3D12, 0x107c),
    ApiCall_ID3D12Device_CreateReservedResource                                                   = MakeApiCallId(ApiFamily_D3D12, 0x107d),
    ApiCall_ID3D12Device_CreateSharedHandle                                                       = MakeApiCallId(ApiFamily_D3D12, 0x107e),
    ApiCall_ID3D12Device_OpenSharedHandle                                                         = MakeApiCallId(ApiFamily_D3D12, 0x107f),
    ApiCall_ID3D12Device_OpenSharedHandleByName                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1080),
    ApiCall_ID3D12Device_MakeResident                                                             = MakeApiCallId(ApiFamily_D3D12, 0x1081),
    ApiCall_ID3D12Device_Evict                                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1082),
    ApiCall_ID3D12Device_CreateFence                                                              = MakeApiCallId(ApiFamily_D3D12, 0x1083),
    ApiCall_ID3D12Device_GetDeviceRemovedReason                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1084),
    ApiCall_ID3D12Device_GetCopyableFootprints                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1085),
    ApiCall_ID3D12Device_CreateQueryHeap                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1086),
    ApiCall_ID3D12Device_SetStablePowerState                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1087),
    ApiCall_ID3D12Device_CreateCommandSignature                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1088),
    ApiCall_ID3D12Device_GetResourceTiling                                                        = MakeApiCallId(ApiFamily_D3D12, 0x1089),
    ApiCall_ID3D12Device_GetAdapterLuid                                                           = MakeApiCallId(ApiFamily_D3D12, 0x108a),
    ApiCall_ID3D12PipelineLibrary_StorePipeline                                                   = MakeApiCallId(ApiFamily_D3D12, 0x108b),
    ApiCall_ID3D12PipelineLibrary_LoadGraphicsPipeline                                            = MakeApiCallId(ApiFamily_D3D12, 0x108c),
    ApiCall_ID3D12PipelineLibrary_LoadComputePipeline                                             = MakeApiCallId(ApiFamily_D3D12, 0x108d),
    ApiCall_ID3D12PipelineLibrary_GetSerializedSize                                               = MakeApiCallId(ApiFamily_D3D12, 0x108e),
    ApiCall_ID3D12PipelineLibrary_Serialize                                                       = MakeApiCallId(ApiFamily_D3D12, 0x108f),
    ApiCall_ID3D12PipelineLibrary1_LoadPipeline                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1090),
    ApiCall_ID3D12Device1_CreatePipelineLibrary                                                   = MakeApiCallId(ApiFamily_D3D12, 0x1091),
    ApiCall_ID3D12Device1_SetEventOnMultipleFenceCompletion                                       = MakeApiCallId(ApiFamily_D3D12, 0x1092),
    ApiCall_ID3D12Device1_SetResidencyPriority                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1093),
    ApiCall_ID3D12Device2_CreatePipelineState                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1094),
    ApiCall_ID3D12Device3_OpenExistingHeapFromAddress                                             = MakeApiCallId(ApiFamily_D3D12, 0x1095),
    ApiCall_ID3D12Device3_OpenExistingHeapFromFileMapping                                         = MakeApiCallId(ApiFamily_D3D12, 0x1096),
    ApiCall_ID3D12Device3_EnqueueMakeResident                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1097),
    ApiCall_ID3D12ProtectedSession_GetStatusFence                                                 = MakeApiCallId(ApiFamily_D3D12, 0x1098),
    ApiCall_ID3D12ProtectedSession_GetSessionStatus                                               = MakeApiCallId(ApiFamily_D3D12, 0x1099),
    ApiCall_ID3D12ProtectedResourceSession_GetDesc                                                = MakeApiCallId(ApiFamily_D3D12, 0x109a),
    ApiCall_ID3D12Device4_CreateCommandList1                                                      = MakeApiCallId(ApiFamily_D3D12, 0x109b),
    ApiCall_ID3D12Device4_CreateProtectedResourceSession                                          = MakeApiCallId(ApiFamily_D3D12, 0x109c),
    ApiCall_ID3D12Device4_CreateCommittedResource1                                                = MakeApiCallId(ApiFamily_D3D12, 0x109d),
    ApiCall_ID3D12Device4_CreateHeap1                                                             = MakeApiCallId(ApiFamily_D3D12, 0x109e),
    ApiCall_ID3D12Device4_CreateReservedResource1                                                 = MakeApiCallId(ApiFamily_D3D12, 0x109f),
    ApiCall_ID3D12Device4_GetResourceAllocationInfo1                                              = MakeApiCallId(ApiFamily_D3D12, 0x10a0),
    ApiCall_ID3D12LifetimeOwner_LifetimeStateUpdated                                              = MakeApiCallId(ApiFamily_D3D12, 0x10a1),
    ApiCall_ID3D12SwapChainAssistant_GetLUID                                                      = MakeApiCallId(ApiFamily_D3D12, 0x10a2),
    ApiCall_ID3D12SwapChainAssistant_GetSwapChainObject                                           = MakeApiCallId(ApiFamily_D3D12, 0x10a3),
    ApiCall_ID3D12SwapChainAssistant_GetCurrentResourceAndCommandQueue                            = MakeApiCallId(ApiFamily_D3D12, 0x10a4),
    ApiCall_ID3D12SwapChainAssistant_InsertImplicitSync                                           = MakeApiCallId(ApiFamily_D3D12, 0x10a5),
    ApiCall_ID3D12LifetimeTracker_DestroyOwnedObject                                              = MakeApiCallId(ApiFamily_D3D12, 0x10a6),
    ApiCall_ID3D12StateObjectProperties_GetShaderIdentifier                                       = MakeApiCallId(ApiFamily_D3D12, 0x10a7),
    ApiCall_ID3D12StateObjectProperties_GetShaderStackSize                                        = MakeApiCallId(ApiFamily_D3D12, 0x10a8),
    ApiCall_ID3D12StateObjectProperties_GetPipelineStackSize                                      = MakeApiCallId(ApiFamily_D3D12, 0x10a9),
    ApiCall_ID3D12StateObjectProperties_SetPipelineStackSize                                      = MakeApiCallId(ApiFamily_D3D12, 0x10aa),
    ApiCall_ID3D12Device5_CreateLifetimeTracker                                                   = MakeApiCallId(ApiFamily_D3D12, 0x10ab),
    ApiCall_ID3D12Device5_RemoveDevice                                                            = MakeApiCallId(ApiFamily_D3D12, 0x10ac),
    ApiCall_ID3D12Device5_EnumerateMetaCommands                                                   = MakeApiCallId(ApiFamily_D3D12, 0x10ad),
    ApiCall_ID3D12Device5_EnumerateMetaCommandParameters                                          = MakeApiCallId(ApiFamily_D3D12, 0x10ae),
    ApiCall_ID3D12Device5_CreateMetaCommand                                                       = MakeApiCallId(ApiFamily_D3D12, 0x10af),
    ApiCall_ID3D12Device5_CreateStateObject                                                       = MakeApiCallId(ApiFamily_D3D12, 0x10b0),
    ApiCall_ID3D12Device5_GetRaytracingAccelerationStructurePrebuildInfo                          = MakeApiCallId(ApiFamily_D3D12, 0x10b1),
    ApiCall_ID3D12Device5_CheckDriverMatchingIdentifier                                           = MakeApiCallId(ApiFamily_D3D12, 0x10b2),
    ApiCall_ID3D12DeviceRemovedExtendedDataSettings_SetAutoBreadcrumbsEnablement                  = MakeApiCallId(ApiFamily_D3D12, 0x10b3),
    ApiCall_ID3D12DeviceRemovedExtendedDataSettings_SetPageFaultEnablement                        = MakeApiCallId(ApiFamily_D3D12, 0x10b4),
    ApiCall_ID3D12DeviceRemovedExtendedDataSettings_SetWatsonDumpEnablement                       = MakeApiCallId(ApiFamily_D3D12, 0x10b5),
    ApiCall_ID3D12DeviceRemovedExtendedDataSettings1_SetBreadcrumbContextEnablement               = MakeApiCallId(ApiFamily_D3D12, 0x10b6),
    ApiCall_ID3D12DeviceRemovedExtendedData_GetAutoBreadcrumbsOutput                              = MakeApiCallId(ApiFamily_D3D12, 0x10b7),
    ApiCall_ID3D12DeviceRemovedExtendedData_GetPageFaultAllocationOutput                          = MakeApiCallId(ApiFamily_D3D12, 0x10b8),
    ApiCall_ID3D12DeviceRemovedExtendedData1_GetAutoBreadcrumbsOutput1                            = MakeApiCallId(ApiFamily_D3D12, 0x10b9),
    ApiCall_ID3D12DeviceRemovedExtendedData1_GetPageFaultAllocationOutput1                        = MakeApiCallId(ApiFamily_D3D12, 0x10ba),
    ApiCall_ID3D12Device6_SetBackgroundProcessingMode                                             = MakeApiCallId(ApiFamily_D3D12, 0x10bb),
    ApiCall_ID3D12ProtectedResourceSession1_GetDesc1                                              = MakeApiCallId(ApiFamily_D3D12, 0x10bc),
    ApiCall_ID3D12Device7_AddToStateObject                                                        = MakeApiCallId(ApiFamily_D3D12, 0x10bd),
    ApiCall_ID3D12Device7_CreateProtectedResourceSession1                                         = MakeApiCallId(ApiFamily_D3D12, 0x10be),
    ApiCall_ID3D12Device8_GetResourceAllocationInfo2                                              = MakeApiCallId(ApiFamily_D3D12, 0x10bf),
    ApiCall_ID3D12Device8_CreateCommittedResource2                                                = MakeApiCallId(ApiFamily_D3D12, 0x10c0),
    ApiCall_ID3D12Device8_CreatePlacedResource1                                                   = MakeApiCallId(ApiFamily_D3D12, 0x10c1),
    ApiCall_ID3D12Device8_CreateSamplerFeedbackUnorderedAccessView                                = MakeApiCallId(ApiFamily_D3D12, 0x10c2),
    ApiCall_ID3D12Device8_GetCopyableFootprints1                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10c3),
    ApiCall_ID3D12Resource1_GetProtectedResourceSession                                           = MakeApiCallId(ApiFamily_D3D12, 0x10c4),
    ApiCall_ID3D12Resource2_GetDesc1                                                              = MakeApiCallId(ApiFamily_D3D12, 0x10c5),
    ApiCall_ID3D12Heap1_GetProtectedResourceSession                                               = MakeApiCallId(ApiFamily_D3D12, 0x10c6),
    ApiCall_ID3D12GraphicsCommandList3_SetProtectedResourceSession                                = MakeApiCallId(ApiFamily_D3D12, 0x10c7),
    ApiCall_ID3D12MetaCommand_GetRequiredParameterResourceSize                                    = MakeApiCallId(ApiFamily_D3D12, 0x10c8),
    ApiCall_ID3D12GraphicsCommandList4_BeginRenderPass                                            = MakeApiCallId(ApiFamily_D3D12, 0x10c9),
    ApiCall_ID3D12GraphicsCommandList4_EndRenderPass                                              = MakeApiCallId(ApiFamily_D3D12, 0x10ca),
    ApiCall_ID3D12GraphicsCommandList4_InitializeMetaCommand                                      = MakeApiCallId(ApiFamily_D3D12, 0x10cb),
    ApiCall_ID3D12GraphicsCommandList4_ExecuteMetaCommand                                         = MakeApiCallId(ApiFamily_D3D12, 0x10cc),
    ApiCall_ID3D12GraphicsCommandList4_BuildRaytracingAccelerationStructure                       = MakeApiCallId(ApiFamily_D3D12, 0x10cd),
    ApiCall_ID3D12GraphicsCommandList4_EmitRaytracingAccelerationStructurePostbuildInfo           = MakeApiCallId(ApiFamily_D3D12, 0x10ce),
    ApiCall_ID3D12GraphicsCommandList4_CopyRaytracingAccelerationStructure                        = MakeApiCallId(ApiFamily_D3D12, 0x10cf),
    ApiCall_ID3D12GraphicsCommandList4_SetPipelineState1                                          = MakeApiCallId(ApiFamily_D3D12, 0x10d0),
    ApiCall_ID3D12GraphicsCommandList4_DispatchRays                                               = MakeApiCallId(ApiFamily_D3D12, 0x10d1),
    ApiCall_ID3D12Tools_EnableShaderInstrumentation                                               = MakeApiCallId(ApiFamily_D3D12, 0x10d2),
    ApiCall_ID3D12Tools_ShaderInstrumentationEnabled                                              = MakeApiCallId(ApiFamily_D3D12, 0x10d3),
    ApiCall_ID3D12GraphicsCommandList5_RSSetShadingRate                                           = MakeApiCallId(ApiFamily_D3D12, 0x10d4),
    ApiCall_ID3D12GraphicsCommandList5_RSSetShadingRateImage                                      = MakeApiCallId(ApiFamily_D3D12, 0x10d5),
    ApiCall_ID3D12GraphicsCommandList6_DispatchMesh                                               = MakeApiCallId(ApiFamily_D3D12, 0x10d6),

    // d3dcommon.h
    ApiCall_ID3D10Blob_GetBufferPointer                                                           = MakeApiCallId(ApiFamily_D3D12, 0x10d7),
    ApiCall_ID3D10Blob_GetBufferSize                                                              = MakeApiCallId(ApiFamily_D3D12, 0x10d8),
    ApiCall_ID3DDestructionNotifier_RegisterDestructionCallback                                   = MakeApiCallId(ApiFamily_D3D12, 0x10d9),
    ApiCall_ID3DDestructionNotifier_UnregisterDestructionCallback                                 = MakeApiCallId(ApiFamily_D3D12, 0x10da),

    // d3d12sdklayers.h
    ApiCall_ID3D12Debug_EnableDebugLayer                                                          = MakeApiCallId(ApiFamily_D3D12, 0x10db),
    ApiCall_ID3D12Debug1_EnableDebugLayer                                                         = MakeApiCallId(ApiFamily_D3D12, 0x10dc),
    ApiCall_ID3D12Debug1_SetEnableGPUBasedValidation                                              = MakeApiCallId(ApiFamily_D3D12, 0x10dd),
    ApiCall_ID3D12Debug1_SetEnableSynchronizedCommandQueueValidation                              = MakeApiCallId(ApiFamily_D3D12, 0x10de),
    ApiCall_ID3D12Debug2_SetGPUBasedValidationFlags                                               = MakeApiCallId(ApiFamily_D3D12, 0x10df),
    ApiCall_ID3D12Debug3_SetEnableGPUBasedValidation                                              = MakeApiCallId(ApiFamily_D3D12, 0x10e0),
    ApiCall_ID3D12Debug3_SetEnableSynchronizedCommandQueueValidation                              = MakeApiCallId(ApiFamily_D3D12, 0x10e1),
    ApiCall_ID3D12Debug3_SetGPUBasedValidationFlags                                               = MakeApiCallId(ApiFamily_D3D12, 0x10e2),
    ApiCall_ID3D12DebugDevice1_SetDebugParameter                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10e3),
    ApiCall_ID3D12DebugDevice1_GetDebugParameter                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10e4),
    ApiCall_ID3D12DebugDevice1_ReportLiveDeviceObjects                                            = MakeApiCallId(ApiFamily_D3D12, 0x10e5),
    ApiCall_ID3D12DebugDevice_SetFeatureMask                                                      = MakeApiCallId(ApiFamily_D3D12, 0x10e6),
    ApiCall_ID3D12DebugDevice_GetFeatureMask                                                      = MakeApiCallId(ApiFamily_D3D12, 0x10e7),
    ApiCall_ID3D12DebugDevice_ReportLiveDeviceObjects                                             = MakeApiCallId(ApiFamily_D3D12, 0x10e8),
    ApiCall_ID3D12DebugDevice2_SetDebugParameter                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10e9),
    ApiCall_ID3D12DebugDevice2_GetDebugParameter                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10ea),
    ApiCall_ID3D12DebugCommandQueue_AssertResourceState                                           = MakeApiCallId(ApiFamily_D3D12, 0x10eb),
    ApiCall_ID3D12DebugCommandList1_AssertResourceState                                           = MakeApiCallId(ApiFamily_D3D12, 0x10ec),
    ApiCall_ID3D12DebugCommandList1_SetDebugParameter                                             = MakeApiCallId(ApiFamily_D3D12, 0x10ed),
    ApiCall_ID3D12DebugCommandList1_GetDebugParameter                                             = MakeApiCallId(ApiFamily_D3D12, 0x10ee),
    ApiCall_ID3D12DebugCommandList_AssertResourceState                                            = MakeApiCallId(ApiFamily_D3D12, 0x10ef),
    ApiCall_ID3D12DebugCommandList_SetFeatureMask                                                 = MakeApiCallId(ApiFamily_D3D12, 0x10f0),
    ApiCall_ID3D12DebugCommandList_GetFeatureMask                                                 = MakeApiCallId(ApiFamily_D3D12, 0x10f1),
    ApiCall_ID3D12DebugCommandList2_SetDebugParameter                                             = MakeApiCallId(ApiFamily_D3D12, 0x10f2),
    ApiCall_ID3D12DebugCommandList2_GetDebugParameter                                             = MakeApiCallId(ApiFamily_D3D12, 0x10f3),
    ApiCall_ID3D12SharingContract_Present                                                         = MakeApiCallId(ApiFamily_D3D12, 0x10f4),
    ApiCall_ID3D12SharingContract_SharedFenceSignal                                               = MakeApiCallId(ApiFamily_D3D12, 0x10f5),
    ApiCall_ID3D12SharingContract_BeginCapturableWork                                             = MakeApiCallId(ApiFamily_D3D12, 0x10f6),
    ApiCall_ID3D12SharingContract_EndCapturableWork                                               = MakeApiCallId(ApiFamily_D3D12, 0x10f7),
    ApiCall_ID3D12InfoQueue_SetMessageCountLimit                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10f8),
    ApiCall_ID3D12InfoQueue_ClearStoredMessages                                                   = MakeApiCallId(ApiFamily_D3D12, 0x10f9),
    ApiCall_ID3D12InfoQueue_GetMessage                                                            = MakeApiCallId(ApiFamily_D3D12, 0x10fa),
    ApiCall_ID3D12InfoQueue_GetNumMessagesAllowedByStorageFilter                                  = MakeApiCallId(ApiFamily_D3D12, 0x10fb),
    ApiCall_ID3D12InfoQueue_GetNumMessagesDeniedByStorageFilter                                   = MakeApiCallId(ApiFamily_D3D12, 0x10fc),
    ApiCall_ID3D12InfoQueue_GetNumStoredMessages                                                  = MakeApiCallId(ApiFamily_D3D12, 0x10fd),
    ApiCall_ID3D12InfoQueue_GetNumStoredMessagesAllowedByRetrievalFilter                          = MakeApiCallId(ApiFamily_D3D12, 0x10fe),
    ApiCall_ID3D12InfoQueue_GetNumMessagesDiscardedByMessageCountLimit                            = MakeApiCallId(ApiFamily_D3D12, 0x10ff),
    ApiCall_ID3D12InfoQueue_GetMessageCountLimit                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1100),
    ApiCall_ID3D12InfoQueue_AddStorageFilterEntries                                               = MakeApiCallId(ApiFamily_D3D12, 0x1101),
    ApiCall_ID3D12InfoQueue_GetStorageFilter                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1102),
    ApiCall_ID3D12InfoQueue_ClearStorageFilter                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1103),
    ApiCall_ID3D12InfoQueue_PushEmptyStorageFilter                                                = MakeApiCallId(ApiFamily_D3D12, 0x1104),
    ApiCall_ID3D12InfoQueue_PushCopyOfStorageFilter                                               = MakeApiCallId(ApiFamily_D3D12, 0x1105),
    ApiCall_ID3D12InfoQueue_PushStorageFilter                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1106),
    ApiCall_ID3D12InfoQueue_PopStorageFilter                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1107),
    ApiCall_ID3D12InfoQueue_GetStorageFilterStackSize                                             = MakeApiCallId(ApiFamily_D3D12, 0x1108),
    ApiCall_ID3D12InfoQueue_AddRetrievalFilterEntries                                             = MakeApiCallId(ApiFamily_D3D12, 0x1109),
    ApiCall_ID3D12InfoQueue_GetRetrievalFilter                                                    = MakeApiCallId(ApiFamily_D3D12, 0x110a),
    ApiCall_ID3D12InfoQueue_ClearRetrievalFilter                                                  = MakeApiCallId(ApiFamily_D3D12, 0x110b),
    ApiCall_ID3D12InfoQueue_PushEmptyRetrievalFilter                                              = MakeApiCallId(ApiFamily_D3D12, 0x110c),
    ApiCall_ID3D12InfoQueue_PushCopyOfRetrievalFilter                                             = MakeApiCallId(ApiFamily_D3D12, 0x110d),
    ApiCall_ID3D12InfoQueue_PushRetrievalFilter                                                   = MakeApiCallId(ApiFamily_D3D12, 0x110e),
    ApiCall_ID3D12InfoQueue_PopRetrievalFilter                                                    = MakeApiCallId(ApiFamily_D3D12, 0x110f),
    ApiCall_ID3D12InfoQueue_GetRetrievalFilterStackSize                                           = MakeApiCallId(ApiFamily_D3D12, 0x1110),
    ApiCall_ID3D12InfoQueue_AddMessage                                                            = MakeApiCallId(ApiFamily_D3D12, 0x1111),
    ApiCall_ID3D12InfoQueue_AddApplicationMessage                                                 = MakeApiCallId(ApiFamily_D3D12, 0x1112),
    ApiCall_ID3D12InfoQueue_SetBreakOnCategory                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1113),
    ApiCall_ID3D12InfoQueue_SetBreakOnSeverity                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1114),
    ApiCall_ID3D12InfoQueue_SetBreakOnID                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1115),
    ApiCall_ID3D12InfoQueue_GetBreakOnCategory                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1116),
    ApiCall_ID3D12InfoQueue_GetBreakOnSeverity                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1117),
    ApiCall_ID3D12InfoQueue_GetBreakOnID                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1118),
    ApiCall_ID3D12InfoQueue_SetMuteDebugOutput                                                    = MakeApiCallId(ApiFamily_D3D12, 0x1119),
    ApiCall_ID3D12InfoQueue_GetMuteDebugOutput                                                    = MakeApiCallId(ApiFamily_D3D12, 0x111a),

    //win10 sdk 10.0.20348.0
    ApiCall_D3D12GetInterface                                                                     = MakeApiCallId(ApiFamily_D3D12, 0x111b),
    ApiCall_ID3D12DeviceRemovedExtendedData2_GetPageFaultAllocationOutput2                        = MakeApiCallId(ApiFamily_D3D12, 0x111c),
    ApiCall_ID3D12DeviceRemovedExtendedData2_GetDeviceState                                       = MakeApiCallId(ApiFamily_D3D12, 0x111d),
    ApiCall_ID3D12ShaderCacheSession_FindValue                                                    = MakeApiCallId(ApiFamily_D3D12, 0x111e),
    ApiCall_ID3D12ShaderCacheSession_StoreValue                                                   = MakeApiCallId(ApiFamily_D3D12, 0x111f),
    ApiCall_ID3D12ShaderCacheSession_SetDeleteOnDestroy                                           = MakeApiCallId(ApiFamily_D3D12, 0x1120),
    ApiCall_ID3D12ShaderCacheSession_GetDesc                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1121),
    ApiCall_ID3D12Device9_CreateShaderCacheSession                                                = MakeApiCallId(ApiFamily_D3D12, 0x1122),
    ApiCall_ID3D12Device9_ShaderCacheControl                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1123),
    ApiCall_ID3D12Device9_CreateCommandQueue1                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1124),
    ApiCall_ID3D12SDKConfiguration_SetSDKVersion                                                  = MakeApiCallId(ApiFamily_D3D12, 0x1125),
    ApiCall_ID3D12Debug4_DisableDebugLayer                                                        = MakeApiCallId(ApiFamily_D3D12, 0x1126),
    ApiCall_ID3D12Debug5_SetEnableAutoName                                                        = MakeApiCallId(ApiFamily_D3D12, 0x1127),
    ApiCall_ID3D12InfoQueue1_UnregisterMessageCallback                                            = MakeApiCallId(ApiFamily_D3D12, 0x1128),
    ApiCall_ID3D12InfoQueue1_RegisterMessageCallback                                              = MakeApiCallId(ApiFamily_D3D12, 0x1129),

    // Agility SDK 1.606.4
    ApiCall_ID3D12Device10_CreateCommittedResource3                                               = MakeApiCallId(ApiFamily_D3D12, 0x112a),
    ApiCall_ID3D12Device10_CreatePlacedResource2                                                  = MakeApiCallId(ApiFamily_D3D12, 0x112b),
    ApiCall_ID3D12Device10_CreateReservedResource2                                                = MakeApiCallId(ApiFamily_D3D12, 0x112c),
    ApiCall_ID3D12Device11_CreateSampler2                                                         = MakeApiCallId(ApiFamily_D3D12, 0x112d),
    ApiCall_ID3D12VirtualizationGuestDevice_ShareWithHost                                         = MakeApiCallId(ApiFamily_D3D12, 0x112e),
    ApiCall_ID3D12VirtualizationGuestDevice_CreateFenceFd                                         = MakeApiCallId(ApiFamily_D3D12, 0x112f),
    ApiCall_ID3D12SDKConfiguration1_CreateDeviceFactory                                           = MakeApiCallId(ApiFamily_D3D12, 0x1130),
    ApiCall_ID3D12SDKConfiguration1_FreeUnusedSDKs                                                = MakeApiCallId(ApiFamily_D3D12, 0x1131),
    ApiCall_ID3D12DeviceFactory_InitializeFromGlobalState                                         = MakeApiCallId(ApiFamily_D3D12, 0x1132),
    ApiCall_ID3D12DeviceFactory_ApplyToGlobalState                                                = MakeApiCallId(ApiFamily_D3D12, 0x1133),
    ApiCall_ID3D12DeviceFactory_SetFlags                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1134),
    ApiCall_ID3D12DeviceFactory_GetFlags                                                          = MakeApiCallId(ApiFamily_D3D12, 0x1135),
    ApiCall_ID3D12DeviceFactory_GetConfigurationInterface                                         = MakeApiCallId(ApiFamily_D3D12, 0x1136),
    ApiCall_ID3D12DeviceFactory_EnableExperimentalFeatures                                        = MakeApiCallId(ApiFamily_D3D12, 0x1137),
    ApiCall_ID3D12DeviceFactory_CreateDevice                                                      = MakeApiCallId(ApiFamily_D3D12, 0x1138),
    ApiCall_ID3D12DeviceConfiguration_GetDesc                                                     = MakeApiCallId(ApiFamily_D3D12, 0x1139),
    ApiCall_ID3D12DeviceConfiguration_GetEnabledExperimentalFeatures                              = MakeApiCallId(ApiFamily_D3D12, 0x113a),
    ApiCall_ID3D12DeviceConfiguration_SerializeVersionedRootSignature                             = MakeApiCallId(ApiFamily_D3D12, 0x113b),
    ApiCall_ID3D12DeviceConfiguration_CreateVersionedRootSignatureDeserializer                    = MakeApiCallId(ApiFamily_D3D12, 0x113c),
    ApiCall_ID3D12GraphicsCommandList7_Barrier                                                    = MakeApiCallId(ApiFamily_D3D12, 0x113d),
    ApiCall_ID3D12GraphicsCommandList8_OMSetFrontAndBackStencilRef                                = MakeApiCallId(ApiFamily_D3D12, 0x113e),
    ApiCall_ID3D12DeviceRemovedExtendedDataSettings2_UseMarkersOnlyAutoBreadcrumbs                = MakeApiCallId(ApiFamily_D3D12, 0x113f),
    ApiCall_ID3D12Debug6_SetForceLegacyBarrierValidation                                          = MakeApiCallId(ApiFamily_D3D12, 0x1140),
    ApiCall_ID3D12DebugCommandQueue1_AssertResourceAccess                                         = MakeApiCallId(ApiFamily_D3D12, 0x1141),
    ApiCall_ID3D12DebugCommandQueue1_AssertTextureLayout                                          = MakeApiCallId(ApiFamily_D3D12, 0x1142),
    ApiCall_ID3D12DebugCommandList3_AssertResourceAccess                                          = MakeApiCallId(ApiFamily_D3D12, 0x1143),
    ApiCall_ID3D12DebugCommandList3_AssertTextureLayout                                           = MakeApiCallId(ApiFamily_D3D12, 0x1144),

    // Agility SDK 1.610.5
    ApiCall_ID3D12Device12_GetResourceAllocationInfo3                                             = MakeApiCallId(ApiFamily_D3D12, 0x1145),
    ApiCall_ID3D12GraphicsCommandList9_RSSetDepthBias                                             = MakeApiCallId(ApiFamily_D3D12, 0x1146),
    ApiCall_ID3D12GraphicsCommandList9_IASetIndexBufferStripCutValue                              = MakeApiCallId(ApiFamily_D3D12, 0x1147),
    ApiCall_ID3D12DSRDeviceFactory_CreateDSRDevice                                                = MakeApiCallId(ApiFamily_D3D12, 0x1148),
    ApiCall_ID3D12ManualWriteTrackingResource_TrackWrite                                          = MakeApiCallId(ApiFamily_D3D12, 0x1149),

    // AGS API
    // amd_ags.h
    ApiCall_Ags_agsInitialize_6_0_1                                                               = MakeApiCallId(ApiFamily_AGS, 0x1000),
    ApiCall_Ags_agsDeInitialize_6_0_1                                                             = MakeApiCallId(ApiFamily_AGS, 0x1001),
    ApiCall_Ags_agsDriverExtensionsDX12_CreateDevice_6_0_1                                        = MakeApiCallId(ApiFamily_AGS, 0x1002),
    ApiCall_Ags_agsDriverExtensionsDX12_DestroyDevice_6_0_1                                       = MakeApiCallId(ApiFamily_AGS, 0x1003),
    ApiCall_Ags_agsCheckDriverVersion_6_0_1                                                       = MakeApiCallId(ApiFamily_AGS, 0x1004),
    ApiCall_Ags_agsGetVersionNumber_6_0_1                                                         = MakeApiCallId(ApiFamily_AGS, 0x1005),
    ApiCall_Ags_agsSetDisplayMode_6_0_1                                                           = MakeApiCallId(ApiFamily_AGS, 0x1006),
    ApiCall_Ags_agsDriverExtensionsDX12_PushMarker_6_0_1                                          = MakeApiCallId(ApiFamily_AGS, 0x1007),
    ApiCall_Ags_agsDriverExtensionsDX12_PopMarker_6_0_1                                           = MakeApiCallId(ApiFamily_AGS, 0x1008),
    ApiCall_Ags_agsDriverExtensionsDX12_SetMarker_6_0_1                                           = MakeApiCallId(ApiFamily_AGS, 0x1009),

    ApiCall_D3D12Last,

    // clang-format on
};

GFXRECON_END_NAMESPACE(gfxrecon)
GFXRECON_END_NAMESPACE(format)

#endif // GFXRECON_FORMAT_API_CALL_ID_H
