# -*- coding: utf-8 -*-
"""This module provides a simple way to compute the interpolation of a
:py:class:`Function <dolfin.functions.function.Function>` or
:py:class:`Expression <dolfin.functions.expression.Expression>` onto a
finite element space."""

# Copyright (C) 2009-2017 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Jørgen S. Dokken, 2017
# First added:  2009-08-29
# Last changed: 2017-03-17

# Local imports
import dolfin.cpp as cpp

from dolfin.functions.functionspace import FunctionSpace, MultiMeshFunctionSpace
from dolfin.functions.function import Function
from dolfin.functions.multimeshfunction import MultiMeshFunction

__all__ = ["interpolate"]


def interpolate(v, V):
    """Return interpolation of a given function into a given finite
    element space. Also supports MultiMesh.

    *Arguments*
        v
            a :py:class:`Function <dolfin.functions.function.Function>` or
            an :py:class:`Expression <dolfin.functions.expression.Expression>`
            or an :py:class`MultiMeshFunction <dolfin.functions.multimeshfunction.MultiMeshFunction>`
        V
            a :py:class:`FunctionSpace (standard, mixed, etc.)
            <dolfin.functions.functionspace.FunctionSpace>` or
            a :py:class:`MultiMeshFunctionSpace
            <dolfin.cpp.function.MultiMeshFunctionSpace>`.


    *Example of usage*

        .. code-block:: python

            v = Expression("sin(pi*x[0])")
            V = FunctionSpace(mesh, "Lagrange", 1)
            Iv = interpolate(v, V)

    """

    # Check arguments
    if not isinstance(V, (FunctionSpace, cpp.MultiMeshFunctionSpace)):
        cpp.dolfin_error("interpolation.py",
                         "compute interpolation",
                         "Illegal function space for interpolation, not a FunctionSpace (%s)" % str(v))
    if isinstance(V, cpp.MultiMeshFunctionSpace):
        Pv = MultiMeshFunction(V)
        Pv.interpolate(v)
        return Pv

    # Compute interpolation
    Pv = Function(V)
    Pv.interpolate(v)

    return Pv
