//! Abstract definitions for data parsing backends.

/**
 * Copyright (c) 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
use std::collections::HashMap;
use std::error;

pub mod ini;

/// The variables read from the input data and the first section name.
///
/// The [`Backend::read_file`] method returns two values: a mapping of
/// section names to variable => value mappings, and the name of the first
/// section encountered in the input data. The latter is particularly
/// useful for the "ini" backend where `confget` will behave differently
/// depending on the [section_override][`crate::Config::section_override`]
/// setting.
pub type DataRead = (HashMap<String, HashMap<String, String>>, String);

/// A backend that implements parsing a specific type of configuration data.
///
/// `Backend` objects should never be constructed manually; rather,
/// the [get_backend][`crate::get_backend`] or
/// [read_ini_file][`crate::read_ini_file`] functions should be used.
pub trait Backend {
    /// Obtain and parse the input data in a backend-specific way.
    fn read_file(&self) -> Result<DataRead, Box<dyn error::Error>>;
}
