// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_VIEWS_TRANSLATE_TRANSLATE_BUBBLE_VIEW_H_
#define CHROME_BROWSER_UI_VIEWS_TRANSLATE_TRANSLATE_BUBBLE_VIEW_H_

#include <map>
#include <string>

#include "base/basictypes.h"
#include "chrome/browser/translate/translate_tab_helper.h"
#include "chrome/browser/ui/translate/language_combobox_model.h"
#include "chrome/browser/ui/translate/translate_bubble_model.h"
#include "components/translate/core/common/translate_errors.h"
#include "content/public/browser/web_contents_observer.h"
#include "ui/views/bubble/bubble_delegate.h"
#include "ui/views/controls/button/button.h"
#include "ui/views/controls/combobox/combobox_listener.h"
#include "ui/views/controls/link_listener.h"

class PrefService;

namespace views {
class Checkbox;
class GridLayout;
class LabelButton;
class Link;
class View;
}

class TranslateBubbleView : public views::BubbleDelegateView,
                            public views::ButtonListener,
                            public views::ComboboxListener,
                            public views::LinkListener,
                            public content::WebContentsObserver {
 public:
  virtual ~TranslateBubbleView();

  // Shows the Translate bubble.
  static void ShowBubble(views::View* anchor_view,
                         content::WebContents* web_contents,
                         TranslateTabHelper::TranslateStep step,
                         TranslateErrors::Type error_type);

  // If true, the Translate bubble is being shown.
  static bool IsShowing();

  // Returns the bubble view currently shown. This may return NULL.
  static TranslateBubbleView* GetCurrentBubble();

  TranslateBubbleModel* model() { return model_.get(); }

  // views::BubbleDelegateView methods.
  virtual void Init() OVERRIDE;
  virtual void ButtonPressed(views::Button* sender,
                             const ui::Event& event) OVERRIDE;

  // views::WidgetDelegate method.
  virtual void WindowClosing() OVERRIDE;

  // views::View methods.
  virtual bool AcceleratorPressed(const ui::Accelerator& accelerator) OVERRIDE;
  virtual gfx::Size GetPreferredSize() OVERRIDE;

  // views::CombboxListener methods.
  virtual void OnPerformAction(views::Combobox* combobox) OVERRIDE;

  // views::LinkListener method.
  virtual void LinkClicked(views::Link* source, int event_flags) OVERRIDE;

  // content::WebContentsObserver method.
  virtual void WebContentsDestroyed(content::WebContents* web_contents)
      OVERRIDE;

  // Returns the current view state.
  TranslateBubbleModel::ViewState GetViewState() const;

 private:
  enum LinkID {
    LINK_ID_ADVANCED,
    LINK_ID_LANGUAGE_SETTINGS,
  };

  enum ButtonID {
    BUTTON_ID_TRANSLATE,
    BUTTON_ID_DONE,
    BUTTON_ID_CANCEL,
    BUTTON_ID_SHOW_ORIGINAL,
    BUTTON_ID_TRY_AGAIN,
    BUTTON_ID_ALWAYS_TRANSLATE,
  };

  enum ComboboxID {
    COMBOBOX_ID_DENIAL,
    COMBOBOX_ID_SOURCE_LANGUAGE,
    COMBOBOX_ID_TARGET_LANGUAGE,
  };

  friend class TranslateBubbleViewTest;
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest, TranslateButton);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest, AdvancedLink);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest, ShowOriginalButton);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest, TryAgainButton);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest,
                           AlwaysTranslateCheckboxAndCancelButton);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest,
                           AlwaysTranslateCheckboxAndDoneButton);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest, DoneButton);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest,
                           DoneButtonWithoutTranslating);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest,
                           CancelButtonReturningBeforeTranslate);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest,
                           CancelButtonReturningAfterTranslate);
  FRIEND_TEST_ALL_PREFIXES(TranslateBubbleViewTest, CancelButtonReturningError);

  TranslateBubbleView(views::View* anchor_view,
                      scoped_ptr<TranslateBubbleModel> model,
                      TranslateErrors::Type error_type,
                      content::WebContents* web_contents);

  // Returns the current child view.
  views::View* GetCurrentView();

  // Handles the event when the user presses a button.
  void HandleButtonPressed(ButtonID sender_id);

  // Handles the event when the user clicks a link.
  void HandleLinkClicked(LinkID sender_id);

  // Handles the event when the user changes an index of a combobox.
  void HandleComboboxPerformAction(ComboboxID sender_id);

  // Updates the visibilities of child views according to the current view type.
  void UpdateChildVisibilities();

  // Creates the 'before translate' view. Caller takes ownership of the returned
  // view.
  views::View* CreateViewBeforeTranslate();

  // Creates the 'translating' view. Caller takes ownership of the returned
  // view.
  views::View* CreateViewTranslating();

  // Creates the 'after translate' view. Caller takes ownership of the returned
  // view.
  views::View* CreateViewAfterTranslate();

  // Creates the 'error' view. Caller takes ownership of the returned view.
  views::View* CreateViewError();

  // Creates the 'advanced' view. Caller takes ownership of the returned view.
  views::View* CreateViewAdvanced();

  // Switches the view type.
  void SwitchView(TranslateBubbleModel::ViewState view_state);

  // Switches to the error view.
  void SwitchToErrorView(TranslateErrors::Type error_type);

  // Updates the advanced view.
  void UpdateAdvancedView();

  static TranslateBubbleView* translate_bubble_view_;

  views::View* before_translate_view_;
  views::View* translating_view_;
  views::View* after_translate_view_;
  views::View* error_view_;
  views::View* advanced_view_;

  scoped_ptr<LanguageComboboxModel> source_language_combobox_model_;
  scoped_ptr<LanguageComboboxModel> target_language_combobox_model_;

  views::Combobox* denial_combobox_;
  views::Combobox* source_language_combobox_;
  views::Combobox* target_language_combobox_;

  views::Checkbox* always_translate_checkbox_;

  views::LabelButton* advanced_cancel_button_;
  views::LabelButton* advanced_done_button_;

  scoped_ptr<TranslateBubbleModel> model_;

  TranslateErrors::Type error_type_;

  // Whether the window is an incognito window.
  const bool is_in_incognito_window_;

  // Whether the translation is acutually executed.
  bool translate_executed_;

  // Whether one of denial buttons is clicked.
  bool denial_button_clicked_;

  DISALLOW_COPY_AND_ASSIGN(TranslateBubbleView);
};

#endif  // CHROME_BROWSER_UI_VIEWS_TRANSLATE_TRANSLATE_BUBBLE_VIEW_H_
