/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jena.graph;

/**
    GraphEvents is the base class for Jena general graph events. Each graph event
    has a title and some content.

 */
public class GraphEvents
	{
	public static final GraphEvents removeAll = new GraphEvents( "removeAll", "" ){
    	@Override
		public Triple getTriple() { return Triple.ANY; }
      };
	
	public static final GraphEvents startRead = new GraphEvents( "startRead", "" );
	
	public static final GraphEvents finishRead = new GraphEvents( "finishRead", "" );
	
    final private String title;
    final protected Object content;
    
    public GraphEvents( String title, Object content )
        { this.title = title;
        this.content = content; }
    
    @Override
    public boolean equals( Object o )
        { return o instanceof GraphEvents && same( (GraphEvents) o ); }
    
    public boolean same( GraphEvents o )
        { return title.equals( o.title ) && content.equals( o.content ); }
   
	public static GraphEvents remove( Node s, Node p, Node o )
	    { return new GraphEvents( "remove", Triple.createMatch( s, p, o ) ){
	    	@Override
			public Triple getTriple() { return (Triple)content; }
	      };
	    }
	
	@Override
    public String toString()
	    { return "<GE " + title + ">"; }

	/**
     * A fluid triple relevant to the event, or null if not applicable.
     * For events generated by the Jena code base the title is always a String,
     * and is one of:
     * <dl>
     * <dt><q>startRead</q></dt>
     * <dt><q>finishRead</q></dt><dd><code>null</code></dd>
     * <dt><q>removeAll</q></dt><dd>{@link org.apache.jena.graph.Triple#ANY}</dd>
     * <dt><q>remove</q></dt><dd>The fluid triple being removed.</dd>
     * </dl>
     * @return An interesting triple.
     */
    public Triple getTriple()
        { return null; }
	
	
	/**
     * For events generated by the Jena code base the content is one of:
     * <dl>
     * <dt><q>startRead</q></dt>
     * <dt><q>finishRead</q></dt>
     * <dt><q>removeAll</q></dt><dd>The empty string.</dd>
     * <dt><q>remove</q></dt><dd>The (non-concrete) triple being removed.</dd>
     * </dl>
     */
    public Object getContent()
        { return content; }
    

    /**
     * For events generated by the Jena code base the title is always a String,
     * and is one of:
     * <dl>
     * <dt><q>startRead</q></dt><dd>{@link #startRead}</dd>
     * <dt><q>finishRead</q></dt><dd>{@link #finishRead}</dd>
     * <dt><q>removeAll</q></dt><dd>{@link #removeAll}</dd>
     * <dt><q>remove</q></dt><dd>{@link #remove(Node, Node, Node)}</dd>
     * </dl>
     */
    public Object getTitle()
        { return title; }
	}
