/*
 * Copyright (C) 2018 Tobias Brunner
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <test_suite.h>

#include "../eap_aka_3gpp_functions.h"

static eap_aka_3gpp_functions_t *functions;

START_SETUP(functions_setup)
{
	functions = eap_aka_3gpp_functions_create();
	ck_assert(functions);
}
END_SETUP

START_TEARDOWN(functions_teardown)
{
	functions->destroy(functions);
}
END_TEARDOWN

/**
 * Test vectors from 3GPP TS 35.207
 */
static struct {
	uint8_t k[AKA_K_LEN];
	uint8_t rand[AKA_RAND_LEN];
	uint8_t sqn[AKA_SQN_LEN];
	uint8_t amf[AKA_AMF_LEN];
	uint8_t opc[AKA_OPC_LEN];
	uint8_t f1[AKA_MAC_LEN];
	uint8_t f1star[AKA_MAC_LEN];
	uint8_t f2[AKA_RES_LEN];
	uint8_t f3[AKA_CK_LEN];
	uint8_t f4[AKA_IK_LEN];
	uint8_t f5[AKA_AK_LEN];
	uint8_t f5star[AKA_AK_LEN];
} test_data[] = {
	{
		.k = {0x46,0x5b,0x5c,0xe8,0xb1,0x99,0xb4,0x9f,0xaa,0x5f,0x0a,0x2e,0xe2,0x38,0xa6,0xbc},
		.rand = {0x23,0x55,0x3c,0xbe,0x96,0x37,0xa8,0x9d,0x21,0x8a,0xe6,0x4d,0xae,0x47,0xbf,0x35},
		.sqn = {0xff,0x9b,0xb4,0xd0,0xb6,0x07},
		.amf = {0xb9,0xb9},
		.opc = {0xcd,0x63,0xcb,0x71,0x95,0x4a,0x9f,0x4e,0x48,0xa5,0x99,0x4e,0x37,0xa0,0x2b,0xaf},
		.f1 = {0x4a,0x9f,0xfa,0xc3,0x54,0xdf,0xaf,0xb3},
		.f1star = {0x01,0xcf,0xaf,0x9e,0xc4,0xe8,0x71,0xe9},
		.f2 = {0xa5,0x42,0x11,0xd5,0xe3,0xba,0x50,0xbf},
		.f3 = {0xb4,0x0b,0xa9,0xa3,0xc5,0x8b,0x2a,0x05,0xbb,0xf0,0xd9,0x87,0xb2,0x1b,0xf8,0xcb},
		.f4 = {0xf7,0x69,0xbc,0xd7,0x51,0x04,0x46,0x04,0x12,0x76,0x72,0x71,0x1c,0x6d,0x34,0x41},
		.f5 = {0xaa,0x68,0x9c,0x64,0x83,0x70},
		.f5star = {0x45,0x1e,0x8b,0xec,0xa4,0x3b},
	},
	{
		.k = {0x03,0x96,0xeb,0x31,0x7b,0x6d,0x1c,0x36,0xf1,0x9c,0x1c,0x84,0xcd,0x6f,0xfd,0x16},
		.rand = {0xc0,0x0d,0x60,0x31,0x03,0xdc,0xee,0x52,0xc4,0x47,0x81,0x19,0x49,0x42,0x02,0xe8},
		.sqn = {0xfd,0x8e,0xef,0x40,0xdf,0x7d},
		.amf = {0xaf,0x17},
		.opc = {0x53,0xc1,0x56,0x71,0xc6,0x0a,0x4b,0x73,0x1c,0x55,0xb4,0xa4,0x41,0xc0,0xbd,0xe2},
		.f1 = {0x5d,0xf5,0xb3,0x18,0x07,0xe2,0x58,0xb0},
		.f1star = {0xa8,0xc0,0x16,0xe5,0x1e,0xf4,0xa3,0x43},
		.f2 = {0xd3,0xa6,0x28,0xed,0x98,0x86,0x20,0xf0},
		.f3 = {0x58,0xc4,0x33,0xff,0x7a,0x70,0x82,0xac,0xd4,0x24,0x22,0x0f,0x2b,0x67,0xc5,0x56},
		.f4 = {0x21,0xa8,0xc1,0xf9,0x29,0x70,0x2a,0xdb,0x3e,0x73,0x84,0x88,0xb9,0xf5,0xc5,0xda},
		.f5 = {0xc4,0x77,0x83,0x99,0x5f,0x72},
		.f5star = {0x30,0xf1,0x19,0x70,0x61,0xc1},
	},
	{
		.k = {0xfe,0xc8,0x6b,0xa6,0xeb,0x70,0x7e,0xd0,0x89,0x05,0x75,0x7b,0x1b,0xb4,0x4b,0x8f},
		.rand = {0x9f,0x7c,0x8d,0x02,0x1a,0xcc,0xf4,0xdb,0x21,0x3c,0xcf,0xf0,0xc7,0xf7,0x1a,0x6a},
		.sqn = {0x9d,0x02,0x77,0x59,0x5f,0xfc},
		.amf = {0x72,0x5c},
		.opc = {0x10,0x06,0x02,0x0f,0x0a,0x47,0x8b,0xf6,0xb6,0x99,0xf1,0x5c,0x06,0x2e,0x42,0xb3},
		.f1 = {0x9c,0xab,0xc3,0xe9,0x9b,0xaf,0x72,0x81},
		.f1star = {0x95,0x81,0x4b,0xa2,0xb3,0x04,0x43,0x24},
		.f2 = {0x80,0x11,0xc4,0x8c,0x0c,0x21,0x4e,0xd2},
		.f3 = {0x5d,0xbd,0xbb,0x29,0x54,0xe8,0xf3,0xcd,0xe6,0x65,0xb0,0x46,0x17,0x9a,0x50,0x98},
		.f4 = {0x59,0xa9,0x2d,0x3b,0x47,0x6a,0x04,0x43,0x48,0x70,0x55,0xcf,0x88,0xb2,0x30,0x7b},
		.f5 = {0x33,0x48,0x4d,0xc2,0x13,0x6b},
		.f5star = {0xde,0xac,0xdd,0x84,0x8c,0xc6},
	},
	{
		.k = {0x9e,0x59,0x44,0xae,0xa9,0x4b,0x81,0x16,0x5c,0x82,0xfb,0xf9,0xf3,0x2d,0xb7,0x51},
		.rand = {0xce,0x83,0xdb,0xc5,0x4a,0xc0,0x27,0x4a,0x15,0x7c,0x17,0xf8,0x0d,0x01,0x7b,0xd6},
		.sqn = {0x0b,0x60,0x4a,0x81,0xec,0xa8},
		.amf = {0x9e,0x09},
		.opc = {0xa6,0x4a,0x50,0x7a,0xe1,0xa2,0xa9,0x8b,0xb8,0x8e,0xb4,0x21,0x01,0x35,0xdc,0x87},
		.f1 = {0x74,0xa5,0x82,0x20,0xcb,0xa8,0x4c,0x49},
		.f1star = {0xac,0x2c,0xc7,0x4a,0x96,0x87,0x18,0x37},
		.f2 = {0xf3,0x65,0xcd,0x68,0x3c,0xd9,0x2e,0x96},
		.f3 = {0xe2,0x03,0xed,0xb3,0x97,0x15,0x74,0xf5,0xa9,0x4b,0x0d,0x61,0xb8,0x16,0x34,0x5d},
		.f4 = {0x0c,0x45,0x24,0xad,0xea,0xc0,0x41,0xc4,0xdd,0x83,0x0d,0x20,0x85,0x4f,0xc4,0x6b},
		.f5 = {0xf0,0xb9,0xc0,0x8a,0xd0,0x2e},
		.f5star = {0x60,0x85,0xa8,0x6c,0x6f,0x63},
	},
	{
		.k = {0x4a,0xb1,0xde,0xb0,0x5c,0xa6,0xce,0xb0,0x51,0xfc,0x98,0xe7,0x7d,0x02,0x6a,0x84},
		.rand = {0x74,0xb0,0xcd,0x60,0x31,0xa1,0xc8,0x33,0x9b,0x2b,0x6c,0xe2,0xb8,0xc4,0xa1,0x86},
		.sqn = {0xe8,0x80,0xa1,0xb5,0x80,0xb6},
		.amf = {0x9f,0x07},
		.opc = {0xdc,0xf0,0x7c,0xbd,0x51,0x85,0x52,0x90,0xb9,0x2a,0x07,0xa9,0x89,0x1e,0x52,0x3e},
		.f1 = {0x49,0xe7,0x85,0xdd,0x12,0x62,0x6e,0xf2},
		.f1star = {0x9e,0x85,0x79,0x03,0x36,0xbb,0x3f,0xa2},
		.f2 = {0x58,0x60,0xfc,0x1b,0xce,0x35,0x1e,0x7e},
		.f3 = {0x76,0x57,0x76,0x6b,0x37,0x3d,0x1c,0x21,0x38,0xf3,0x07,0xe3,0xde,0x92,0x42,0xf9},
		.f4 = {0x1c,0x42,0xe9,0x60,0xd8,0x9b,0x8f,0xa9,0x9f,0x27,0x44,0xe0,0x70,0x8c,0xcb,0x53},
		.f5 = {0x31,0xe1,0x1a,0x60,0x91,0x18},
		.f5star = {0xfe,0x25,0x55,0xe5,0x4a,0xa9},
	},
	{
		.k = {0x6c,0x38,0xa1,0x16,0xac,0x28,0x0c,0x45,0x4f,0x59,0x33,0x2e,0xe3,0x5c,0x8c,0x4f},
		.rand = {0xee,0x64,0x66,0xbc,0x96,0x20,0x2c,0x5a,0x55,0x7a,0xbb,0xef,0xf8,0xba,0xbf,0x63},
		.sqn = {0x41,0x4b,0x98,0x22,0x21,0x81},
		.amf = {0x44,0x64},
		.opc = {0x38,0x03,0xef,0x53,0x63,0xb9,0x47,0xc6,0xaa,0xa2,0x25,0xe5,0x8f,0xae,0x39,0x34},
		.f1 = {0x07,0x8a,0xdf,0xb4,0x88,0x24,0x1a,0x57},
		.f1star = {0x80,0x24,0x6b,0x8d,0x01,0x86,0xbc,0xf1},
		.f2 = {0x16,0xc8,0x23,0x3f,0x05,0xa0,0xac,0x28},
		.f3 = {0x3f,0x8c,0x75,0x87,0xfe,0x8e,0x4b,0x23,0x3a,0xf6,0x76,0xae,0xde,0x30,0xba,0x3b},
		.f4 = {0xa7,0x46,0x6c,0xc1,0xe6,0xb2,0xa1,0x33,0x7d,0x49,0xd3,0xb6,0x6e,0x95,0xd7,0xb4},
		.f5 = {0x45,0xb0,0xf6,0x9a,0xb0,0x6c},
		.f5star = {0x1f,0x53,0xcd,0x2b,0x11,0x13},
	},
};

START_TEST(test_f1)
{
	uint8_t mac[AKA_MAC_LEN];

	ck_assert(functions->f1(functions, test_data[_i].k, test_data[_i].opc,
							test_data[_i].rand, test_data[_i].sqn,
							test_data[_i].amf, mac));
	ck_assert(memeq(test_data[_i].f1, mac, sizeof(mac)));
}
END_TEST

START_TEST(test_f1star)
{
	uint8_t mac[AKA_MAC_LEN];

	ck_assert(functions->f1star(functions, test_data[_i].k,
							test_data[_i].opc, test_data[_i].rand,
							test_data[_i].sqn, test_data[_i].amf, mac));
	ck_assert(memeq(test_data[_i].f1star, mac, sizeof(mac)));
}
END_TEST

START_TEST(test_f2345)
{
	uint8_t res[AKA_RES_LEN], ck[AKA_CK_LEN], ik[AKA_IK_LEN], ak[AKA_AK_LEN];

	ck_assert(functions->f2345(functions, test_data[_i].k,
							test_data[_i].opc, test_data[_i].rand,
							res, ck, ik, ak));

	ck_assert(memeq(test_data[_i].f2, res, sizeof(res)));
	ck_assert(memeq(test_data[_i].f3, ck, sizeof(ck)));
	ck_assert(memeq(test_data[_i].f4, ik, sizeof(ik)));
	ck_assert(memeq(test_data[_i].f5, ak, sizeof(ak)));
}
END_TEST

START_TEST(test_f5star)
{
	uint8_t ak[AKA_AK_LEN];

	ck_assert(functions->f5star(functions, test_data[_i].k,
							test_data[_i].opc, test_data[_i].rand, ak));

	ck_assert(memeq(test_data[_i].f5star, ak, sizeof(ak)));
}
END_TEST

Suite *test_vectors_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("eap-aka-3gpp");

	tc = tcase_create("f1, f1*");
	tcase_add_checked_fixture(tc, functions_setup, functions_teardown);
	tcase_add_loop_test(tc, test_f1, 0, countof(test_data));
	tcase_add_loop_test(tc, test_f1star, 0, countof(test_data));
	suite_add_tcase(s, tc);

	tc = tcase_create("f2, f3, f4 and f5");
	tcase_add_checked_fixture(tc, functions_setup, functions_teardown);
	tcase_add_loop_test(tc, test_f2345, 0, countof(test_data));
	suite_add_tcase(s, tc);

	tc = tcase_create("f5*");
	tcase_add_checked_fixture(tc, functions_setup, functions_teardown);
	tcase_add_loop_test(tc, test_f5star, 0, countof(test_data));
	suite_add_tcase(s, tc);

	return s;
}
