/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.parameter;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import edu.jhu.ece.iacl.jist.pipeline.PerformanceSummary;
import edu.jhu.ece.iacl.jist.pipeline.factory.ParamOptionFactory;
import edu.jhu.ece.iacl.jist.utility.JistXMLUtil;

/**
 * Option Parameter to select between different text options.
 * 
 * @author Blake Lucas
 */
public class ParamOption extends ParamModel<String> {
	
	/** The options. */
	protected List<String> options;
	
	/** The index. */
	protected int index;

	public boolean xmlEncodeParam(Document document, Element parent) {
		super.xmlEncodeParam(document, parent);		 
		Element em;				



		em = document.createElement("options");
		boolean val = false;
		for(String str : options) 
		{
			Element em2 = document.createElement("option");
			em2.appendChild(document.createTextNode(str));
			em.appendChild(em2);
			val = true;

		}
		if(val)
			parent.appendChild(em);

		 em = document.createElement("index");		
		 em.appendChild(document.createTextNode(index+""));
		 parent.appendChild(em);
		return true;
	}
	
	 public void xmlDecodeParam(Document document, Element parent) {
			super.xmlDecodeParam(document, parent);
			
			options = new ArrayList<String>();
			Element el = JistXMLUtil.xmlReadElement(parent, "options");
			if(el!=null) {
				for(Element opt : JistXMLUtil.xmlReadElementList(el, "option")) {
					options.add(opt.getFirstChild().getNodeValue());
				}
			}		

			index= Integer.valueOf(JistXMLUtil.xmlReadTag(parent, "index"));			 
			
		
		}

	/**
	 * Construct a list of possible options.
	 * 
	 * @param options
	 *            the options
	 */
	public ParamOption(List<String> options) {
		this.options = options;
		factory = new ParamOptionFactory(this);
		index = 0;
	}

	/**
	 * Construct a list of possible options.
	 * 
	 * @param name
	 *            parameter name
	 * @param options
	 *            the options
	 */
	public ParamOption(String name, List<String> options) {
		this(options);
		setName(name);
	}

	/**
	 * Construct a list of possible options.
	 * 
	 * @param name
	 *            parameter name
	 * @param options
	 *            the options
	 */
	public ParamOption(String name, String[] options) {
		this(options);
		setName(name);
	}
	
	public ParamOption() {this("invalid",(String[])null);};

	/**
	 * Construct a list of possible options.
	 * 
	 * @param options
	 *            the options
	 */
	public ParamOption(String[] options) {
		this.options = new ArrayList<String>();
		if(options!=null)
		for (String option : options) {
			this.options.add(option);
		}
		factory = new ParamOptionFactory(this);
		index = 0;
	}

	/**
	 * Add option to list.
	 * 
	 * @param opt
	 *            option name
	 */
	public void add(String opt) {
		if (options == null) {
			this.options = new ArrayList<String>();
		}
		if (!options.contains(opt)) {
			options.add(opt);
			if (getInputView() != null) {
				getInputView().update();
			}
		}
	}

	/**
	 * Clone object.
	 * 
	 * @return the param option
	 */
	public ParamOption clone() {
		ParamOption param = new ParamOption(new ArrayList<String>(options));
		param.index = index;
		param.setName(this.getName());
		param.label=this.label;
		param.setHidden(this.isHidden());
		param.setMandatory(this.isMandatory());
		param.shortLabel=shortLabel;
		param.cliTag=cliTag;
		return param;
	}

	/**
	 * Compare the options of one parameter to another to determine which list
	 * of options is more restrictive.
	 * 
	 * @param model
	 *            the model
	 * @return the int
	 */
	public int compareTo(ParamModel model) {
		if (model instanceof ParamOption) {
			List<String> modelOptions = ((ParamOption) model).getOptions();
			return (int) Math.signum(modelOptions.size() - options.size());
		} else {
			return 1;
		}
	}

	/**
	 * Get selected option index.
	 * 
	 * @return the index
	 */
	public int getIndex() {
		return index;
	}

	/**
	 * Get list of possible options.
	 * 
	 * @return the options
	 */
	public List<String> getOptions() {
		return options;
	}

	/**
	 * Get selected option name.
	 * 
	 * @return selected option string
	 */
	public String getValue() {
		if (options.size() > 0&&index>=0&&index<options.size()) {
			return new String(options.get(index));
		} else {
			return null;
		}
	}

	/**
	 * Initialize parameter.
	 */
	public void init() {
		this.setMaxIncoming(1);
		connectible = true;
		factory = new ParamOptionFactory(this);
	}

	/**
	 * Set list of options.
	 * 
	 * @param options
	 *            options
	 */
	public void setOptions(List<String> options) {
		index = 0;
		if (options == null) {
			this.options = new ArrayList<String>();
		} else {
			this.options = options;
		}
		if (getInputView() != null) {
			getInputView().update();
		}
	}

	/**
	 * Set the selected option. This method will accept the string
	 * representation of the option or an integer index into the option array.
	 * 
	 * @param value
	 *            parameter value
	 */
	public void setValue(Integer value) {
		index = value;
		if ((index < 0) || ((index >= options.size()) && (options.size() > 0))) {
			throw new InvalidParameterValueException(this, value);
		}
	}
	
	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel#setValue(java.lang.Object)
	 */
	public void setValue(String value) {
		if(value==null)
			index=0;
		else
			index = options.indexOf(value);
		if (index < 0) {
			throw new InvalidParameterValueException(this, value);
		}
	}

	/**
	 * Get description of selected option.
	 * 
	 * @return the string
	 */
	public String toString() {
		return (index>=0&&index < options.size()) ? options.get(index) : null;
	}

	/**
	 * Validate that the selected index corresponds to a possible option.
	 * 
	 * @throws InvalidParameterException
	 *             parameter does not meet value restriction
	 */
	public void validate() throws InvalidParameterException {
		if ((index < 0) || (index >= options.size())) {
			throw new InvalidParameterException(this);
		}
	}
	
	@Override
	public String getHumanReadableDataType() {
		StringWriter sw = new StringWriter();
		sw.append("option:");
		for(int i=0;i<options.size();i++) {
			String opt = options.get(i);
			if(i>0)
				sw.append("|");
			sw.append(opt);
		}
		return sw.toString();
	}
	@Override
	public String getXMLValue() {
		return getValue();
	}
	@Override
	public void setXMLValue(String arg) {
		setValue(arg);
	}
	@Override
	public String probeDefaultValue() {
		return getValue();
	}
}
