//------------------------------------------------
// LookUpTable
//------------------------------------------------
//
// LookUpTable for the MarchingCubes 33 Algorithm
// Version 0.2 - 13/07/2002
//
// Thomas Lewiner thomas.lewiner@polytechnique.org
// Math Dept, PUC-Rio
//
//________________________________________________


#ifndef _LOOKUPTABLE_H_
#define _LOOKUPTABLE_H_





//_____________________________________________________________________________
// For each of the possible vertex states listed in this table there is a
// specific triangulation of the edge intersection points.  The table lists
// all of them in the form of 0-5 edge triples with the list terminated by
// the invalid value -1.  For example: case[3] list the 2 triangles
// formed when cube[0] and cube[1] are inside of the surface, but the rest of
// the cube is not.
//
// Cube description:
//         7 ________ 6           _____6__             ________
//         /|       /|         7/|       /|          /|       /|
//       /  |     /  |        /  |     /5 |        /  6     /  |
//   4 /_______ /    |      /__4____ /    10     /_______3/    |
//    |     |  |5    |     |    11  |     |     |     |  |   2 |
//    |    3|__|_____|2    |     |__|__2__|     | 4   |__|_____|
//    |    /   |    /      8   3/   9    /      |    /   |    /
//    |  /     |  /        |  /     |  /1       |  /     5  /
//    |/_______|/          |/___0___|/          |/_1_____|/
//   0          1        0          1
//
//-----------------------------------------------------------------------------
static const char cases[256][2] = {
/*   0:                          */  {  0, -1 },
/*   1: 0,                       */  {  1,  0 },
/*   2:    1,                    */  {  1,  1 },
/*   3: 0, 1,                    */  {  2,  0 },
/*   4:       2,                 */  {  1,  2 },
/*   5: 0,    2,                 */  {  3,  0 },
/*   6:    1, 2,                 */  {  2,  3 },
/*   7: 0, 1, 2,                 */  {  5,  0 },
/*   8:          3,              */  {  1,  3 },
/*   9: 0,       3,              */  {  2,  1 },
/*  10:    1,    3,              */  {  3,  3 },
/*  11: 0, 1,    3,              */  {  5,  1 },
/*  12:       2, 3,              */  {  2,  5 },
/*  13: 0,    2, 3,              */  {  5,  4 },
/*  14:    1, 2, 3,              */  {  5,  9 },
/*  15: 0, 1, 2, 3,              */  {  8,  0 },
/*  16:             4,           */  {  1,  4 },
/*  17: 0,          4,           */  {  2,  2 },
/*  18:    1,       4,           */  {  3,  4 },
/*  19: 0, 1,       4,           */  {  5,  2 },
/*  20:       2,    4,           */  {  4,  2 },
/*  21: 0,    2,    4,           */  {  6,  2 },
/*  22:    1, 2,    4,           */  {  6,  9 },
/*  23: 0, 1, 2,    4,           */  { 11,  0 },
/*  24:          3, 4,           */  {  3,  8 },
/*  25: 0,       3, 4,           */  {  5,  5 },
/*  26:    1,    3, 4,           */  {  7,  3 },
/*  27: 0, 1,    3, 4,           */  {  9,  1 },
/*  28:       2, 3, 4,           */  {  6, 16 },
/*  29: 0,    2, 3, 4,           */  { 14,  3 },
/*  30:    1, 2, 3, 4,           */  { 12, 12 },
/*  31: 0, 1, 2, 3, 4,           */  {  5, 24 },
/*  32:                5,        */  {  1,  5 },
/*  33: 0,             5,        */  {  3,  1 },
/*  34:    1,          5,        */  {  2,  4 },
/*  35: 0, 1,          5,        */  {  5,  3 },
/*  36:       2,       5,        */  {  3,  6 },
/*  37: 0,    2,       5,        */  {  7,  0 },
/*  38:    1, 2,       5,        */  {  5, 10 },
/*  39: 0, 1, 2,       5,        */  {  9,  0 },
/*  40:          3,    5,        */  {  4,  3 },
/*  41: 0,       3,    5,        */  {  6,  4 },
/*  42:    1,    3,    5,        */  {  6, 11 },
/*  43: 0, 1,    3,    5,        */  { 14,  1 },
/*  44:       2, 3,    5,        */  {  6, 17 },
/*  45: 0,    2, 3,    5,        */  { 12,  4 },
/*  46:    1, 2, 3,    5,        */  { 11,  6 },
/*  47: 0, 1, 2, 3,    5,        */  {  5, 25 },
/*  48:             4, 5,        */  {  2,  8 },
/*  49: 0,          4, 5,        */  {  5,  7 },
/*  50:    1,       4, 5,        */  {  5, 12 },
/*  51: 0, 1,       4, 5,        */  {  8,  1 },
/*  52:       2,    4, 5,        */  {  6, 18 },
/*  53: 0,    2,    4, 5,        */  { 12,  5 },
/*  54:    1, 2,    4, 5,        */  { 14,  7 },
/*  55: 0, 1, 2,    4, 5,        */  {  5, 28 },
/*  56:          3, 4, 5,        */  {  6, 21 },
/*  57: 0,       3, 4, 5,        */  { 11,  4 },
/*  58:    1,    3, 4, 5,        */  { 12, 15 },
/*  59: 0, 1,    3, 4, 5,        */  {  5, 30 },
/*  60:       2, 3, 4, 5,        */  { 10,  5 },
/*  61: 0,    2, 3, 4, 5,        */  {  6, 32 },
/*  62:    1, 2, 3, 4, 5,        */  {  6, 39 },
/*  63: 0, 1, 2, 3, 4, 5,        */  {  2, 12 },
/*  64:                   6,     */  {  1,  6 },
/*  65: 0,                6,     */  {  4,  0 },
/*  66:    1,             6,     */  {  3,  5 },
/*  67: 0, 1,             6,     */  {  6,  0 },
/*  68:       2,          6,     */  {  2,  6 },
/*  69: 0,    2,          6,     */  {  6,  3 },
/*  70:    1, 2,          6,     */  {  5, 11 },
/*  71: 0, 1, 2,          6,     */  { 14,  0 },
/*  72:          3,       6,     */  {  3,  9 },
/*  73: 0,       3,       6,     */  {  6,  5 },
/*  74:    1,    3,       6,     */  {  7,  4 },
/*  75: 0, 1,    3,       6,     */  { 12,  1 },
/*  76:       2, 3,       6,     */  {  5, 14 },
/*  77: 0,    2, 3,       6,     */  { 11,  3 },
/*  78:    1, 2, 3,       6,     */  {  9,  4 },
/*  79: 0, 1, 2, 3,       6,     */  {  5, 26 },
/*  80:             4,    6,     */  {  3, 10 },
/*  81: 0,          4,    6,     */  {  6,  6 },
/*  82:    1,       4,    6,     */  {  7,  5 },
/*  83: 0, 1,       4,    6,     */  { 12,  2 },
/*  84:       2,    4,    6,     */  {  6, 19 },
/*  85: 0,    2,    4,    6,     */  { 10,  1 },
/*  86:    1, 2,    4,    6,     */  { 12, 13 },
/*  87: 0, 1, 2,    4,    6,     */  {  6, 24 },
/*  88:          3, 4,    6,     */  {  7,  7 },
/*  89: 0,       3, 4,    6,     */  { 12,  9 },
/*  90:    1,    3, 4,    6,     */  { 13,  1 },
/*  91: 0, 1,    3, 4,    6,     */  {  7,  9 },
/*  92:       2, 3, 4,    6,     */  { 12, 20 },
/*  93: 0,    2, 3, 4,    6,     */  {  6, 33 },
/*  94:    1, 2, 3, 4,    6,     */  {  7, 13 },
/*  95: 0, 1, 2, 3, 4,    6,     */  {  3, 12 },
/*  96:                5, 6,     */  {  2, 10 },
/*  97: 0,             5, 6,     */  {  6,  7 },
/*  98:    1,          5, 6,     */  {  5, 13 },
/*  99: 0, 1,          5, 6,     */  { 11,  2 },
/* 100:       2,       5, 6,     */  {  5, 16 },
/* 101: 0,    2,       5, 6,     */  { 12,  7 },
/* 102:    1, 2,       5, 6,     */  {  8,  3 },
/* 103: 0, 1, 2,       5, 6,     */  {  5, 29 },
/* 104:          3,    5, 6,     */  {  6, 22 },
/* 105: 0,       3,    5, 6,     */  { 10,  2 },
/* 106:    1,    3,    5, 6,     */  { 12, 17 },
/* 107: 0, 1,    3,    5, 6,     */  {  6, 27 },
/* 108:       2, 3,    5, 6,     */  { 14,  9 },
/* 109: 0,    2, 3,    5, 6,     */  {  6, 34 },
/* 110:    1, 2, 3,    5, 6,     */  {  5, 39 },
/* 111: 0, 1, 2, 3,    5, 6,     */  {  2, 14 },
/* 112:             4, 5, 6,     */  {  5, 20 },
/* 113: 0,          4, 5, 6,     */  { 14,  5 },
/* 114:    1,       4, 5, 6,     */  {  9,  5 },
/* 115: 0, 1,       4, 5, 6,     */  {  5, 32 },
/* 116:       2,    4, 5, 6,     */  { 11, 10 },
/* 117: 0,    2,    4, 5, 6,     */  {  6, 35 },
/* 118:    1, 2,    4, 5, 6,     */  {  5, 41 },
/* 119: 0, 1, 2,    4, 5, 6,     */  {  2, 16 },
/* 120:          3, 4, 5, 6,     */  { 12, 23 },
/* 121: 0,       3, 4, 5, 6,     */  {  6, 37 },
/* 122:    1,    3, 4, 5, 6,     */  {  7, 14 },
/* 123: 0, 1,    3, 4, 5, 6,     */  {  3, 16 },
/* 124:       2, 3, 4, 5, 6,     */  {  6, 46 },
/* 125: 0,    2, 3, 4, 5, 6,     */  {  4,  6 },
/* 126:    1, 2, 3, 4, 5, 6,     */  {  3, 21 },
/* 127: 0, 1, 2, 3, 4, 5, 6,     */  {  1,  8 },
/* 128:                      7,  */  {  1,  7 },
/* 129: 0,                   7,  */  {  3,  2 },
/* 130:    1,                7,  */  {  4,  1 },
/* 131: 0, 1,                7,  */  {  6,  1 },
/* 132:       2,             7,  */  {  3,  7 },
/* 133: 0,    2,             7,  */  {  7,  1 },
/* 134:    1, 2,             7,  */  {  6, 10 },
/* 135: 0, 1, 2,             7,  */  { 12,  0 },
/* 136:          3,          7,  */  {  2,  7 },
/* 137: 0,       3,          7,  */  {  5,  6 },
/* 138:    1,    3,          7,  */  {  6, 12 },
/* 139: 0, 1,    3,          7,  */  { 11,  1 },
/* 140:       2, 3,          7,  */  {  5, 15 },
/* 141: 0,    2, 3,          7,  */  {  9,  2 },
/* 142:    1, 2, 3,          7,  */  { 14,  6 },
/* 143: 0, 1, 2, 3,          7,  */  {  5, 27 },
/* 144:             4,       7,  */  {  2,  9 },
/* 145: 0,          4,       7,  */  {  5,  8 },
/* 146:    1,       4,       7,  */  {  6, 13 },
/* 147: 0, 1,       4,       7,  */  { 14,  2 },
/* 148:       2,    4,       7,  */  {  6, 20 },
/* 149: 0,    2,    4,       7,  */  { 12,  6 },
/* 150:    1, 2,    4,       7,  */  { 10,  3 },
/* 151: 0, 1, 2,    4,       7,  */  {  6, 25 },
/* 152:          3, 4,       7,  */  {  5, 18 },
/* 153: 0,       3, 4,       7,  */  {  8,  2 },
/* 154:    1,    3, 4,       7,  */  { 12, 16 },
/* 155: 0, 1,    3, 4,       7,  */  {  5, 31 },
/* 156:       2, 3, 4,       7,  */  { 11,  9 },
/* 157: 0,    2, 3, 4,       7,  */  {  5, 34 },
/* 158:    1, 2, 3, 4,       7,  */  {  6, 40 },
/* 159: 0, 1, 2, 3, 4,       7,  */  {  2, 13 },
/* 160:                5,    7,  */  {  3, 11 },
/* 161: 0,             5,    7,  */  {  7,  2 },
/* 162:    1,          5,    7,  */  {  6, 14 },
/* 163: 0, 1,          5,    7,  */  { 12,  3 },
/* 164:       2,       5,    7,  */  {  7,  6 },
/* 165: 0,    2,       5,    7,  */  { 13,  0 },
/* 166:    1, 2,       5,    7,  */  { 12, 14 },
/* 167: 0, 1, 2,       5,    7,  */  {  7,  8 },
/* 168:          3,    5,    7,  */  {  6, 23 },
/* 169: 0,       3,    5,    7,  */  { 12, 10 },
/* 170:    1,    3,    5,    7,  */  { 10,  4 },
/* 171: 0, 1,    3,    5,    7,  */  {  6, 28 },
/* 172:       2, 3,    5,    7,  */  { 12, 21 },
/* 173: 0,    2, 3,    5,    7,  */  {  7, 10 },
/* 174:    1, 2, 3,    5,    7,  */  {  6, 41 },
/* 175: 0, 1, 2, 3,    5,    7,  */  {  3, 13 },
/* 176:             4, 5,    7,  */  {  5, 21 },
/* 177: 0,          4, 5,    7,  */  {  9,  3 },
/* 178:    1,       4, 5,    7,  */  { 11,  8 },
/* 179: 0, 1,       4, 5,    7,  */  {  5, 33 },
/* 180:       2,    4, 5,    7,  */  { 12, 22 },
/* 181: 0,    2,    4, 5,    7,  */  {  7, 11 },
/* 182:    1, 2,    4, 5,    7,  */  {  6, 42 },
/* 183: 0, 1, 2,    4, 5,    7,  */  {  3, 14 },
/* 184:          3, 4, 5,    7,  */  { 14, 11 },
/* 185: 0,       3, 4, 5,    7,  */  {  5, 36 },
/* 186:    1,    3, 4, 5,    7,  */  {  6, 44 },
/* 187: 0, 1,    3, 4, 5,    7,  */  {  2, 17 },
/* 188:       2, 3, 4, 5,    7,  */  {  6, 47 },
/* 189: 0,    2, 3, 4, 5,    7,  */  {  3, 18 },
/* 190:    1, 2, 3, 4, 5,    7,  */  {  4,  7 },
/* 191: 0, 1, 2, 3, 4, 5,    7,  */  {  1,  9 },
/* 192:                   6, 7,  */  {  2, 11 },
/* 193: 0,                6, 7,  */  {  6,  8 },
/* 194:    1,             6, 7,  */  {  6, 15 },
/* 195: 0, 1,             6, 7,  */  { 10,  0 },
/* 196:       2,          6, 7,  */  {  5, 17 },
/* 197: 0,    2,          6, 7,  */  { 12,  8 },
/* 198:    1, 2,          6, 7,  */  { 11,  7 },
/* 199: 0, 1, 2,          6, 7,  */  {  6, 26 },
/* 200:          3,       6, 7,  */  {  5, 19 },
/* 201: 0,       3,       6, 7,  */  { 14,  4 },
/* 202:    1,    3,       6, 7,  */  { 12, 18 },
/* 203: 0, 1,    3,       6, 7,  */  {  6, 29 },
/* 204:       2, 3,       6, 7,  */  {  8,  4 },
/* 205: 0,    2, 3,       6, 7,  */  {  5, 35 },
/* 206:    1, 2, 3,       6, 7,  */  {  5, 40 },
/* 207: 0, 1, 2, 3,       6, 7,  */  {  2, 15 },
/* 208:             4,    6, 7,  */  {  5, 22 },
/* 209: 0,          4,    6, 7,  */  { 11,  5 },
/* 210:    1,       4,    6, 7,  */  { 12, 19 },
/* 211: 0, 1,       4,    6, 7,  */  {  6, 30 },
/* 212:       2,    4,    6, 7,  */  { 14, 10 },
/* 213: 0,    2,    4,    6, 7,  */  {  6, 36 },
/* 214:    1, 2,    4,    6, 7,  */  {  6, 43 },
/* 215: 0, 1, 2,    4,    6, 7,  */  {  4,  4 },
/* 216:          3, 4,    6, 7,  */  {  9,  7 },
/* 217: 0,       3, 4,    6, 7,  */  {  5, 37 },
/* 218:    1,    3, 4,    6, 7,  */  {  7, 15 },
/* 219: 0, 1,    3, 4,    6, 7,  */  {  3, 17 },
/* 220:       2, 3, 4,    6, 7,  */  {  5, 44 },
/* 221: 0,    2, 3, 4,    6, 7,  */  {  2, 19 },
/* 222:    1, 2, 3, 4,    6, 7,  */  {  3, 22 },
/* 223: 0, 1, 2, 3, 4,    6, 7,  */  {  1, 10 },
/* 224:                5, 6, 7,  */  {  5, 23 },
/* 225: 0,             5, 6, 7,  */  { 12, 11 },
/* 226:    1,          5, 6, 7,  */  { 14,  8 },
/* 227: 0, 1,          5, 6, 7,  */  {  6, 31 },
/* 228:       2,       5, 6, 7,  */  {  9,  6 },
/* 229: 0,    2,       5, 6, 7,  */  {  7, 12 },
/* 230:    1, 2,       5, 6, 7,  */  {  5, 42 },
/* 231: 0, 1, 2,       5, 6, 7,  */  {  3, 15 },
/* 232:          3,    5, 6, 7,  */  { 11, 11 },
/* 233: 0,       3,    5, 6, 7,  */  {  6, 38 },
/* 234:    1,    3,    5, 6, 7,  */  {  6, 45 },
/* 235: 0, 1,    3,    5, 6, 7,  */  {  4,  5 },
/* 236:       2, 3,    5, 6, 7,  */  {  5, 45 },
/* 237: 0,    2, 3,    5, 6, 7,  */  {  3, 19 },
/* 238:    1, 2, 3,    5, 6, 7,  */  {  2, 21 },
/* 239: 0, 1, 2, 3,    5, 6, 7,  */  {  1, 11 },
/* 240:             4, 5, 6, 7,  */  {  8,  5 },
/* 241: 0,          4, 5, 6, 7,  */  {  5, 38 },
/* 242:    1,       4, 5, 6, 7,  */  {  5, 43 },
/* 243: 0, 1,       4, 5, 6, 7,  */  {  2, 18 },
/* 244:       2,    4, 5, 6, 7,  */  {  5, 46 },
/* 245: 0,    2,    4, 5, 6, 7,  */  {  3, 20 },
/* 246:    1, 2,    4, 5, 6, 7,  */  {  2, 22 },
/* 247: 0, 1, 2,    4, 5, 6, 7,  */  {  1, 12 },
/* 248:          3, 4, 5, 6, 7,  */  {  5, 47 },
/* 249: 0,       3, 4, 5, 6, 7,  */  {  2, 20 },
/* 250:    1,    3, 4, 5, 6, 7,  */  {  3, 23 },
/* 251: 0, 1,    3, 4, 5, 6, 7,  */  {  1, 13 },
/* 252:       2, 3, 4, 5, 6, 7,  */  {  2, 23 },
/* 253: 0,    2, 3, 4, 5, 6, 7,  */  {  1, 14 },
/* 254:    1, 2, 3, 4, 5, 6, 7,  */  {  1, 15 },
/* 255: 0, 1, 2, 3, 4, 5, 6, 7,  */  {  0, -1 }
};
//_____________________________________________________________________________


//_____________________________________________________________________________
// For each of the case above, the specific triangulation of the edge
// intersection pointsis given.
// When a case is ambiguous, the table contain first the face number to test
// and then the specific triangulations depending on the results
// A minus sign means to invert the result of the test.
//-----------------------------------------------------------------------------
static const char tiling1[16][3] = {
/*   1: 0,                       */  {  0,  8,  3 },
/*   2:    1,                    */  {  0,  1,  9 },
/*   4:       2,                 */  {  1,  2, 10 },
/*   8:          3,              */  {  3, 11,  2 },
/*  16:             4,           */  {  4,  7,  8 },
/*  32:                5,        */  {  9,  5,  4 },
/*  64:                   6,     */  { 10,  6,  5 },
/* 128:                      7,  */  {  7,  6, 11 },
/* 127: 0, 1, 2, 3, 4, 5, 6,     */  {  7, 11,  6 },
/* 191: 0, 1, 2, 3, 4, 5,    7,  */  { 10,  5,  6 },
/* 223: 0, 1, 2, 3, 4,    6, 7,  */  {  9,  4,  5 },
/* 239: 0, 1, 2, 3,    5, 6, 7,  */  {  4,  8,  7 },
/* 247: 0, 1, 2,    4, 5, 6, 7,  */  {  3,  2, 11 },
/* 251: 0, 1,    3, 4, 5, 6, 7,  */  {  1, 10,  2 },
/* 253: 0,    2, 3, 4, 5, 6, 7,  */  {  0,  9,  1 },
/* 254:    1, 2, 3, 4, 5, 6, 7,  */  {  0,  3,  8 }
};
//_____________________________________________________________________________

static const char tiling2[24][6] = {
/*   3: 0, 1,                    */  {  1,  8,  3,  9,  8,  1 },
/*   9: 0,       3,              */  {  0, 11,  2,  8, 11,  0 },
/*  17: 0,          4,           */  {  4,  3,  0,  7,  3,  4 },
/*   6:    1, 2,                 */  {  9,  2, 10,  0,  2,  9 },
/*  34:    1,          5,        */  {  0,  5,  4,  1,  5,  0 },
/*  12:       2, 3,              */  {  3, 10,  1, 11, 10,  3 },
/*  68:       2,          6,     */  {  1,  6,  5,  2,  6,  1 },
/* 136:          3,          7,  */  {  7,  2,  3,  6,  2,  7 },
/*  48:             4, 5,        */  {  9,  7,  8,  5,  7,  9 },
/* 144:             4,       7,  */  {  6,  8,  4, 11,  8,  6 },
/*  96:                5, 6,     */  { 10,  4,  9,  6,  4, 10 },
/* 192:                   6, 7,  */  { 11,  5, 10,  7,  5, 11 },
/*  63: 0, 1, 2, 3, 4, 5,        */  { 11, 10,  5,  7, 11,  5 },
/* 159: 0, 1, 2, 3, 4,       7,  */  { 10,  9,  4,  6, 10,  4 },
/* 111: 0, 1, 2, 3,    5, 6,     */  {  6,  4,  8, 11,  6,  8 },
/* 207: 0, 1, 2, 3,       6, 7,  */  {  9,  8,  7,  5,  9,  7 },
/* 119: 0, 1, 2,    4, 5, 6,     */  {  7,  3,  2,  6,  7,  2 },
/* 187: 0, 1,    3, 4, 5,    7,  */  {  1,  5,  6,  2,  1,  6 },
/* 243: 0, 1,       4, 5, 6, 7,  */  {  3,  1, 10, 11,  3, 10 },
/* 221: 0,    2, 3, 4,    6, 7,  */  {  0,  4,  5,  1,  0,  5 },
/* 249: 0,       3, 4, 5, 6, 7,  */  {  9, 10,  2,  0,  9,  2 },
/* 238:    1, 2, 3,    5, 6, 7,  */  {  4,  0,  3,  7,  4,  3 },
/* 246:    1, 2,    4, 5, 6, 7,  */  {  0,  2, 11,  8,  0, 11 },
/* 252:       2, 3, 4, 5, 6, 7,  */  {  1,  3,  8,  9,  1,  8 }
};
//_____________________________________________________________________________

// One face to test
// When the test on the specified face is positive : 4 first triangles
// When the test on the specified face is negative : 2 last triangles
static const char test3[24] = {
/*   5: 0,    2,                 */    5,
/*  33: 0,             5,        */    1,
/* 129: 0,                   7,  */    4,
/*  10:    1,    3,              */    5,
/*  18:    1,       4,           */    1,
/*  66:    1,             6,     */    2,
/*  36:       2,       5,        */    2,
/* 132:       2,             7,  */    3,
/*  24:          3, 4,           */    4,
/*  72:          3,       6,     */    3,
/*  80:             4,    6,     */    6,
/* 160:                5,    7,  */    6,
/*  95: 0, 1, 2, 3, 4,    6,     */   -6,
/* 175: 0, 1, 2, 3,    5,    7,  */   -6,
/* 183: 0, 1, 2,    4, 5,    7,  */   -3,
/* 231: 0, 1, 2,       5, 6, 7,  */   -4,
/* 123: 0, 1,    3, 4, 5, 6,     */   -3,
/* 219: 0, 1,    3, 4,    6, 7,  */   -2,
/* 189: 0,    2, 3, 4, 5,    7,  */   -2,
/* 237: 0,    2, 3,    5, 6, 7,  */   -1,
/* 245: 0,    2,    4, 5, 6, 7,  */   -5,
/* 126:    1, 2, 3, 4, 5, 6,     */   -4,
/* 222:    1, 2, 3, 4,    6, 7,  */   -1,
/* 250:    1,    3, 4, 5, 6, 7,  */   -5
};

static const char tiling3[24][18] = {
/*   5: 0,    2,                 */  {  2,  3, 10,  3,  8, 10,  0,  1, 10,  0, 10,  8,    0,  8,  3,  1,  2, 10 },
/*  33: 0,             5,        */  {  8,  4,  3,  4,  5,  3,  9,  0,  3,  9,  3,  5,    9,  5,  4,  0,  8,  3 },
/* 129: 0,                   7,  */  {  7,  8,  6,  8,  0,  6,  3, 11,  6,  3,  6,  0,    3,  0,  8, 11,  7,  6 },
/*  10:    1,    3,              */  {  3,  0, 11,  0,  9, 11,  1,  2, 11,  1, 11,  9,    1,  9,  0,  2,  3, 11 },
/*  18:    1,       4,           */  {  4,  9,  7,  9,  1,  7,  0,  8,  7,  0,  7,  1,    0,  1,  9,  8,  4,  7 },
/*  66:    1,             6,     */  { 10,  1,  6,  1,  0,  6,  9,  0,  6,  9,  6,  5,    9,  0,  1,  5, 10,  6 },
/*  36:       2,       5,        */  {  5, 10,  4, 10,  2,  4,  1,  9,  4,  1,  4,  2,    1,  2, 10,  9,  5,  4 },
/* 132:       2,             7,  */  { 11,  2,  7,  2,  1,  7, 10,  6,  7, 10,  7,  1,   10,  1,  2,  6, 11,  7 },
/*  24:          3, 4,           */  { 11,  7,  2,  7,  4,  2,  8,  3,  2,  8,  2,  4,    8,  4,  7,  3, 11,  2 },
/*  72:          3,       6,     */  {  6, 11,  5, 11,  3,  5,  2, 10,  5,  2,  5,  3,    2,  3, 11, 10,  6,  5 },
/*  80:             4,    6,     */  {  7,  6,  8,  6, 10,  8,  5,  4,  8,  5,  8, 10,    5, 10,  6,  4,  7,  8 },
/* 160:                5,    7,  */  {  6,  5, 11,  5,  9, 11,  4,  7, 11,  4, 11,  9,    4,  9,  5,  7,  6, 11 },
/*  95: 0, 1, 2, 3, 4,    6,     */  {  6,  5, 11,  5,  9, 11,  4,  7, 11,  4, 11,  9,    4,  9,  5,  7,  6, 11 },
/* 175: 0, 1, 2, 3,    5,    7,  */  {  7,  6,  8,  6, 10,  8,  5,  4,  8,  5,  8, 10,    5, 10,  6,  4,  7,  8 },
/* 183: 0, 1, 2,    4, 5,    7,  */  {  6, 11,  5, 11,  3,  5,  2, 10,  5,  2,  5,  3,    2,  3, 11, 10,  6,  5 },
/* 231: 0, 1, 2,       5, 6, 7,  */  { 11,  7,  2,  7,  4,  2,  8,  3,  2,  8,  2,  4,    8,  4,  7,  3, 11,  2 },
/* 123: 0, 1,    3, 4, 5, 6,     */  { 11,  2,  7,  2,  1,  7, 10,  6,  7, 10,  7,  1,   10,  1,  2,  6, 11,  7 },
/* 219: 0, 1,    3, 4,    6, 7,  */  {  5, 10,  4, 10,  2,  4,  1,  9,  4,  1,  4,  2,    1,  2, 10,  9,  5,  4 },
/* 189: 0,    2, 3, 4, 5,    7,  */  { 10,  1,  6,  1,  0,  6,  9,  0,  6,  9,  6,  5,    9,  0,  1,  5, 10,  6 },
/* 237: 0,    2, 3,    5, 6, 7,  */  {  4,  9,  7,  9,  1,  7,  0,  8,  7,  0,  7,  1,    0,  1,  9,  8,  4,  7 },
/* 245: 0,    2,    4, 5, 6, 7,  */  {  3,  0, 11,  0,  9, 11,  1,  2, 11,  1, 11,  9,    1,  9,  0,  2,  3, 11 },
/* 126:    1, 2, 3, 4, 5, 6,     */  {  7,  8,  6,  8,  0,  6,  3, 11,  6,  3,  6,  0,    3,  0,  8, 11,  7,  6 },
/* 222:    1, 2, 3, 4,    6, 7,  */  {  8,  4,  3,  4,  5,  3,  9,  0,  3,  9,  3,  5,    9,  5,  4,  0,  8,  3 },
/* 250:    1,    3, 4, 5, 6, 7,  */  {  2,  3, 10,  3,  8, 10,  0,  1, 10,  0, 10,  8,    0,  8,  3,  1,  2, 10 }
};
//_____________________________________________________________________________

// Interior to test
// When the test on the interior is negative : 2 first triangles
// When the test on the interior is positive : 6 last triangles
static const char test4[8] = {
/*  65: 0,                6,     */   7,
/* 130:    1,                7,  */   7,
/*  20:       2,    4,           */   7,
/*  40:          3,    5,        */   7,
/* 215: 0, 1, 2,    4,    6, 7,  */  -7,
/* 235: 0, 1,    3,    5, 6, 7,  */  -7,
/* 125: 0,    2, 3, 4, 5, 6,     */  -7,
/* 190:    1, 2, 3, 4, 5,    7,  */  -7
};

static const char tiling4[8][24] = {
/*  65: 0,                6,     */  { 0,  8,  3,  5, 10,  6,    0,  5,  8,  5,  8,  6,  8,  6,  3,  6,  3, 10,  3, 10,  0, 10,  0,  5 },
/* 130:    1,                7,  */  { 0,  1,  9, 11,  7,  6,    1,  6,  9,  6,  9,  7,  9,  7,  0,  7,  0, 11,  0, 11,  1, 11,  1,  6 },
/*  20:       2,    4,           */  { 1,  2, 10,  8,  4,  7,    2,  7, 10,  7, 10,  4, 10,  4,  1,  4,  1,  8,  1,  8,  2,  8,  2,  7 },
/*  40:          3,    5,        */  { 9,  5,  4,  2,  3, 11,    3,  4, 11,  4, 11,  5, 11,  5,  2,  5,  2,  9,  2,  9,  3,  9,  3,  4 },
/* 215: 0, 1, 2,    4,    6, 7,  */  { 9,  5,  4,  2,  3, 11,    3,  4, 11,  4, 11,  5, 11,  5,  2,  5,  2,  9,  2,  9,  3,  9,  3,  4 },
/* 235: 0, 1,    3,    5, 6, 7,  */  { 1,  2, 10,  8,  4,  7,    2,  7, 10,  7, 10,  4, 10,  4,  1,  4,  1,  8,  1,  8,  2,  8,  2,  7 },
/* 125: 0,    2, 3, 4, 5, 6,     */  { 0,  1,  9, 11,  7,  6,    1,  6,  9,  6,  9,  7,  9,  7,  0,  7,  0, 11,  0, 11,  1, 11,  1,  6 },
/* 190:    1, 2, 3, 4, 5,    7,  */  { 0,  8,  3,  5, 10,  6,    0,  5,  8,  5,  8,  6,  8,  6,  3,  6,  3, 10,  3, 10,  0, 10,  0,  5 }
};
//_____________________________________________________________________________

static const char tiling5[48][9] = {
/*   7: 0, 1, 2,                 */  {  2,  8,  3,  2, 10,  8, 10,  9,  8 },
/*  11: 0, 1,    3,              */  {  1, 11,  2,  1,  9, 11,  9,  8, 11 },
/*  19: 0, 1,       4,           */  {  4,  1,  9,  4,  7,  1,  7,  3,  1 },
/*  35: 0, 1,          5,        */  {  8,  5,  4,  8,  3,  5,  3,  1,  5 },
/*  13: 0,    2, 3,              */  {  0, 10,  1,  0,  8, 10,  8, 11, 10 },
/*  25: 0,       3, 4,           */  { 11,  4,  7, 11,  2,  4,  2,  0,  4 },
/* 137: 0,       3,          7,  */  {  7,  0,  8,  7,  6,  0,  6,  2,  0 },
/*  49: 0,          4, 5,        */  {  9,  3,  0,  9,  5,  3,  5,  7,  3 },
/* 145: 0,          4,       7,  */  {  3,  6, 11,  3,  0,  6,  0,  4,  6 },
/*  14:    1, 2, 3,              */  {  3,  9,  0,  3, 11,  9, 11, 10,  9 },
/*  38:    1, 2,       5,        */  {  5,  2, 10,  5,  4,  2,  4,  0,  2 },
/*  70:    1, 2,          6,     */  {  9,  6,  5,  9,  0,  6,  0,  2,  6 },
/*  50:    1,       4, 5,        */  {  0,  7,  8,  0,  1,  7,  1,  5,  7 },
/*  98:    1,          5, 6,     */  { 10,  0,  1, 10,  6,  0,  6,  4,  0 },
/*  76:       2, 3,       6,     */  {  6,  3, 11,  6,  5,  3,  5,  1,  3 },
/* 140:       2, 3,          7,  */  { 10,  7,  6, 10,  1,  7,  1,  3,  7 },
/* 100:       2,       5, 6,     */  {  1,  4,  9,  1,  2,  4,  2,  6,  4 },
/* 196:       2,          6, 7,  */  { 11,  1,  2, 11,  7,  1,  7,  5,  1 },
/* 152:          3, 4,       7,  */  {  8,  2,  3,  8,  4,  2,  4,  6,  2 },
/* 200:          3,       6, 7,  */  {  2,  5, 10,  2,  3,  5,  3,  7,  5 },
/* 112:             4, 5, 6,     */  {  7, 10,  6,  7,  8, 10,  8,  9, 10 },
/* 176:             4, 5,    7,  */  {  6,  9,  5,  6, 11,  9, 11,  8,  9 },
/* 208:             4,    6, 7,  */  {  5,  8,  4,  5, 10,  8, 10, 11,  8 },
/* 224:                5, 6, 7,  */  {  4, 11,  7,  4,  9, 11,  9, 10, 11 },
/*  31: 0, 1, 2, 3, 4,           */  {  4,  7, 11,  4, 11,  9,  9, 11, 10 },
/*  47: 0, 1, 2, 3,    5,        */  {  5,  4,  8,  5,  8, 10, 10,  8, 11 },
/*  79: 0, 1, 2, 3,       6,     */  {  6,  5,  9,  6,  9, 11, 11,  9,  8 },
/* 143: 0, 1, 2, 3,          7,  */  {  7,  6, 10,  7, 10,  8,  8, 10,  9 },
/*  55: 0, 1, 2,    4, 5,        */  {  2, 10,  5,  2,  5,  3,  3,  5,  7 },
/* 103: 0, 1, 2,       5, 6,     */  {  8,  3,  2,  8,  2,  4,  4,  2,  6 },
/*  59: 0, 1,    3, 4, 5,        */  { 11,  2,  1, 11,  1,  7,  7,  1,  5 },
/* 155: 0, 1,    3, 4,       7,  */  {  1,  9,  4,  1,  4,  2,  2,  4,  6 },
/* 115: 0, 1,       4, 5, 6,     */  { 10,  6,  7, 10,  7,  1,  1,  7,  3 },
/* 179: 0, 1,       4, 5,    7,  */  {  6, 11,  3,  6,  3,  5,  5,  3,  1 },
/* 157: 0,    2, 3, 4,       7,  */  { 10,  1,  0, 10,  0,  6,  6,  0,  4 },
/* 205: 0,    2, 3,       6, 7,  */  {  0,  8,  7,  0,  7,  1,  1,  7,  5 },
/* 185: 0,       3, 4, 5,    7,  */  {  9,  5,  6,  9,  6,  0,  0,  6,  2 },
/* 217: 0,       3, 4,    6, 7,  */  {  5, 10,  2,  5,  2,  4,  4,  2,  0 },
/* 241: 0,          4, 5, 6, 7,  */  {  3,  0,  9,  3,  9, 11, 11,  9, 10 },
/* 110:    1, 2, 3,    5, 6,     */  {  3, 11,  6,  3,  6,  0,  0,  6,  4 },
/* 206:    1, 2, 3,       6, 7,  */  {  9,  0,  3,  9,  3,  5,  5,  3,  7 },
/* 118:    1, 2,    4, 5, 6,     */  {  7,  8,  0,  7,  0,  6,  6,  0,  2 },
/* 230:    1, 2,       5, 6, 7,  */  { 11,  7,  4, 11,  4,  2,  2,  4,  0 },
/* 242:    1,       4, 5, 6, 7,  */  {  0,  1, 10,  0, 10,  8,  8, 10, 11 },
/* 220:       2, 3, 4,    6, 7,  */  {  8,  4,  5,  8,  5,  3,  3,  5,  1 },
/* 236:       2, 3,    5, 6, 7,  */  {  4,  9,  1,  4,  1,  7,  7,  1,  3 },
/* 244:       2,    4, 5, 6, 7,  */  {  1,  2, 11,  1, 11,  9,  9, 11,  8 },
/* 248:          3, 4, 5, 6, 7,  */  {  2,  3,  8,  2,  8, 10, 10,  8,  9 }
};
//_____________________________________________________________________________

// 1 face to test + eventually the interior
// When the test on the specified face is positive : 5 first triangles
// When the test on the specified face is negative :
// - if the test on the interior is negative : 3 middle triangles
// - if the test on the interior is positive : 8 last triangles

static const char test6[48][2] = {
/*  67: 0, 1,             6,     */  {  2,  7 },
/* 131: 0, 1,                7,  */  {  4,  7 },
/*  21: 0,    2,    4,           */  {  5,  7 },
/*  69: 0,    2,          6,     */  {  5,  7 },
/*  41: 0,       3,    5,        */  {  1,  7 },
/*  73: 0,       3,       6,     */  {  3,  7 },
/*  81: 0,          4,    6,     */  {  6,  7 },
/*  97: 0,             5, 6,     */  {  1,  7 },
/* 193: 0,                6, 7,  */  {  4,  7 },
/*  22:    1, 2,    4,           */  {  1,  7 },
/* 134:    1, 2,             7,  */  {  3,  7 },
/*  42:    1,    3,    5,        */  {  5,  7 },
/* 138:    1,    3,          7,  */  {  5,  7 },
/* 146:    1,       4,       7,  */  {  1,  7 },
/* 162:    1,          5,    7,  */  {  6,  7 },
/* 194:    1,             6, 7,  */  {  2,  7 },
/*  28:       2, 3, 4,           */  {  4,  7 },
/*  44:       2, 3,    5,        */  {  2,  7 },
/*  52:       2,    4, 5,        */  {  2,  7 },
/*  84:       2,    4,    6,     */  {  6,  7 },
/* 148:       2,    4,       7,  */  {  3,  7 },
/*  56:          3, 4, 5,        */  {  4,  7 },
/* 104:          3,    5, 6,     */  {  3,  7 },
/* 168:          3,    5,    7,  */  {  6,  7 },
/*  87: 0, 1, 2,    4,    6,     */  { -6, -7 },
/* 151: 0, 1, 2,    4,       7,  */  { -3, -7 },
/* 199: 0, 1, 2,          6, 7,  */  { -4, -7 },
/* 107: 0, 1,    3,    5, 6,     */  { -3, -7 },
/* 171: 0, 1,    3,    5,    7,  */  { -6, -7 },
/* 203: 0, 1,    3,       6, 7,  */  { -2, -7 },
/* 211: 0, 1,       4,    6, 7,  */  { -2, -7 },
/* 227: 0, 1,          5, 6, 7,  */  { -4, -7 },
/*  61: 0,    2, 3, 4, 5,        */  { -2, -7 },
/*  93: 0,    2, 3, 4,    6,     */  { -6, -7 },
/* 109: 0,    2, 3,    5, 6,     */  { -1, -7 },
/* 117: 0,    2,    4, 5, 6,     */  { -5, -7 },
/* 213: 0,    2,    4,    6, 7,  */  { -5, -7 },
/* 121: 0,       3, 4, 5, 6,     */  { -3, -7 },
/* 233: 0,       3,    5, 6, 7,  */  { -1, -7 },
/*  62:    1, 2, 3, 4, 5,        */  { -4, -7 },
/* 158:    1, 2, 3, 4,       7,  */  { -1, -7 },
/* 174:    1, 2, 3,    5,    7,  */  { -6, -7 },
/* 182:    1, 2,    4, 5,    7,  */  { -3, -7 },
/* 214:    1, 2,    4,    6, 7,  */  { -1, -7 },
/* 186:    1,    3, 4, 5,    7,  */  { -5, -7 },
/* 234:    1,    3,    5, 6, 7,  */  { -5, -7 },
/* 124:       2, 3, 4, 5, 6,     */  { -4, -7 },
/* 188:       2, 3, 4, 5,    7,  */  { -2, -7 }
};

static const char tiling6[48][45] = {
/*  67: 0, 1,             6,     */  {  1, 10,  3, 10,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  9,   10,  5,  6,  3,  1,  8,  1,  8,  9,     1, 10,  3, 10,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  9,  5,  9,  1,  5,  1, 10 },
/* 131: 0, 1,                7,  */  {  3, 11,  1, 11,  1,  6,  1,  6,  9,  6,  9,  7,  9,  7,  8,   11,  7,  6,  1,  3,  9,  3,  9,  8,     3, 11,  1, 11,  1,  6,  1,  6,  9,  6,  9,  7,  9,  7,  8,  7,  8,  3,  7,  3, 11 },
/*  21: 0,    2,    4,           */  {  0,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7,  2,  7,  2,  3,    1,  2, 10,  4,  0,  7,  0,  7,  3,     0,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7,  2,  7,  2,  3,  2,  3,  0,  2,  0,  1 },
/*  69: 0,    2,          6,     */  {  2,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  0,  5,  0,  1,    3,  0,  8,  6,  2,  5,  2,  5,  1,     2,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  0,  5,  0,  1,  0,  1,  2,  0,  2,  3 },
/*  41: 0,       3,    5,        */  {  0,  9,  2,  9,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  8,    9,  4,  5,  2,  0, 11,  0, 11,  8,     0,  9,  2,  9,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  8,  4,  8,  0,  4,  0,  9 },
/*  73: 0,       3,       6,     */  {  2, 10,  0, 10,  0,  5,  0,  5,  8,  5,  8,  6,  8,  6, 11,   10,  6,  5,  0,  2,  8,  2,  8, 11,     2, 10,  0, 10,  0,  5,  0,  5,  8,  5,  8,  6,  8,  6, 11,  6, 11,  2,  6,  2, 10 },
/*  81: 0,          4,    6,     */  {  4,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3,  6,  3,  6,  7,    5,  6, 10,  0,  4,  3,  4,  3,  7,     4,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3,  6,  3,  6,  7,  6,  7,  4,  6,  4,  5 },
/*  97: 0,             5, 6,     */  {  4,  8,  6,  8,  6,  3,  6,  3, 10,  3, 10,  0, 10,  0,  9,    8,  0,  3,  6,  4, 10,  4, 10,  9,     4,  8,  6,  8,  6,  3,  6,  3, 10,  3, 10,  0, 10,  0,  9,  0,  9,  4,  0,  4,  8 },
/* 193: 0,                6, 7,  */  {  7,  8,  5,  8,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3, 11,    8,  3,  0,  5,  7, 10,  7, 10, 11,     7,  8,  5,  8,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3, 11,  3, 11,  7,  3,  7,  8 },
/*  22:    1, 2,    4,           */  {  0,  8,  2,  8,  2,  7,  2,  7, 10,  7, 10,  4, 10,  4,  9,    8,  4,  7,  2,  0, 10,  0, 10,  9,     0,  8,  2,  8,  2,  7,  2,  7, 10,  7, 10,  4, 10,  4,  9,  4,  9,  0,  4,  0,  8 },
/* 134:    1, 2,             7,  */  {  2, 11,  0, 11,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6, 10,   11,  6,  7,  0,  2,  9,  2,  9, 10,     2, 11,  0, 11,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6, 10,  6, 10,  2,  6,  2, 11 },
/*  42:    1,    3,    5,        */  {  1,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  3,  4,  3,  0,    2,  3, 11,  5,  1,  4,  1,  4,  0,     1,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  3,  4,  3,  0,  3,  0,  1,  3,  1,  2 },
/* 138:    1,    3,          7,  */  {  3,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6,  1,  6,  1,  2,    0,  1,  9,  7,  3,  6,  3,  6,  2,     3,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6,  1,  6,  1,  2,  1,  2,  3,  1,  3,  0 },
/* 146:    1,       4,       7,  */  {  4,  9,  6,  9,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  8,    9,  0,  1,  6,  4, 11,  4, 11,  8,     4,  9,  6,  9,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  8,  0,  8,  4,  0,  4,  9 },
/* 162:    1,          5,    7,  */  {  5,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  7,  0,  7,  4,    6,  7, 11,  1,  5,  0,  5,  0,  4,     5,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  7,  0,  7,  4,  7,  4,  5,  7,  5,  6 },
/* 194:    1,             6, 7,  */  {  5,  9,  7,  9,  7,  0,  7,  0, 11,  0, 11,  1, 11,  1, 10,    9,  1,  0,  7,  5, 11,  5, 11, 10,     5,  9,  7,  9,  7,  0,  7,  0, 11,  0, 11,  1, 11,  1, 10,  1, 10,  5,  1,  5,  9 },
/*  28:       2, 3, 4,           */  {  3,  8,  1,  8,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7, 11,    8,  7,  4,  1,  3, 10,  3, 10, 11,     3,  8,  1,  8,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7, 11,  7, 11,  3,  7,  3,  8 },
/*  44:       2, 3,    5,        */  {  1,  9,  3,  9,  3,  4,  3,  4, 11,  4, 11,  5, 11,  5, 10,    9,  5,  4,  3,  1, 11,  1, 11, 10,     1,  9,  3,  9,  3,  4,  3,  4, 11,  4, 11,  5, 11,  5, 10,  5, 10,  1,  5,  1,  9 },
/*  52:       2,    4, 5,        */  {  5, 10,  7, 10,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  9,   10,  1,  2,  7,  5,  8,  5,  8,  9,     5, 10,  7, 10,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  9,  1,  9,  5,  1,  5, 10 },
/*  84:       2,    4,    6,     */  {  6,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  4,  1,  4,  5,    7,  4,  8,  2,  6,  1,  6,  1,  5,     6,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  4,  1,  4,  5,  4,  5,  6,  4,  6,  7 },
/* 148:       2,    4,       7,  */  {  6, 10,  4, 10,  4,  1,  4,  1,  8,  1,  8,  2,  8,  2, 11,   10,  2,  1,  4,  6,  8,  6,  8, 11,     6, 10,  4, 10,  4,  1,  4,  1,  8,  1,  8,  2,  8,  2, 11,  2, 11,  6,  2,  6, 10 },
/*  56:          3, 4, 5,        */  {  7, 11,  5, 11,  5,  2,  5,  2,  9,  2,  9,  3,  9,  3,  8,   11,  3,  2,  5,  7,  9,  7,  9,  8,     7, 11,  5, 11,  5,  2,  5,  2,  9,  2,  9,  3,  9,  3,  8,  3,  8,  7,  3,  7, 11 },
/* 104:          3,    5, 6,     */  {  6, 11,  4, 11,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2, 10,   11,  2,  3,  4,  6,  9,  6,  9, 10,     6, 11,  4, 11,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2, 10,  2, 10,  6,  2,  6, 11 },
/* 168:          3,    5,    7,  */  {  7,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2,  5,  2,  5,  6,    4,  5,  9,  3,  7,  2,  7,  2,  6,     7,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2,  5,  2,  5,  6,  5,  6,  7,  5,  7,  4 },
/*  87: 0, 1, 2,    4,    6,     */  {  7,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2,  5,  2,  5,  6,    4,  5,  9,  3,  7,  2,  7,  2,  6,     7,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2,  5,  2,  5,  6,  5,  6,  7,  5,  7,  4 },
/* 151: 0, 1, 2,    4,       7,  */  {  6, 11,  4, 11,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2, 10,   11,  2,  3,  4,  6,  9,  6,  9, 10,     6, 11,  4, 11,  4,  3,  4,  3,  9,  3,  9,  2,  9,  2, 10,  2, 10,  6,  2,  6, 11 },
/* 199: 0, 1, 2,          6, 7,  */  {  7, 11,  5, 11,  5,  2,  5,  2,  9,  2,  9,  3,  9,  3,  8,   11,  3,  2,  5,  7,  9,  7,  9,  8,     7, 11,  5, 11,  5,  2,  5,  2,  9,  2,  9,  3,  9,  3,  8,  3,  8,  7,  3,  7, 11 },
/* 107: 0, 1,    3,    5, 6,     */  {  6, 10,  4, 10,  4,  1,  4,  1,  8,  1,  8,  2,  8,  2, 11,   10,  2,  1,  4,  6,  8,  6,  8, 11,     6, 10,  4, 10,  4,  1,  4,  1,  8,  1,  8,  2,  8,  2, 11,  2, 11,  6,  2,  6, 10 },
/* 171: 0, 1,    3,    5,    7,  */  {  6,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  4,  1,  4,  5,    7,  4,  8,  2,  6,  1,  6,  1,  5,     6,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  4,  1,  4,  5,  4,  5,  6,  4,  6,  7 },
/* 203: 0, 1,    3,       6, 7,  */  {  5, 10,  7, 10,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  9,   10,  1,  2,  7,  5,  8,  5,  8,  9,     5, 10,  7, 10,  7,  2,  7,  2,  8,  2,  8,  1,  8,  1,  9,  1,  9,  5,  1,  5, 10 },
/* 211: 0, 1,       4,    6, 7,  */  {  1,  9,  3,  9,  3,  4,  3,  4, 11,  4, 11,  5, 11,  5, 10,    9,  5,  4,  3,  1, 11,  1, 11, 10,     1,  9,  3,  9,  3,  4,  3,  4, 11,  4, 11,  5, 11,  5, 10,  5, 10,  1,  5,  1,  9 },
/* 227: 0, 1,          5, 6, 7,  */  {  3,  8,  1,  8,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7, 11,    8,  7,  4,  1,  3, 10,  3, 10, 11,     3,  8,  1,  8,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7, 11,  7, 11,  3,  7,  3,  8 },
/*  61: 0,    2, 3, 4, 5,        */  {  5,  9,  7,  9,  7,  0,  7,  0, 11,  0, 11,  1, 11,  1, 10,    9,  1,  0,  7,  5, 11,  5, 11, 10,     5,  9,  7,  9,  7,  0,  7,  0, 11,  0, 11,  1, 11,  1, 10,  1, 10,  5,  1,  5,  9 },
/*  93: 0,    2, 3, 4,    6,     */  {  5,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  7,  0,  7,  4,    6,  7, 11,  1,  5,  0,  5,  0,  4,     5,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  7,  0,  7,  4,  7,  4,  5,  7,  5,  6 },
/* 109: 0,    2, 3,    5, 6,     */  {  4,  9,  6,  9,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  8,    9,  0,  1,  6,  4, 11,  4, 11,  8,     4,  9,  6,  9,  6,  1,  6,  1, 11,  1, 11,  0, 11,  0,  8,  0,  8,  4,  0,  4,  9 },
/* 117: 0,    2,    4, 5, 6,     */  {  3,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6,  1,  6,  1,  2,    0,  1,  9,  7,  3,  6,  3,  6,  2,     3,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6,  1,  6,  1,  2,  1,  2,  3,  1,  3,  0 },
/* 213: 0,    2,    4,    6, 7,  */  {  1,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  3,  4,  3,  0,    2,  3, 11,  5,  1,  4,  1,  4,  0,     1,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  3,  4,  3,  0,  3,  0,  1,  3,  1,  2 },
/* 121: 0,       3, 4, 5, 6,     */  {  2, 11,  0, 11,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6, 10,   11,  6,  7,  0,  2,  9,  2,  9, 10,     2, 11,  0, 11,  0,  7,  0,  7,  9,  7,  9,  6,  9,  6, 10,  6, 10,  2,  6,  2, 11 },
/* 233: 0,       3,    5, 6, 7,  */  {  0,  8,  2,  8,  2,  7,  2,  7, 10,  7, 10,  4, 10,  4,  9,    8,  4,  7,  2,  0, 10,  0, 10,  9,     0,  8,  2,  8,  2,  7,  2,  7, 10,  7, 10,  4, 10,  4,  9,  4,  9,  0,  4,  0,  8 },
/*  62:    1, 2, 3, 4, 5,        */  {  7,  8,  5,  8,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3, 11,    8,  3,  0,  5,  7, 10,  7, 10, 11,     7,  8,  5,  8,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3, 11,  3, 11,  7,  3,  7,  8 },
/* 158:    1, 2, 3, 4,       7,  */  {  4,  8,  6,  8,  6,  3,  6,  3, 10,  3, 10,  0, 10,  0,  9,    8,  0,  3,  6,  4, 10,  4, 10,  9,     4,  8,  6,  8,  6,  3,  6,  3, 10,  3, 10,  0, 10,  0,  9,  0,  9,  4,  0,  4,  8 },
/* 174:    1, 2, 3,    5,    7,  */  {  4,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3,  6,  3,  6,  7,    5,  6, 10,  0,  4,  3,  4,  3,  7,     4,  5,  0,  5,  0, 10,  0, 10,  3, 10,  3,  6,  3,  6,  7,  6,  7,  4,  6,  4,  5 },
/* 182:    1, 2,    4, 5,    7,  */  {  2, 10,  0, 10,  0,  5,  0,  5,  8,  5,  8,  6,  8,  6, 11,   10,  6,  5,  0,  2,  8,  2,  8, 11,     2, 10,  0, 10,  0,  5,  0,  5,  8,  5,  8,  6,  8,  6, 11,  6, 11,  2,  6,  2, 10 },
/* 214:    1, 2,    4,    6, 7,  */  {  0,  9,  2,  9,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  8,    9,  4,  5,  2,  0, 11,  0, 11,  8,     0,  9,  2,  9,  2,  5,  2,  5, 11,  5, 11,  4, 11,  4,  8,  4,  8,  0,  4,  0,  9 },
/* 186:    1,    3, 4, 5,    7,  */  {  2,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  0,  5,  0,  1,    3,  0,  8,  6,  2,  5,  2,  5,  1,     2,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  0,  5,  0,  1,  0,  1,  2,  0,  2,  3 },
/* 234:    1,    3,    5, 6, 7,  */  {  0,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7,  2,  7,  2,  3,    1,  2, 10,  4,  0,  7,  0,  7,  3,     0,  1,  4,  1,  4, 10,  4, 10,  7, 10,  7,  2,  7,  2,  3,  2,  3,  0,  2,  0,  1 },
/* 124:       2, 3, 4, 5, 6,     */  {  3, 11,  1, 11,  1,  6,  1,  6,  9,  6,  9,  7,  9,  7,  8,   11,  7,  6,  1,  3,  9,  3,  9,  8,     3, 11,  1, 11,  1,  6,  1,  6,  9,  6,  9,  7,  9,  7,  8,  7,  8,  3,  7,  3, 11 },
/* 188:       2, 3, 4, 5,    7,  */  {  1, 10,  3, 10,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  9,   10,  5,  6,  3,  1,  8,  1,  8,  9,     1, 10,  3, 10,  3,  6,  3,  6,  8,  6,  8,  5,  8,  5,  9,  5,  9,  1,  5,  1, 10 }
};
//_____________________________________________________________________________

// 3 faces to test + eventually the interior
// When the tests on the 3 specified faces are positive :
// - if the test on the interior is positive : 5 first triangles
// - if the test on the interior is negative : 9 next triangles
// When the tests on the first  and the second specified faces are positive : 9 next triangles
// When the tests on the first  and the third  specified faces are positive : 9 next triangles
// When the tests on the second and the third  specified faces are positive : 9 next triangles
// When the test on the first  specified face is positive : 5 next triangles
// When the test on the second specified face is positive : 5 next triangles
// When the test on the third  specified face is positive : 5 next triangles
// When the tests on the 3 specified faces are negative : 3 last triangles

static const char test7[16][4] = {
/*  37: 0,    2,       5,        */  {  1,  2,  5,  7 },
/* 133: 0,    2,             7,  */  {  3,  4,  5,  7 },
/* 161: 0,             5,    7,  */  {  4,  1,  6,  7 },
/*  26:    1,    3, 4,           */  {  4,  1,  5,  7 },
/*  74:    1,    3,       6,     */  {  2,  3,  5,  7 },
/*  82:    1,       4,    6,     */  {  1,  2,  6,  7 },
/* 164:       2,       5,    7,  */  {  2,  3,  6,  7 },
/*  88:          3, 4,    6,     */  {  3,  4,  6,  7 },
/* 167: 0, 1, 2,       5,    7,  */  { -3, -4, -6, -7 },
/*  91: 0, 1,    3, 4,    6,     */  { -2, -3, -6, -7 },
/* 173: 0,    2, 3,    5,    7,  */  { -1, -2, -6, -7 },
/* 181: 0,    2,    4, 5,    7,  */  { -2, -3, -5, -7 },
/* 229: 0,    2,       5, 6, 7,  */  { -4, -1, -5, -7 },
/*  94:    1, 2, 3, 4,    6,     */  { -4, -1, -6, -7 },
/* 122:    1,    3, 4, 5, 6,     */  { -3, -4, -5, -7 },
/* 218:    1,    3, 4,    6, 7,  */  { -1, -2, -5, -7 }
};

/* 7: sub configs */
static const char subconfig7[8] = {
/*  0: 0,0,0 */ 7,
/*  1: 1,0,0 */ 6,
/*  2: 0,1,0 */ 5,
/*  3: 1,1,0 */ 3,
/*  4: 0,0,1 */ 4,
/*  5: 1,0,1 */ 2,
/*  6: 0,1,1 */ 1,
/*  7: 1,1,1 */ 0,
};


static const char tiling7[16][177] = {
/*  37: 0,    2,       5,        */  {  8,  4,  3,  4,  3, 10,  3, 10,  2,  4, 10,  5,  0,  1,  9,    8,  4,  9,  4,  9,  5,  9,  5, 10,  9, 10,  1, 10,  1,  2,  1,  2,  0,  2,  0,  3,  0,  3,  8,  0,  8,  9,    12,  2, 10, 12, 10,  5, 12,  5,  4, 12,  4,  8, 12,  8,  3, 12,  3,  0, 12,  0,  9, 12,  9,  1, 12,  1,  2,    12,  5,  4, 12,  4,  8, 12,  8,  3, 12,  3,  2, 12,  2, 10, 12, 10,  1, 12,  1,  0, 12,  0,  9, 12,  9,  5,    12,  4,  5, 12,  5, 10, 12, 10,  2, 12,  2,  3, 12,  3,  8, 12,  8,  0, 12,  0,  1, 12,  1,  9, 12,  9,  4,     1,  2, 10,  8,  4,  3,  4,  3,  5,  3,  5,  0,  5,  0,  9,     3,  0,  8,  9,  1,  4,  1,  4,  2,  4,  2,  5,  2,  5, 10,     4,  5,  9,  1, 10,  0, 10,  0,  8, 10,  8,  2,  8,  2,  3,     4,  5,  9, 10,  1,  2,  0,  3,  8 },
/* 133: 0,    2,             7,  */  { 10,  6,  1,  6,  1,  8,  1,  8,  0,  6,  8,  7,  2,  3, 11,   10,  6, 11,  6, 11,  7, 11,  7,  8, 11,  8,  3,  8,  3,  0,  3,  0,  2,  0,  2,  1,  2,  1, 10,  2, 10, 11,    12,  0,  8, 12,  8,  7, 12,  7,  6, 12,  6, 10, 12, 10,  1, 12,  1,  2, 12,  2, 11, 12, 11,  3, 12,  3,  0,    12,  7,  6, 12,  6, 10, 12, 10,  1, 12,  1,  0, 12,  0,  8, 12,  8,  3, 12,  3,  2, 12,  2, 11, 12, 11,  7,    12,  6,  7, 12,  7,  8, 12,  8,  0, 12,  0,  1, 12,  1, 10, 12, 10,  2, 12,  2,  3, 12,  3, 11, 12, 11,  6,     3,  0,  8, 10,  6,  1,  6,  1,  7,  1,  7,  2,  7,  2, 11,     1,  2, 10, 11,  3,  6,  3,  6,  0,  6,  0,  7,  0,  7,  8,     6,  7, 11,  3,  8,  2,  8,  2, 10,  8, 10,  0, 10,  0,  1,     6,  7, 11,  8,  3,  0,  2,  1, 10 },
/* 161: 0,             5,    7,  */  { 11,  3,  6,  3,  6,  9,  6,  9,  5,  3,  9,  0,  7,  4,  8,   11,  3,  8,  3,  8,  0,  8,  0,  9,  8,  9,  4,  9,  4,  5,  4,  5,  7,  5,  7,  6,  7,  6, 11,  7, 11,  8,    12,  5,  9, 12,  9,  0, 12,  0,  3, 12,  3, 11, 12, 11,  6, 12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5,    12,  0,  3, 12,  3, 11, 12, 11,  6, 12,  6,  5, 12,  5,  9, 12,  9,  4, 12,  4,  7, 12,  7,  8, 12,  8,  0,    12,  3,  0, 12,  0,  9, 12,  9,  5, 12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4, 12,  4,  8, 12,  8,  3,     4,  5,  9, 11,  3,  6,  3,  6,  0,  6,  0,  7,  0,  7,  8,     6,  7, 11,  8,  4,  3,  4,  3,  5,  3,  5,  0,  5,  0,  9,     3,  0,  8,  4,  9,  7,  9,  7, 11,  9, 11,  5, 11,  5,  6,     3,  0,  8,  9,  4,  5,  7,  6, 11 },
/*  26:    1,    3, 4,           */  { 11,  7,  2,  7,  2,  9,  2,  9,  1,  7,  9,  4,  3,  0,  8,   11,  7,  8,  7,  8,  4,  8,  4,  9,  8,  9,  0,  9,  0,  1,  0,  1,  3,  1,  3,  2,  3,  2, 11,  3, 11,  8,    12,  1,  9, 12,  9,  4, 12,  4,  7, 12,  7, 11, 12, 11,  2, 12,  2,  3, 12,  3,  8, 12,  8,  0, 12,  0,  1,    12,  4,  7, 12,  7, 11, 12, 11,  2, 12,  2,  1, 12,  1,  9, 12,  9,  0, 12,  0,  3, 12,  3,  8, 12,  8,  4,    12,  7,  4, 12,  4,  9, 12,  9,  1, 12,  1,  2, 12,  2, 11, 12, 11,  3, 12,  3,  0, 12,  0,  8, 12,  8,  7,     0,  1,  9, 11,  7,  2,  7,  2,  4,  2,  4,  3,  4,  3,  8,     2,  3, 11,  8,  0,  7,  0,  7,  1,  7,  1,  4,  1,  4,  9,     7,  4,  8,  0,  9,  3,  9,  3, 11,  9, 11,  1, 11,  1,  2,     7,  4,  8,  9,  0,  1,  3,  2, 11 },
/*  74:    1,    3,       6,     */  {  9,  5,  0,  5,  0, 11,  0, 11,  3,  5, 11,  6,  1,  2, 10,    9,  5, 10,  5, 10,  6, 10,  6, 11, 10, 11,  2, 11,  2,  3,  2,  3,  1,  3,  1,  0,  1,  0,  9,  1,  9, 10,    12,  3, 11, 12, 11,  6, 12,  6,  5, 12,  5,  9, 12,  9,  0, 12,  0,  1, 12,  1, 10, 12, 10,  2, 12,  2,  3,    12,  6,  5, 12,  5,  9, 12,  9,  0, 12,  0,  3, 12,  3, 11, 12, 11,  2, 12,  2,  1, 12,  1, 10, 12, 10,  6,    12,  5,  6, 12,  6, 11, 12, 11,  3, 12,  3,  0, 12,  0,  9, 12,  9,  1, 12,  1,  2, 12,  2, 10, 12, 10,  5,     2,  3, 11,  9,  5,  0,  5,  0,  6,  0,  6,  1,  6,  1, 10,     0,  1,  9, 10,  2,  5,  2,  5,  3,  5,  3,  6,  3,  6, 11,     5,  6, 10,  2, 11,  1, 11,  1,  9, 11,  9,  3,  9,  3,  0,     5,  6, 10, 11,  2,  3,  1,  0,  9 },
/*  82:    1,       4,    6,     */  {  8,  0,  7,  0,  7, 10,  7, 10,  6,  0, 10,  1,  4,  5,  9,    8,  0,  9,  0,  9,  1,  9,  1, 10,  9, 10,  5, 10,  5,  6,  5,  6,  4,  6,  4,  7,  4,  7,  8,  4,  8,  9,    12,  6, 10, 12, 10,  1, 12,  1,  0, 12,  0,  8, 12,  8,  7, 12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6,    12,  1,  0, 12,  0,  8, 12,  8,  7, 12,  7,  6, 12,  6, 10, 12, 10,  5, 12,  5,  4, 12,  4,  9, 12,  9,  1,    12,  0,  1, 12,  1, 10, 12, 10,  6, 12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5, 12,  5,  9, 12,  9,  0,     5,  6, 10,  8,  0,  7,  0,  7,  1,  7,  1,  4,  1,  4,  9,     7,  4,  8,  9,  5,  0,  5,  0,  6,  0,  6,  1,  6,  1, 10,     0,  1,  9,  5, 10,  4, 10,  4,  8, 10,  8,  6,  8,  6,  7,     0,  1,  9, 10,  5,  6,  4,  7,  8 },
/* 164:       2,       5,    7,  */  {  9,  1,  4,  1,  4, 11,  4, 11,  7,  1, 11,  2,  5,  6, 10,    9,  1, 10,  1, 10,  2, 10,  2, 11, 10, 11,  6, 11,  6,  7,  6,  7,  5,  7,  5,  4,  5,  4,  9,  5,  9, 10,    12,  7, 11, 12, 11,  2, 12,  2,  1, 12,  1,  9, 12,  9,  4, 12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7,    12,  2,  1, 12,  1,  9, 12,  9,  4, 12,  4,  7, 12,  7, 11, 12, 11,  6, 12,  6,  5, 12,  5, 10, 12, 10,  2,    12,  1,  2, 12,  2, 11, 12, 11,  7, 12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6, 12,  6, 10, 12, 10,  1,     6,  7, 11,  9,  1,  4,  1,  4,  2,  4,  2,  5,  2,  5, 10,     4,  5,  9, 10,  6,  1,  6,  1,  7,  1,  7,  2,  7,  2, 11,     1,  2, 10,  6, 11,  5, 11,  5,  9, 11,  9,  7,  9,  7,  4,     1,  2, 10, 11,  6,  7,  5,  4,  9 },
/*  88:          3, 4,    6,     */  { 10,  2,  5,  2,  5,  8,  5,  8,  4,  2,  8,  3,  6,  7, 11,   10,  2, 11,  2, 11,  3, 11,  3,  8, 11,  8,  7,  8,  7,  4,  7,  4,  6,  4,  6,  5,  6,  5, 10,  6, 10, 11,    12,  4,  8, 12,  8,  3, 12,  3,  2, 12,  2, 10, 12, 10,  5, 12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4,    12,  3,  2, 12,  2, 10, 12, 10,  5, 12,  5,  4, 12,  4,  8, 12,  8,  7, 12,  7,  6, 12,  6, 11, 12, 11,  3,    12,  2,  3, 12,  3,  8, 12,  8,  4, 12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7, 12,  7, 11, 12, 11,  2,     7,  4,  8, 10,  2,  5,  2,  5,  3,  5,  3,  6,  3,  6, 11,     5,  6, 10, 11,  7,  2,  7,  2,  4,  2,  4,  3,  4,  3,  8,     2,  3, 11,  7,  8,  6,  8,  6, 10,  8, 10,  4, 10,  4,  5,     2,  3, 11,  8,  7,  4,  6,  5, 10 },
/* 167: 0, 1, 2,       5,    7,  */  { 10,  2,  5,  2,  5,  8,  5,  8,  4,  2,  8,  3,  6,  7, 11,   10,  2, 11,  2, 11,  3, 11,  3,  8, 11,  8,  7,  8,  7,  4,  7,  4,  6,  4,  6,  5,  6,  5, 10,  6, 10, 11,    12,  4,  8, 12,  8,  3, 12,  3,  2, 12,  2, 10, 12, 10,  5, 12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4,    12,  3,  2, 12,  2, 10, 12, 10,  5, 12,  5,  4, 12,  4,  8, 12,  8,  7, 12,  7,  6, 12,  6, 11, 12, 11,  3,    12,  2,  3, 12,  3,  8, 12,  8,  4, 12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7, 12,  7, 11, 12, 11,  2,     7,  4,  8, 10,  2,  5,  2,  5,  3,  5,  3,  6,  3,  6, 11,     5,  6, 10, 11,  7,  2,  7,  2,  4,  2,  4,  3,  4,  3,  8,     2,  3, 11,  7,  8,  6,  8,  6, 10,  8, 10,  4, 10,  4,  5,     2,  3, 11,  8,  7,  4,  6,  5, 10 },
/*  91: 0, 1,    3, 4,    6,     */  {  9,  1,  4,  1,  4, 11,  4, 11,  7,  1, 11,  2,  5,  6, 10,    9,  1, 10,  1, 10,  2, 10,  2, 11, 10, 11,  6, 11,  6,  7,  6,  7,  5,  7,  5,  4,  5,  4,  9,  5,  9, 10,    12,  7, 11, 12, 11,  2, 12,  2,  1, 12,  1,  9, 12,  9,  4, 12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7,    12,  2,  1, 12,  1,  9, 12,  9,  4, 12,  4,  7, 12,  7, 11, 12, 11,  6, 12,  6,  5, 12,  5, 10, 12, 10,  2,    12,  1,  2, 12,  2, 11, 12, 11,  7, 12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6, 12,  6, 10, 12, 10,  1,     6,  7, 11,  9,  1,  4,  1,  4,  2,  4,  2,  5,  2,  5, 10,     4,  5,  9, 10,  6,  1,  6,  1,  7,  1,  7,  2,  7,  2, 11,     1,  2, 10,  6, 11,  5, 11,  5,  9, 11,  9,  7,  9,  7,  4,     1,  2, 10, 11,  6,  7,  5,  4,  9 },
/* 173: 0,    2, 3,    5,    7,  */  {  8,  0,  7,  0,  7, 10,  7, 10,  6,  0, 10,  1,  4,  5,  9,    8,  0,  9,  0,  9,  1,  9,  1, 10,  9, 10,  5, 10,  5,  6,  5,  6,  4,  6,  4,  7,  4,  7,  8,  4,  8,  9,    12,  6, 10, 12, 10,  1, 12,  1,  0, 12,  0,  8, 12,  8,  7, 12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6,    12,  1,  0, 12,  0,  8, 12,  8,  7, 12,  7,  6, 12,  6, 10, 12, 10,  5, 12,  5,  4, 12,  4,  9, 12,  9,  1,    12,  0,  1, 12,  1, 10, 12, 10,  6, 12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5, 12,  5,  9, 12,  9,  0,     5,  6, 10,  8,  0,  7,  0,  7,  1,  7,  1,  4,  1,  4,  9,     7,  4,  8,  9,  5,  0,  5,  0,  6,  0,  6,  1,  6,  1, 10,     0,  1,  9,  5, 10,  4, 10,  4,  8, 10,  8,  6,  8,  6,  7,     0,  1,  9, 10,  5,  6,  4,  7,  8 },
/* 181: 0,    2,    4, 5,    7,  */  {  9,  5,  0,  5,  0, 11,  0, 11,  3,  5, 11,  6,  1,  2, 10,    9,  5, 10,  5, 10,  6, 10,  6, 11, 10, 11,  2, 11,  2,  3,  2,  3,  1,  3,  1,  0,  1,  0,  9,  1,  9, 10,    12,  3, 11, 12, 11,  6, 12,  6,  5, 12,  5,  9, 12,  9,  0, 12,  0,  1, 12,  1, 10, 12, 10,  2, 12,  2,  3,    12,  6,  5, 12,  5,  9, 12,  9,  0, 12,  0,  3, 12,  3, 11, 12, 11,  2, 12,  2,  1, 12,  1, 10, 12, 10,  6,    12,  5,  6, 12,  6, 11, 12, 11,  3, 12,  3,  0, 12,  0,  9, 12,  9,  1, 12,  1,  2, 12,  2, 10, 12, 10,  5,     2,  3, 11,  9,  5,  0,  5,  0,  6,  0,  6,  1,  6,  1, 10,     0,  1,  9, 10,  2,  5,  2,  5,  3,  5,  3,  6,  3,  6, 11,     5,  6, 10,  2, 11,  1, 11,  1,  9, 11,  9,  3,  9,  3,  0,     5,  6, 10, 11,  2,  3,  1,  0,  9 },
/* 229: 0,    2,       5, 6, 7,  */  { 11,  7,  2,  7,  2,  9,  2,  9,  1,  7,  9,  4,  3,  0,  8,   11,  7,  8,  7,  8,  4,  8,  4,  9,  8,  9,  0,  9,  0,  1,  0,  1,  3,  1,  3,  2,  3,  2, 11,  3, 11,  8,    12,  1,  9, 12,  9,  4, 12,  4,  7, 12,  7, 11, 12, 11,  2, 12,  2,  3, 12,  3,  8, 12,  8,  0, 12,  0,  1,    12,  4,  7, 12,  7, 11, 12, 11,  2, 12,  2,  1, 12,  1,  9, 12,  9,  0, 12,  0,  3, 12,  3,  8, 12,  8,  4,    12,  7,  4, 12,  4,  9, 12,  9,  1, 12,  1,  2, 12,  2, 11, 12, 11,  3, 12,  3,  0, 12,  0,  8, 12,  8,  7,     0,  1,  9, 11,  7,  2,  7,  2,  4,  2,  4,  3,  4,  3,  8,     2,  3, 11,  8,  0,  7,  0,  7,  1,  7,  1,  4,  1,  4,  9,     7,  4,  8,  0,  9,  3,  9,  3, 11,  9, 11,  1, 11,  1,  2,     7,  4,  8,  9,  0,  1,  3,  2, 11 },
/*  94:    1, 2, 3, 4,    6,     */  { 11,  3,  6,  3,  6,  9,  6,  9,  5,  3,  9,  0,  7,  4,  8,   11,  3,  8,  3,  8,  0,  8,  0,  9,  8,  9,  4,  9,  4,  5,  4,  5,  7,  5,  7,  6,  7,  6, 11,  7, 11,  8,    12,  5,  9, 12,  9,  0, 12,  0,  3, 12,  3, 11, 12, 11,  6, 12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5,    12,  0,  3, 12,  3, 11, 12, 11,  6, 12,  6,  5, 12,  5,  9, 12,  9,  4, 12,  4,  7, 12,  7,  8, 12,  8,  0,    12,  3,  0, 12,  0,  9, 12,  9,  5, 12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4, 12,  4,  8, 12,  8,  3,     4,  5,  9, 11,  3,  6,  3,  6,  0,  6,  0,  7,  0,  7,  8,     6,  7, 11,  8,  4,  3,  4,  3,  5,  3,  5,  0,  5,  0,  9,     3,  0,  8,  4,  9,  7,  9,  7, 11,  9, 11,  5, 11,  5,  6,     3,  0,  8,  9,  4,  5,  7,  6, 11 },
/* 122:    1,    3, 4, 5, 6,     */  { 10,  6,  1,  6,  1,  8,  1,  8,  0,  6,  8,  7,  2,  3, 11,   10,  6, 11,  6, 11,  7, 11,  7,  8, 11,  8,  3,  8,  3,  0,  3,  0,  2,  0,  2,  1,  2,  1, 10,  2, 10, 11,    12,  0,  8, 12,  8,  7, 12,  7,  6, 12,  6, 10, 12, 10,  1, 12,  1,  2, 12,  2, 11, 12, 11,  3, 12,  3,  0,    12,  7,  6, 12,  6, 10, 12, 10,  1, 12,  1,  0, 12,  0,  8, 12,  8,  3, 12,  3,  2, 12,  2, 11, 12, 11,  7,    12,  6,  7, 12,  7,  8, 12,  8,  0, 12,  0,  1, 12,  1, 10, 12, 10,  2, 12,  2,  3, 12,  3, 11, 12, 11,  6,     3,  0,  8, 10,  6,  1,  6,  1,  7,  1,  7,  2,  7,  2, 11,     1,  2, 10, 11,  3,  6,  3,  6,  0,  6,  0,  7,  0,  7,  8,     6,  7, 11,  3,  8,  2,  8,  2, 10,  8, 10,  0, 10,  0,  1,     6,  7, 11,  8,  3,  0,  2,  1, 10 },
/* 218:    1,    3, 4,    6, 7,  */  {  8,  4,  3,  4,  3, 10,  3, 10,  2,  4, 10,  5,  0,  1,  9,    8,  4,  9,  4,  9,  5,  9,  5, 10,  9, 10,  1, 10,  1,  2,  1,  2,  0,  2,  0,  3,  0,  3,  8,  0,  8,  9,    12,  2, 10, 12, 10,  5, 12,  5,  4, 12,  4,  8, 12,  8,  3, 12,  3,  0, 12,  0,  9, 12,  9,  1, 12,  1,  2,    12,  5,  4, 12,  4,  8, 12,  8,  3, 12,  3,  2, 12,  2, 10, 12, 10,  1, 12,  1,  0, 12,  0,  9, 12,  9,  5,    12,  4,  5, 12,  5, 10, 12, 10,  2, 12,  2,  3, 12,  3,  8, 12,  8,  0, 12,  0,  1, 12,  1,  9, 12,  9,  4,     1,  2, 10,  8,  4,  3,  4,  3,  5,  3,  5,  0,  5,  0,  9,     3,  0,  8,  9,  1,  4,  1,  4,  2,  4,  2,  5,  2,  5, 10,     4,  5,  9,  1, 10,  0, 10,  0,  8, 10,  8,  2,  8,  2,  3,     4,  5,  9, 10,  1,  2,  0,  3,  8 }
};
//_____________________________________________________________________________

static const char tiling8[6][6] = {
/*  15: 0, 1, 2, 3,              */  { 9,  8, 10, 10,  8, 11 },
/*  51: 0, 1,       4, 5,        */  { 1,  5,  3,  3,  5,  7 },
/* 153: 0,       3, 4,       7,  */  { 0,  4,  2,  4,  6,  2 },
/* 102:    1, 2,       5, 6,     */  { 0,  2,  4,  4,  2,  6 },
/* 204:       2, 3,       6, 7,  */  { 1,  3,  5,  3,  7,  5 },
/* 240:             4, 5, 6, 7,  */  { 9, 10,  8, 10, 11,  8 }
};
//_____________________________________________________________________________

static const char tiling9[8][12] = {
/*  39: 0, 1, 2,       5,        */  {  2, 10,  5,  3,  2,  5,  3,  5,  4,  3,  4,  8 },
/*  27: 0, 1,    3, 4,           */  {  4,  7, 11,  9,  4, 11,  9, 11,  2,  9,  2,  1 },
/* 141: 0,    2, 3,          7,  */  { 10,  7,  6,  1,  7, 10,  1,  8,  7,  1,  0,  8 },
/* 177: 0,          4, 5,    7,  */  {  3,  6, 11,  0,  6,  3,  0,  5,  6,  0,  9,  5 },
/*  78:    1, 2, 3,       6,     */  {  3, 11,  6,  0,  3,  6,  0,  6,  5,  0,  5,  9 },
/* 114:    1,       4, 5, 6,     */  { 10,  6,  7,  1, 10,  7,  1,  7,  8,  1,  8,  0 },
/* 228:       2,       5, 6, 7,  */  {  4, 11,  7,  9, 11,  4,  9,  2, 11,  9,  1,  2 },
/* 216:          3, 4,    6, 7,  */  {  2,  5, 10,  3,  5,  2,  3,  4,  5,  3,  8,  4 }
};
//_____________________________________________________________________________

// 2 faces to test + eventually the interior
// When the tests on both specified faces are positive : 4 middle triangles (1)
// When the test on the first  specified face is positive : 8 first triangles
// When the test on the second specified face is positive : 8 next triangles
// When the tests on both specified faces are negative :
// - if the test on the interior is negative : 4 middle triangles
// - if the test on the interior is positive : 8 last triangles
static const char test10[6][3] = {
/* 195: 0, 1,             6, 7,  */  {  2,  4,  7 },
/*  85: 0,    2,    4,    6,     */  {  5,  6,  7 },
/* 105: 0,       3,    5, 6,     */  {  1,  3,  7 },
/* 150:    1, 2,    4,       7,  */  {  1,  3,  7 },
/* 170:    1,    3,    5,    7,  */  {  5,  6,  7 },
/*  60:       2, 3, 4, 5,        */  {  2,  4,  7 }
};

static const char tiling10[6][96] = {
/* 195: 0, 1,             6, 7,  */  {  12,  5,  9, 12,  9,  8, 12,  8,  3, 12,  3,  1, 12,  1, 10, 12, 10, 11, 12, 11,  7, 12,  7,  5,   12,  7,  8, 12,  8,  9, 12,  9,  1, 12,  1,  3, 12,  3, 11, 12, 11, 10, 12, 10,  5, 12,  5,  7,     5, 10,  7, 10,  7, 11,  9,  1,  8,  1,  8,  3,     5,  9,  7,  9,  7,  8, 10,  1, 11,  1, 11,  3,     3, 11,  7,  3,  7,  8,  7,  8,  9,  7,  9,  5,  9,  5, 10,  9, 10,  1, 10,  1,  3, 10,  3, 11 },
/*  85: 0,    2,    4,    6,     */  {  12,  1,  0, 12,  0,  4, 12,  4,  7, 12,  7,  3, 12,  3,  2, 12,  2,  6, 12,  6,  5, 12,  5,  1,   12,  5,  4, 12,  4,  0, 12,  0,  3, 12,  3,  7, 12,  7,  6, 12,  6,  2, 12,  2,  1, 12,  1,  5,     1,  2,  5,  2,  5,  6,  0,  3,  4,  3,  4,  7,     3,  2,  7,  2,  7,  6,  0,  1,  4,  1,  4,  5,     7,  6,  5,  7,  5,  4,  5,  4,  0,  5,  0,  1,  0,  1,  2,  0,  2,  3,  2,  3,  7,  2,  7,  6 },
/* 105: 0,       3,    5, 6,     */  {  12,  8,  4, 12,  4,  6, 12,  6, 10, 12, 10,  9, 12,  9,  0, 12,  0,  2, 12,  2, 11, 12, 11,  8,   12, 11,  6, 12,  6,  4, 12,  4,  9, 12,  9, 10, 12, 10,  2, 12,  2,  0, 12,  0,  8, 12,  8, 11,     8,  0, 11,  0, 11,  2,  4,  9,  6,  9,  6, 10,     9,  0, 10,  0, 10,  2,  4,  8,  6,  8,  6, 11,    10,  2, 11, 10, 11,  6, 11,  6,  4, 11,  4,  8,  4,  8,  0,  4,  0,  9,  0,  9, 10,  0, 10,  2 },
/* 150:    1, 2,    4,       7,  */  {  12,  9,  4, 12,  4,  6, 12,  6, 11, 12, 11,  8, 12,  8,  0, 12,  0,  2, 12,  2, 10, 12, 10,  9,   12, 10,  6, 12,  6,  4, 12,  4,  8, 12,  8, 11, 12, 11,  2, 12,  2,  0, 12,  0,  9, 12,  9, 10,     9,  0, 10,  0, 10,  2,  4,  8,  6,  8,  6, 11,     8,  0, 11,  0, 11,  2,  4,  9,  6,  9,  6, 10,    11,  2, 10, 11, 10,  6, 10,  6,  4, 10,  4,  9,  4,  9,  0,  4,  0,  8,  0,  8, 11,  0, 11,  2 },
/* 170:    1,    3,    5,    7,  */  {  12,  3,  0, 12,  0,  4, 12,  4,  5, 12,  5,  1, 12,  1,  2, 12,  2,  6, 12,  6,  7, 12,  7,  3,   12,  7,  4, 12,  4,  0, 12,  0,  1, 12,  1,  5, 12,  5,  6, 12,  6,  2, 12,  2,  3, 12,  3,  7,     3,  2,  7,  2,  7,  6,  0,  1,  4,  1,  4,  5,     1,  2,  5,  2,  5,  6,  0,  3,  4,  3,  4,  7,     5,  6,  7,  5,  7,  4,  7,  4,  0,  7,  0,  3,  0,  3,  2,  0,  2,  1,  2,  1,  5,  2,  5,  6 },
/*  60:       2, 3, 4, 5,        */  {  12,  5, 10, 12, 10, 11, 12, 11,  3, 12,  3,  1, 12,  1,  9, 12,  9,  8, 12,  8,  7, 12,  7,  5,   12,  7, 11, 12, 11, 10, 12, 10,  1, 12,  1,  3, 12,  3,  8, 12,  8,  9, 12,  9,  5, 12,  5,  7,     5,  9,  7,  9,  7,  8, 10,  1, 11,  1, 11,  3,     5, 10,  7, 10,  7, 11,  9,  1,  8,  1,  8,  3,     3,  8,  7,  3,  7, 11,  7, 11, 10,  7, 10,  5, 10,  5,  9, 10,  9,  1,  9,  1,  3,  9,  3,  8 }
};
//_____________________________________________________________________________

static const char tiling11[12][12] = {
/*  23: 0, 1, 2,    4,           */  { 2, 10,  9,  2,  9,  7,  2,  7,  3,  7,  9,  4 },
/* 139: 0, 1,    3,          7,  */  { 1,  6,  2,  1,  8,  6,  1,  9,  8,  8,  7,  6 },
/*  99: 0, 1,          5, 6,     */  { 8,  3,  1,  8,  1,  6,  8,  6,  4,  6,  1, 10 },
/*  77: 0,    2, 3,       6,     */  { 0,  8, 11,  0, 11,  5,  0,  5,  1,  5, 11,  6 },
/*  57: 0,       3, 4, 5,        */  { 9,  5,  7,  9,  7,  2,  9,  2,  0,  2,  7, 11 },
/* 209: 0,          4,    6, 7,  */  { 5,  0,  4,  5, 11,  0,  5, 10, 11, 11,  3,  0 },
/*  46:    1, 2, 3,    5,        */  { 5,  4,  0,  5,  0, 11,  5, 11, 10, 11,  0,  3 },
/* 198:    1, 2,          6, 7,  */  { 9,  7,  5,  9,  2,  7,  9,  0,  2,  2, 11,  7 },
/* 178:    1,       4, 5,    7,  */  { 0, 11,  8,  0,  5, 11,  0,  1,  5,  5,  6, 11 },
/* 156:       2, 3, 4,       7,  */  { 8,  1,  3,  8,  6,  1,  8,  4,  6,  6, 10,  1 },
/* 116:       2,    4, 5, 6,     */  { 1,  2,  6,  1,  6,  8,  1,  8,  9,  8,  6,  7 },
/* 232:          3,    5, 6, 7,  */  { 2,  9, 10,  2,  7,  9,  2,  3,  7,  7,  4,  9 }
};
//_____________________________________________________________________________

// 2 faces to test + eventually the interior
// When the tests on both specified faces are positive : 4 middle triangles (1)
// When the test on the first  specified face is positive : 8 first triangles
// When the test on the second specified face is positive : 8 next triangles
// When the tests on both specified faces are negative :
// - if the test on the interior is negative : 4 middle triangles
// - if the test on the interior is positive : 8 last triangles

static const char test12[24][3] = {
/* 135: 0, 1, 2,             7,  */  {  4,  3,  7 },
/*  75: 0, 1,    3,       6,     */  {  3,  2,  7 },
/*  83: 0, 1,       4,    6,     */  {  2,  6,  7 },
/* 163: 0, 1,          5,    7,  */  {  6,  4,  7 },
/*  45: 0,    2, 3,    5,        */  {  2,  1,  7 },
/*  53: 0,    2,    4, 5,        */  {  5,  2,  7 },
/* 149: 0,    2,    4,       7,  */  {  5,  3,  7 },
/* 101: 0,    2,       5, 6,     */  {  5,  1,  7 },
/* 197: 0,    2,          6, 7,  */  {  5,  4,  7 },
/*  89: 0,       3, 4,    6,     */  {  6,  3,  7 },
/* 169: 0,       3,    5,    7,  */  {  1,  6,  7 },
/* 225: 0,             5, 6, 7,  */  {  1,  4,  7 },
/*  30:    1, 2, 3, 4,           */  {  4,  1,  7 },
/*  86:    1, 2,    4,    6,     */  {  6,  1,  7 },
/* 166:    1, 2,       5,    7,  */  {  3,  6,  7 },
/*  58:    1,    3, 4, 5,        */  {  4,  5,  7 },
/* 154:    1,    3, 4,       7,  */  {  1,  5,  7 },
/* 106:    1,    3,    5, 6,     */  {  3,  5,  7 },
/* 202:    1,    3,       6, 7,  */  {  2,  5,  7 },
/* 210:    1,       4,    6, 7,  */  {  1,  2,  7 },
/*  92:       2, 3, 4,    6,     */  {  4,  6,  7 },
/* 172:       2, 3,    5,    7,  */  {  6,  2,  7 },
/* 180:       2,    4, 5,    7,  */  {  2,  3,  7 },
/* 120:          3, 4, 5, 6,     */  {  3,  4,  7 }
};

static const char tiling12[24][96] = {
/* 135: 0, 1, 2,             7,  */  {  12,  8,  9, 12,  9, 10, 12, 10,  2, 12,  2,  3, 12,  3, 11, 12, 11,  6, 12,  6,  7, 12,  7,  8,   12,  2, 11, 12, 11,  7, 12,  7,  6, 12,  6, 10, 12, 10,  9, 12,  9,  8, 12,  8,  3, 12,  3,  2,     7,  6, 11,  2,  3, 10,  3, 10,  8, 10,  8,  9,     3,  2, 11,  6,  7, 10,  7, 10,  8, 10,  8,  9,    11,  3,  7,  3,  7,  8,  7,  8,  9,  7,  9,  6,  9,  6, 10,  6, 10,  2,  6,  2, 11,  2, 11,  3 },
/*  75: 0, 1,    3,       6,     */  {  12, 11,  8, 12,  8,  9, 12,  9,  1, 12,  1,  2, 12,  2, 10, 12, 10,  5, 12,  5,  6, 12,  6, 11,   12,  1, 10, 12, 10,  6, 12,  6,  5, 12,  5,  9, 12,  9,  8, 12,  8, 11, 12, 11,  2, 12,  2,  1,     6,  5, 10,  1,  2,  9,  2,  9, 11,  9, 11,  8,     2,  1, 10,  5,  6,  9,  6,  9, 11,  9, 11,  8,    10,  2,  6,  2,  6, 11,  6, 11,  8,  6,  8,  5,  8,  5,  9,  5,  9,  1,  5,  1, 10,  1, 10,  2 },
/*  83: 0, 1,       4,    6,     */  {  12,  1,  3, 12,  3,  7, 12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6, 12,  6, 10, 12, 10,  1,   12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7, 12,  7,  3, 12,  3,  1, 12,  1,  9, 12,  9,  4,    10,  6,  5,  4,  9,  7,  9,  7,  1,  7,  1,  3,     9,  4,  5,  6, 10,  7, 10,  7,  1,  7,  1,  3,     5,  9, 10,  9, 10,  1, 10,  1,  3, 10,  3,  6,  3,  6,  7,  6,  7,  4,  6,  4,  5,  4,  5,  9 },
/* 163: 0, 1,          5,    7,  */  {  12,  3,  1, 12,  1,  5, 12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4, 12,  4,  8, 12,  8,  3,   12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5, 12,  5,  1, 12,  1,  3, 12,  3, 11, 12, 11,  6,    11,  6,  7,  4,  8,  5,  8,  5,  3,  5,  3,  1,     8,  4,  7,  6, 11,  5, 11,  5,  3,  5,  3,  1,     7,  8, 11,  8, 11,  3, 11,  3,  1, 11,  1,  6,  1,  6,  5,  6,  5,  4,  6,  4,  7,  4,  7,  8 },
/*  45: 0,    2, 3,    5,        */  {  12, 10, 11, 12, 11,  8, 12,  8,  0, 12,  0,  1, 12,  1,  9, 12,  9,  4, 12,  4,  5, 12,  5, 10,   12,  0,  9, 12,  9,  5, 12,  5,  4, 12,  4,  8, 12,  8, 11, 12, 11, 10, 12, 10,  1, 12,  1,  0,     5,  4,  9,  0,  1,  8,  1,  8, 10,  8, 10, 11,     1,  0,  9,  4,  5,  8,  5,  8, 10,  8, 10, 11,     9,  1,  5,  1,  5, 10,  5, 10, 11,  5, 11,  4, 11,  4,  8,  4,  8,  0,  4,  0,  9,  0,  9,  1 },
/*  53: 0,    2,    4, 5,        */  {  12,  5,  7, 12,  7,  3, 12,  3,  2, 12,  2, 10, 12, 10,  1, 12,  1,  0, 12,  0,  9, 12,  9,  5,   12,  2,  1, 12,  1,  9, 12,  9,  0, 12,  0,  3, 12,  3,  7, 12,  7,  5, 12,  5, 10, 12, 10,  2,    10,  2,  1,  0,  9,  3,  9,  3,  5,  3,  5,  7,     9,  0,  1,  2, 10,  3, 10,  3,  5,  3,  5,  7,     1,  9, 10,  9, 10,  5, 10,  5,  7, 10,  7,  2,  7,  2,  3,  2,  3,  0,  2,  0,  1,  0,  1,  9 },
/* 149: 0,    2,    4,       7,  */  {  12,  6,  4, 12,  4,  0, 12,  0,  1, 12,  1, 10, 12, 10,  2, 12,  2,  3, 12,  3, 11, 12, 11,  6,   12,  1,  2, 12,  2, 11, 12, 11,  3, 12,  3,  0, 12,  0,  4, 12,  4,  6, 12,  6, 10, 12, 10,  1,    10,  1,  2,  3, 11,  0, 11,  0,  6,  0,  6,  4,    11,  3,  2,  1, 10,  0, 10,  0,  6,  0,  6,  4,     2, 11, 10, 11, 10,  6, 10,  6,  4, 10,  4,  1,  4,  1,  0,  1,  0,  3,  1,  3,  2,  3,  2, 11 },
/* 101: 0,    2,       5, 6,     */  {  12,  4,  6, 12,  6,  2, 12,  2,  3, 12,  3,  8, 12,  8,  0, 12,  0,  1, 12,  1,  9, 12,  9,  4,   12,  3,  0, 12,  0,  9, 12,  9,  1, 12,  1,  2, 12,  2,  6, 12,  6,  4, 12,  4,  8, 12,  8,  3,     8,  3,  0,  1,  9,  2,  9,  2,  4,  2,  4,  6,     9,  1,  0,  3,  8,  2,  8,  2,  4,  2,  4,  6,     0,  9,  8,  9,  8,  4,  8,  4,  6,  8,  6,  3,  6,  3,  2,  3,  2,  1,  3,  1,  0,  1,  0,  9 },
/* 197: 0,    2,          6, 7,  */  {  12,  7,  5, 12,  5,  1, 12,  1,  0, 12,  0,  8, 12,  8,  3, 12,  3,  2, 12,  2, 11, 12, 11,  7,   12,  0,  3, 12,  3, 11, 12, 11,  2, 12,  2,  1, 12,  1,  5, 12,  5,  7, 12,  7,  8, 12,  8,  0,     8,  0,  3,  2, 11,  1, 11,  1,  7,  1,  7,  5,    11,  2,  3,  0,  8,  1,  8,  1,  7,  1,  7,  5,     3, 11,  8, 11,  8,  7,  8,  7,  5,  8,  5,  0,  5,  0,  1,  0,  1,  2,  0,  2,  3,  2,  3, 11 },
/*  89: 0,       3, 4,    6,     */  {  12,  2,  0, 12,  0,  4, 12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7, 12,  7, 11, 12, 11,  2,   12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4, 12,  4,  0, 12,  0,  2, 12,  2, 10, 12, 10,  5,    10,  5,  6,  7, 11,  4, 11,  4,  2,  4,  2,  0,    11,  7,  6,  5, 10,  4, 10,  4,  2,  4,  2,  0,     6, 11, 10, 11, 10,  2, 10,  2,  0, 10,  0,  5,  0,  5,  4,  5,  4,  7,  5,  7,  6,  7,  6, 11 },
/* 169: 0,       3,    5,    7,  */  {  12,  0,  2, 12,  2,  6, 12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5, 12,  5,  9, 12,  9,  0,   12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6, 12,  6,  2, 12,  2,  0, 12,  0,  8, 12,  8,  7,     9,  5,  4,  7,  8,  6,  8,  6,  0,  6,  0,  2,     8,  7,  4,  5,  9,  6,  9,  6,  0,  6,  0,  2,     4,  8,  9,  8,  9,  0,  9,  0,  2,  9,  2,  5,  2,  5,  6,  5,  6,  7,  5,  7,  4,  7,  4,  8 },
/* 225: 0,             5, 6, 7,  */  {  12,  9, 10, 12, 10, 11, 12, 11,  7, 12,  7,  4, 12,  4,  8, 12,  8,  3, 12,  3,  0, 12,  0,  9,   12,  7,  8, 12,  8,  0, 12,  0,  3, 12,  3, 11, 12, 11, 10, 12, 10,  9, 12,  9,  4, 12,  4,  7,     0,  3,  8,  7,  4, 11,  4, 11,  9, 11,  9, 10,     4,  7,  8,  3,  0, 11,  0, 11,  9, 11,  9, 10,     8,  4,  0,  4,  0,  9,  0,  9, 10,  0, 10,  3, 10,  3, 11,  3, 11,  7,  3,  7,  8,  7,  8,  4 },
/*  30:    1, 2, 3, 4,           */  {  12,  9, 10, 12, 10, 11, 12, 11,  7, 12,  7,  4, 12,  4,  8, 12,  8,  3, 12,  3,  0, 12,  0,  9,   12,  7,  8, 12,  8,  0, 12,  0,  3, 12,  3, 11, 12, 11, 10, 12, 10,  9, 12,  9,  4, 12,  4,  7,     4,  7,  8,  3,  0, 11,  0, 11,  9, 11,  9, 10,     0,  3,  8,  7,  4, 11,  4, 11,  9, 11,  9, 10,     8,  0,  4,  0,  4,  9,  4,  9, 10,  4, 10,  7, 10,  7, 11,  7, 11,  3,  7,  3,  8,  3,  8,  0 },
/*  86:    1, 2,    4,    6,     */  {  12,  0,  2, 12,  2,  6, 12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5, 12,  5,  9, 12,  9,  0,   12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6, 12,  6,  2, 12,  2,  0, 12,  0,  8, 12,  8,  7,     8,  7,  4,  5,  9,  6,  9,  6,  0,  6,  0,  2,     9,  5,  4,  7,  8,  6,  8,  6,  0,  6,  0,  2,     4,  9,  8,  9,  8,  0,  8,  0,  2,  8,  2,  7,  2,  7,  6,  7,  6,  5,  7,  5,  4,  5,  4,  9 },
/* 166:    1, 2,       5,    7,  */  {  12,  2,  0, 12,  0,  4, 12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7, 12,  7, 11, 12, 11,  2,   12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4, 12,  4,  0, 12,  0,  2, 12,  2, 10, 12, 10,  5,    11,  7,  6,  5, 10,  4, 10,  4,  2,  4,  2,  0,    10,  5,  6,  7, 11,  4, 11,  4,  2,  4,  2,  0,     6, 10, 11, 10, 11,  2, 11,  2,  0, 11,  0,  7,  0,  7,  4,  7,  4,  5,  7,  5,  6,  5,  6, 10 },
/*  58:    1,    3, 4, 5,        */  {  12,  7,  5, 12,  5,  1, 12,  1,  0, 12,  0,  8, 12,  8,  3, 12,  3,  2, 12,  2, 11, 12, 11,  7,   12,  0,  3, 12,  3, 11, 12, 11,  2, 12,  2,  1, 12,  1,  5, 12,  5,  7, 12,  7,  8, 12,  8,  0,    11,  2,  3,  0,  8,  1,  8,  1,  7,  1,  7,  5,     8,  0,  3,  2, 11,  1, 11,  1,  7,  1,  7,  5,     3,  8, 11,  8, 11,  7, 11,  7,  5, 11,  5,  2,  5,  2,  1,  2,  1,  0,  2,  0,  3,  0,  3,  8 },
/* 154:    1,    3, 4,       7,  */  {  12,  4,  6, 12,  6,  2, 12,  2,  3, 12,  3,  8, 12,  8,  0, 12,  0,  1, 12,  1,  9, 12,  9,  4,   12,  3,  0, 12,  0,  9, 12,  9,  1, 12,  1,  2, 12,  2,  6, 12,  6,  4, 12,  4,  8, 12,  8,  3,     9,  1,  0,  3,  8,  2,  8,  2,  4,  2,  4,  6,     8,  3,  0,  1,  9,  2,  9,  2,  4,  2,  4,  6,     0,  8,  9,  8,  9,  4,  9,  4,  6,  9,  6,  1,  6,  1,  2,  1,  2,  3,  1,  3,  0,  3,  0,  8 },
/* 106:    1,    3,    5, 6,     */  {  12,  6,  4, 12,  4,  0, 12,  0,  1, 12,  1, 10, 12, 10,  2, 12,  2,  3, 12,  3, 11, 12, 11,  6,   12,  1,  2, 12,  2, 11, 12, 11,  3, 12,  3,  0, 12,  0,  4, 12,  4,  6, 12,  6, 10, 12, 10,  1,    11,  3,  2,  1, 10,  0, 10,  0,  6,  0,  6,  4,    10,  1,  2,  3, 11,  0, 11,  0,  6,  0,  6,  4,     2, 10, 11, 10, 11,  6, 11,  6,  4, 11,  4,  3,  4,  3,  0,  3,  0,  1,  3,  1,  2,  1,  2, 10 },
/* 202:    1,    3,       6, 7,  */  {  12,  5,  7, 12,  7,  3, 12,  3,  2, 12,  2, 10, 12, 10,  1, 12,  1,  0, 12,  0,  9, 12,  9,  5,   12,  2,  1, 12,  1,  9, 12,  9,  0, 12,  0,  3, 12,  3,  7, 12,  7,  5, 12,  5, 10, 12, 10,  2,     9,  0,  1,  2, 10,  3, 10,  3,  5,  3,  5,  7,    10,  2,  1,  0,  9,  3,  9,  3,  5,  3,  5,  7,     1, 10,  9, 10,  9,  5,  9,  5,  7,  9,  7,  0,  7,  0,  3,  0,  3,  2,  0,  2,  1,  2,  1, 10 },
/* 210:    1,       4,    6, 7,  */  {  12, 10, 11, 12, 11,  8, 12,  8,  0, 12,  0,  1, 12,  1,  9, 12,  9,  4, 12,  4,  5, 12,  5, 10,   12,  0,  9, 12,  9,  5, 12,  5,  4, 12,  4,  8, 12,  8, 11, 12, 11, 10, 12, 10,  1, 12,  1,  0,     1,  0,  9,  4,  5,  8,  5,  8, 10,  8, 10, 11,     5,  4,  9,  0,  1,  8,  1,  8, 10,  8, 10, 11,     9,  5,  1,  5,  1, 10,  1, 10, 11,  1, 11,  0, 11,  0,  8,  0,  8,  4,  0,  4,  9,  4,  9,  5 },
/*  92:       2, 3, 4,    6,     */  {  12,  3,  1, 12,  1,  5, 12,  5,  6, 12,  6, 11, 12, 11,  7, 12,  7,  4, 12,  4,  8, 12,  8,  3,   12,  6,  7, 12,  7,  8, 12,  8,  4, 12,  4,  5, 12,  5,  1, 12,  1,  3, 12,  3, 11, 12, 11,  6,     8,  4,  7,  6, 11,  5, 11,  5,  3,  5,  3,  1,    11,  6,  7,  4,  8,  5,  8,  5,  3,  5,  3,  1,     7, 11,  8, 11,  8,  3,  8,  3,  1,  8,  1,  4,  1,  4,  5,  4,  5,  6,  4,  6,  7,  6,  7, 11 },
/* 172:       2, 3,    5,    7,  */  {  12,  1,  3, 12,  3,  7, 12,  7,  4, 12,  4,  9, 12,  9,  5, 12,  5,  6, 12,  6, 10, 12, 10,  1,   12,  4,  5, 12,  5, 10, 12, 10,  6, 12,  6,  7, 12,  7,  3, 12,  3,  1, 12,  1,  9, 12,  9,  4,     9,  4,  5,  6, 10,  7, 10,  7,  1,  7,  1,  3,    10,  6,  5,  4,  9,  7,  9,  7,  1,  7,  1,  3,     5, 10,  9, 10,  9,  1,  9,  1,  3,  9,  3,  4,  3,  4,  7,  4,  7,  6,  4,  6,  5,  6,  5, 10 },
/* 180:       2,    4, 5,    7,  */  {  12, 11,  8, 12,  8,  9, 12,  9,  1, 12,  1,  2, 12,  2, 10, 12, 10,  5, 12,  5,  6, 12,  6, 11,   12,  1, 10, 12, 10,  6, 12,  6,  5, 12,  5,  9, 12,  9,  8, 12,  8, 11, 12, 11,  2, 12,  2,  1,     2,  1, 10,  5,  6,  9,  6,  9, 11,  9, 11,  8,     6,  5, 10,  1,  2,  9,  2,  9, 11,  9, 11,  8,    10,  6,  2,  6,  2, 11,  2, 11,  8,  2,  8,  1,  8,  1,  9,  1,  9,  5,  1,  5, 10,  5, 10,  6 },
/* 120:          3, 4, 5, 6,     */  {  12,  8,  9, 12,  9, 10, 12, 10,  2, 12,  2,  3, 12,  3, 11, 12, 11,  6, 12,  6,  7, 12,  7,  8,   12,  2, 11, 12, 11,  7, 12,  7,  6, 12,  6, 10, 12, 10,  9, 12,  9,  8, 12,  8,  3, 12,  3,  2,     3,  2, 11,  6,  7, 10,  7, 10,  8, 10,  8,  9,     7,  6, 11,  2,  3, 10,  3, 10,  8, 10,  8,  9,    11,  7,  3,  7,  3,  8,  3,  8,  9,  3,  9,  2,  9,  2, 10,  2, 10,  6,  2,  6, 11,  6, 11,  7 }
};
//_____________________________________________________________________________


/* 13: face test */
static const char test13[2][7] = {
/* 165: 0,    2,       5,    7,  */  { 1,2,3,4,5,6,7 },
/*  90:    1,    3, 4,    6,     */  { 2,3,4,1,5,6,7 },
};


/* 13: sub configs */
static const char subconfig13[64] = {
/*  0: 0,0,0,0,0,0 */   0,
/*  1: 1,0,0,0,0,0 */   1,
/*  2: 0,1,0,0,0,0 */   2,
/*  3: 1,1,0,0,0,0 */   7,
/*  4: 0,0,1,0,0,0 */   3,
/*  5: 1,0,1,0,0,0 */  -1,
/*  6: 0,1,1,0,0,0 */  11,
/*  7: 1,1,1,0,0,0 */  -1,
/*  8: 0,0,0,1,0,0 */   4,
/*  9: 1,0,0,1,0,0 */   8,
/* 10: 0,1,0,1,0,0 */  -1,
/* 11: 1,1,0,1,0,0 */  -1,
/* 12: 0,0,1,1,0,0 */  14,
/* 13: 1,0,1,1,0,0 */  -1,
/* 14: 0,1,1,1,0,0 */  -1,
/* 15: 1,1,1,1,0,0 */  -1,
/* 16: 0,0,0,0,1,0 */   5,
/* 17: 1,0,0,0,1,0 */   9,
/* 18: 0,1,0,0,1,0 */  12,
/* 19: 1,1,0,0,1,0 */  23,
/* 20: 0,0,1,0,1,0 */  15,
/* 21: 1,0,1,0,1,0 */  -1,
/* 22: 0,1,1,0,1,0 */  21,
/* 23: 1,1,1,0,1,0 */  38,
/* 24: 0,0,0,1,1,0 */  17,
/* 25: 1,0,0,1,1,0 */  20,
/* 26: 0,1,0,1,1,0 */  -1,
/* 27: 1,1,0,1,1,0 */  36,
/* 28: 0,0,1,1,1,0 */  26,
/* 29: 1,0,1,1,1,0 */  33,
/* 30: 0,1,1,1,1,0 */  30,
/* 31: 1,1,1,1,1,0 */  44,
/* 32: 0,0,0,0,0,1 */   6,
/* 33: 1,0,0,0,0,1 */  10,
/* 34: 0,1,0,0,0,1 */  13,
/* 35: 1,1,0,0,0,1 */  19,
/* 36: 0,0,1,0,0,1 */  16,
/* 37: 1,0,1,0,0,1 */  -1,
/* 38: 0,1,1,0,0,1 */  25,
/* 39: 1,1,1,0,0,1 */  37,
/* 40: 0,0,0,1,0,1 */  18,
/* 41: 1,0,0,1,0,1 */  24,
/* 42: 0,1,0,1,0,1 */  -1,
/* 43: 1,1,0,1,0,1 */  35,
/* 44: 0,0,1,1,0,1 */  22,
/* 45: 1,0,1,1,0,1 */  32,
/* 46: 0,1,1,1,0,1 */  29,
/* 47: 1,1,1,1,0,1 */  43,
/* 48: 0,0,0,0,1,1 */  -1,
/* 49: 1,0,0,0,1,1 */  -1,
/* 50: 0,1,0,0,1,1 */  -1,
/* 51: 1,1,0,0,1,1 */  34,
/* 52: 0,0,1,0,1,1 */  -1,
/* 53: 1,0,1,0,1,1 */  -1,
/* 54: 0,1,1,0,1,1 */  28,
/* 55: 1,1,1,0,1,1 */  42,
/* 56: 0,0,0,1,1,1 */  -1,
/* 57: 1,0,0,1,1,1 */  31,
/* 58: 0,1,0,1,1,1 */  -1,
/* 59: 1,1,0,1,1,1 */  41,
/* 60: 0,0,1,1,1,1 */  27,
/* 61: 1,0,1,1,1,1 */  40,
/* 62: 0,1,1,1,1,1 */  39,
/* 63: 1,1,1,1,1,1 */  45,
};



/* 13.1 */
static const char tiling13_1[2][12] = {
/* 165: 0,    2,       5,    7,  */  { 6,7,11,1,2,10,0,3,8,4,5,9 },
/*  90:    1,    3, 4,    6,     */  { 7,4,8,2,3,11,1,0,9,5,6,10 }
};



/* 13.2 */
static const char tiling13_2[2][6][18] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1 */ { 1,2,10,6,7,11,8,4,3,4,3,5,3,5,0,5,0,9 },
/* 2 */ { 0,3,8,6,7,11,9,1,4,1,4,2,4,2,5,2,5,10 },
/* 3 */ { 4,5,9,0,3,8,10,6,1,6,1,7,1,7,2,7,2,11 },
/* 4 */ { 4,5,9,1,2,10,11,3,6,3,6,0,6,0,7,0,7,8 },
/* 5 */ { 4,5,9,6,7,11,1,10,0,10,0,8,10,8,2,8,2,3 },
/* 6 */ { 1,2,10,3,0,8,4,9,7,9,7,11,9,11,5,11,5,6 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1 */ { 2,3,11,7,4,8,9,5,0,5,0,6,0,6,1,6,1,10 },
/* 2 */ { 1,0,9,7,4,8,10,2,5,2,5,3,5,3,6,3,6,11 },
/* 3 */ { 5,6,10,1,0,9,11,7,2,7,2,4,2,4,3,4,3,8 },
/* 4 */ { 5,6,10,2,3,11,8,0,7,0,7,1,7,1,4,1,4,9 },
/* 5 */ { 5,6,10,7,4,8,2,11,1,11,1,9,11,9,3,9,3,0 },
/* 6 */ { 2,3,11,0,1,9,5,10,4,10,4,8,10,8,6,8,6,7 }
} };



/* 13.3 */
static const char tiling13_3[2][12][30] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1,2 */ { 6,7,11,12,2,10,12,10,5,12,5,4,12,4,8,12,8,3,12,3,0,12,0,9,12,9,1,12,1,2 },
/* 1,4 */ { 1,2,10,12,5,9,12,9,0,12,0,3,12,3,11,12,11,6,12,6,7,12,7,8,12,8,4,12,4,5 },
/* 1,5 */ { 6,7,11,12,5,4,12,4,8,12,8,3,12,3,2,12,2,10,12,10,1,12,1,0,12,0,9,12,9,5 },
/* 1,6 */ { 1,2,10,12,3,0,12,0,9,12,9,5,12,5,6,12,6,11,12,11,7,12,7,4,12,4,8,12,8,3 },
/* 2,3 */ { 0,3,8,12,7,11,12,11,2,12,2,1,12,1,9,12,9,4,12,4,5,12,5,10,12,10,6,12,6,7 },
/* 2,5 */ { 6,7,11,12,4,5,12,5,10,12,10,2,12,2,3,12,3,8,12,8,0,12,0,1,12,1,9,12,9,4 },
/* 2,6 */ { 0,3,8,12,2,1,12,1,9,12,9,4,12,4,7,12,7,11,12,11,6,12,6,5,12,5,10,12,10,2 },
/* 3,4 */ { 4,5,9,12,0,8,12,8,7,12,7,6,12,6,10,12,10,1,12,1,2,12,2,11,12,11,3,12,3,0 },
/* 3,5 */ { 4,5,9,12,7,6,12,6,10,12,10,1,12,1,0,12,0,8,12,8,3,12,3,2,12,2,11,12,11,7 },
/* 3,6 */ { 0,3,8,12,1,2,12,2,11,12,11,7,12,7,4,12,4,9,12,9,5,12,5,6,12,6,10,12,10,1 },
/* 4,5 */ { 4,5,9,12,6,7,12,7,8,12,8,0,12,0,1,12,1,10,12,10,2,12,2,3,12,3,11,12,11,6 },
/* 4,6 */ { 1,2,10,12,0,3,12,3,11,12,11,6,12,6,5,12,5,9,12,9,4,12,4,7,12,7,8,12,8,0 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1,2 */ { 7,4,8,12,3,11,12,11,6,12,6,5,12,5,9,12,9,0,12,0,1,12,1,10,12,10,2,12,2,3 },
/* 1,4 */ { 2,3,11,12,6,10,12,10,1,12,1,0,12,0,8,12,8,7,12,7,4,12,4,9,12,9,5,12,5,6 },
/* 1,5 */ { 7,4,8,12,6,5,12,5,9,12,9,0,12,0,3,12,3,11,12,11,2,12,2,1,12,1,10,12,10,6 },
/* 1,6 */ { 2,3,11,12,0,1,12,1,10,12,10,6,12,6,7,12,7,8,12,8,4,12,4,5,12,5,9,12,9,0 },
/* 2,3 */ { 0,1,9,12,4,8,12,8,3,12,3,2,12,2,10,12,10,5,12,5,6,12,6,11,12,11,7,12,7,4 },
/* 2,5 */ { 7,4,8,12,5,6,12,6,11,12,11,3,12,3,0,12,0,9,12,9,1,12,1,2,12,2,10,12,10,5 },
/* 2,6 */ { 1,0,9,12,3,2,12,2,10,12,10,5,12,5,4,12,4,8,12,8,7,12,7,6,12,6,11,12,11,3 },
/* 3,4 */ { 5,6,10,12,1,9,12,9,4,12,4,7,12,7,11,12,11,2,12,2,3,12,3,8,12,8,0,12,0,1 },
/* 3,5 */ { 5,6,10,12,4,7,12,7,11,12,11,2,12,2,1,12,1,9,12,9,0,12,0,3,12,3,8,12,8,4 },
/* 3,6 */ { 1,0,9,12,2,3,12,3,8,12,8,4,12,4,5,12,5,10,12,10,6,12,6,7,12,7,11,12,11,2 },
/* 4,5 */ { 5,6,10,12,7,4,12,4,9,12,9,1,12,1,2,12,2,11,12,11,3,12,3,0,12,0,8,12,8,7 },
/* 4,6 */ { 2,3,11,12,1,0,12,0,8,12,8,7,12,7,6,12,6,10,12,10,5,12,5,4,12,4,9,12,9,1 }
} };



/* 13.4 */
static const char tiling13_4[2][4][36] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1,2,6 */ { 12,2,10,12,10,5,12,5,6,12,6,11,12,11,7,12,7,4,12,4,8,12,8,3,12,3,0,12,0,9,12,9,1,12,1,2 },
/* 1,4,5 */ { 12,3,11,12,11,6,12,6,7,12,7,8,12,8,4,12,4,5,12,5,9,12,9,0,12,0,1,12,1,10,12,10,2,12,2,3 },
/* 2,3,5 */ { 12,1,9,12,9,4,12,4,5,12,5,10,12,10,6,12,6,7,12,7,11,12,11,2,12,2,3,12,3,8,12,8,0,12,0,1 },
/* 3,4,6 */ { 12,0,8,12,8,7,12,7,4,12,4,9,12,9,5,12,5,6,12,6,10,12,10,1,12,1,2,12,2,11,12,11,3,12,3,0 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1,2,6 */ { 12,3,11,12,11,6,12,6,7,12,7,8,12,8,4,12,4,5,12,5,9,12,9,0,12,0,1,12,1,10,12,10,2,12,2,3 },
/* 1,4,5 */ { 12,0,8,12,8,7,12,7,4,12,4,9,12,9,5,12,5,6,12,6,10,12,10,1,12,1,2,12,2,11,12,11,3,12,3,0 },
/* 2,3,5 */ { 12,2,10,12,10,5,12,5,6,12,6,11,12,11,7,12,7,4,12,4,8,12,8,3,12,3,0,12,0,9,12,9,1,12,1,2 },
/* 3,4,6 */ { 12,1,9,12,9,4,12,4,5,12,5,10,12,10,6,12,6,7,12,7,11,12,11,2,12,2,3,12,3,8,12,8,0,12,0,1 }
} };



/* 13.5.1 */
static const char tiling13_5_1[2][4][18] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1,2,5 */ { 6,7,11,1,0,9,2,3,10,3,10,5,3,5,8,5,8,4 },
/* 1,4,6 */ { 2,1,10,7,4,8,3,0,11,0,11,6,0,6,9,6,9,5 },
/* 2,3,6 */ { 0,3,8,5,6,10,1,2,9,2,9,4,2,4,11,4,11,7 },
/* 3,4,5 */ { 4,5,9,3,2,11,0,1,8,1,8,7,1,7,10,7,10,6 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1,2,5 */ { 7,4,8,2,1,10,3,0,11,0,11,6,0,6,9,6,9,5 },
/* 1,4,6 */ { 3,2,11,4,5,9,0,1,8,1,8,7,1,7,10,7,10,6 },
/* 2,3,6 */ { 1,0,9,6,7,11,2,3,10,3,10,5,3,5,8,5,8,4 },
/* 3,4,5 */ { 5,6,10,0,3,8,1,2,9,2,9,4,2,4,11,4,11,7 }
} };



/* 13.5.2 */
static const char tiling13_5_2[2][4][30] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1,2,5 */ { 0,1,9,7,4,8,7,8,3,7,3,11,3,11,2,11,2,10,11,10,6,10,6,5,6,5,7,5,7,4 },
/* 1,4,6 */ { 7,4,8,2,3,11,2,11,6,2,6,10,6,10,5,10,5,9,10,9,1,9,1,0,1,0,2,0,2,3 },
/* 2,3,6 */ { 5,6,10,0,1,9,0,9,4,0,4,8,4,8,7,8,7,11,8,11,3,11,3,2,3,2,0,2,0,1 },
/* 3,4,5 */ { 2,3,11,5,6,10,5,10,1,5,1,9,1,9,0,9,0,8,9,8,4,4,8,7,4,7,5,7,5,6 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1,2,5 */ { 1,2,10,4,5,9,4,9,0,4,0,8,0,8,3,8,3,11,8,11,7,11,7,6,7,6,4,6,4,5 },
/* 1,4,6 */ { 4,5,9,3,0,8,3,8,7,3,7,11,7,11,6,11,6,10,11,10,2,10,2,1,2,1,3,1,3,0 },
/* 2,3,6 */ { 6,7,11,1,2,10,1,10,5,1,5,9,5,9,4,9,4,8,9,8,0,8,0,3,0,3,1,3,1,2 },
/* 3,4,5 */ { 3,0,8,6,7,11,6,11,2,6,2,10,2,10,1,10,1,9,10,9,5,5,9,4,5,4,6,4,6,7 }
} };



/* 13.3 */
static const char tiling13_3_[2][12][30] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1,2 */ { 3,2,11,12,7,8,12,8,0,12,0,1,12,1,10,12,10,6,12,6,5,12,5,9,12,9,4,12,4,7 },
/* 1,4 */ { 6,5,10,12,2,11,12,11,7,12,7,4,12,4,9,12,9,1,12,1,0,12,0,8,12,8,3,12,3,2 },
/* 1,5 */ { 5,6,10,12,7,4,12,4,9,12,9,1,12,1,2,12,2,11,12,11,3,12,3,0,12,0,8,12,8,7 },
/* 1,6 */ { 2,3,11,12,1,0,12,0,8,12,8,7,12,7,6,12,6,10,12,10,5,12,5,4,12,4,9,12,9,1 },
/* 2,3 */ { 7,4,8,12,3,11,12,11,6,12,6,5,12,5,9,12,9,0,12,0,1,12,1,10,12,10,2,12,2,3 },
/* 2,5 */ { 7,4,8,12,6,5,12,5,9,12,9,0,12,0,3,12,3,11,12,11,2,12,2,1,12,1,10,12,10,6 },
/* 2,6 */ { 2,3,11,12,0,1,12,1,10,12,10,6,12,6,7,12,7,8,12,8,4,12,4,5,12,5,9,12,9,0 },
/* 3,4 */ { 1,0,9,12,4,8,12,8,3,12,3,2,12,2,10,12,10,5,12,5,6,12,6,11,12,11,7,12,7,4 },
/* 3,5 */ { 7,4,8,12,5,6,12,6,11,12,11,3,12,3,0,12,0,9,12,9,1,12,1,2,12,2,10,12,10,5 },
/* 3,6 */ { 1,0,9,12,3,2,12,2,10,12,10,5,12,5,4,12,4,8,12,8,7,12,7,6,12,6,11,12,11,3 },
/* 4,5 */ { 5,6,10,12,4,7,12,7,11,12,11,2,12,2,1,12,1,9,12,9,0,12,0,3,12,3,8,12,8,4 },
/* 4,6 */ { 0,1,9,12,2,3,12,3,8,12,8,4,12,4,5,12,5,10,12,10,6,12,6,7,12,7,11,12,11,2 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1,2 */ { 0,3,8,12,4,9,12,9,1,12,1,2,12,2,11,12,11,7,12,7,6,12,6,10,12,10,5,12,5,4 },
/* 1,4 */ { 7,6,11,12,3,8,12,8,4,12,4,5,12,5,10,12,10,2,12,2,1,12,1,9,12,9,0,12,0,3 },
/* 1,5 */ { 6,7,11,12,4,5,12,5,10,12,10,2,12,2,3,12,3,8,12,8,0,12,0,1,12,1,9,12,9,4 },
/* 1,6 */ { 3,0,8,12,2,1,12,1,9,12,9,4,12,4,7,12,7,11,12,11,6,12,6,5,12,5,10,12,10,2 },
/* 2,3 */ { 4,5,9,12,0,8,12,8,7,12,7,6,12,6,10,12,10,1,12,1,2,12,2,11,12,11,3,12,3,0 },
/* 2,5 */ { 4,5,9,12,7,6,12,6,10,12,10,1,12,1,0,12,0,8,12,8,3,12,3,2,12,2,11,12,11,7 },
/* 2,6 */ { 3,0,8,12,1,2,12,2,11,12,11,7,12,7,4,12,4,9,12,9,5,12,5,6,12,6,10,12,10,1 },
/* 3,4 */ { 2,1,10,12,5,9,12,9,0,12,0,3,12,3,11,12,11,6,12,6,7,12,7,8,12,8,4,12,4,5 },
/* 3,5 */ { 4,5,9,12,6,7,12,7,8,12,8,0,12,0,1,12,1,10,12,10,2,12,2,3,12,3,11,12,11,6 },
/* 3,6 */ { 2,1,10,12,0,3,12,3,11,12,11,6,12,6,5,12,5,9,12,9,4,12,4,7,12,7,8,12,8,0 },
/* 4,5 */ { 6,7,11,12,5,4,12,4,8,12,8,3,12,3,2,12,2,10,12,10,1,12,1,0,12,0,9,12,9,5 },
/* 4,6 */ { 1,2,10,12,3,0,12,0,9,12,9,5,12,5,6,12,6,11,12,11,7,12,7,4,12,4,8,12,8,3 }
} };



/* 13.2 */
static const char tiling13_2_[2][6][18] = {
/* 165: 0,    2,       5,    7,  */  {
/* 1 */ { 5,6,10,2,3,11,8,0,7,0,7,1,7,1,4,1,4,9 },
/* 2 */ { 2,3,11,7,4,8,9,5,0,5,0,6,0,6,1,6,1,10 },
/* 3 */ { 1,0,9,7,4,8,10,2,5,2,5,3,5,3,6,3,6,11 },
/* 4 */ { 5,6,10,1,0,9,11,7,2,7,2,4,2,4,3,4,3,8 },
/* 5 */ { 5,6,10,7,4,8,2,11,1,11,1,9,11,9,3,9,3,0 },
/* 6 */ { 2,3,11,0,1,9,5,10,4,10,4,8,10,8,6,8,6,7 }
},
/*  90:    1,    3, 4,    6,     */  {
/* 1 */ { 6,7,11,3,0,8,9,1,4,1,4,2,4,2,5,2,5,10 },
/* 2 */ { 3,0,8,4,5,9,10,6,1,6,1,7,1,7,2,7,2,11 },
/* 3 */ { 2,1,10,4,5,9,11,3,6,3,6,0,6,0,7,0,7,8 },
/* 4 */ { 6,7,11,2,1,10,8,4,3,4,3,5,3,5,0,5,0,9 },
/* 5 */ { 6,7,11,4,5,9,3,8,2,8,2,10,8,10,0,10,0,1 },
/* 6 */ { 3,0,8,1,2,10,6,11,5,11,5,9,11,9,7,9,7,4 }
} };

/* 13.1 */
static const char tiling13_1_[2][12] = {
/* 165: 0,    2,       5,    7,  */  { 7,4,8,2,3,11,1,0,9,5,6,10 },
/*  90:    1,    3, 4,    6,     */  { 6,7,11,1,2,10,0,3,8,4,5,9 }
};

//_____________________________________________________________________________

static const char tiling14[12][12] = {
/*  71: 0, 1, 2,          6,     */  { 5,  9,  8,  5,  8,  2,  5,  2,  6,  3,  2,  8 },
/*  43: 0, 1,    3,    5,        */  { 2,  1,  5,  2,  5,  8,  2,  8, 11,  4,  8,  5 },
/* 147: 0, 1,       4,       7,  */  { 9,  4,  6,  9,  6,  3,  9,  3,  1, 11,  3,  6 },
/*  29: 0,    2, 3, 4,           */  { 1, 11, 10,  1,  4, 11,  1,  0,  4,  7, 11,  4 },
/* 201: 0,       3,       6, 7,  */  { 8,  2,  0,  8,  5,  2,  8,  7,  5, 10,  2,  5 },
/* 113: 0,          4, 5, 6,     */  { 0,  7,  3,  0, 10,  7,  0,  9, 10,  6,  7, 10 },
/* 142:    1, 2, 3,          7,  */  { 0,  3,  7,  0,  7, 10,  0, 10,  9,  6, 10,  7 },
/*  54:    1, 2,    4, 5,        */  { 8,  0,  2,  8,  2,  5,  8,  5,  7, 10,  5,  2 },
/* 226:    1,          5, 6, 7,  */  { 1, 10, 11,  1, 11,  4,  1,  4,  0,  7,  4, 11 },
/* 108:       2, 3,    5, 6,     */  { 9,  6,  4,  9,  3,  6,  9,  1,  3, 11,  6,  3 },
/* 212:       2,    4,    6, 7,  */  { 2,  5,  1,  2,  8,  5,  2, 11,  8,  4,  5,  8 },
/* 184:          3, 4, 5,    7,  */  { 5,  8,  9,  5,  2,  8,  5,  6,  2,  3,  8,  2 }
};
//_____________________________________________________________________________



//_____________________________________________________________________________
// For each of the possible vertex states listed in this table there is a
// specific triangulation of the edge intersection points.  The table lists
// all of them in the form of 0-5 edge triples with the list terminated by
// the invalid value -1.  For example: casesClassic[3] list the 2 triangles
// formed when cube[0] and cube[1] are inside of the surface, but the rest of
// the cube is not.
//-----------------------------------------------------------------------------
static const char casesClassic[256][16] = {
  {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  1,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  8,  3,  9,  8,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  3,  1,  2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 9,  2, 10,  0,  2,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 2,  8,  3,  2, 10,  8, 10,  9,  8, -1, -1, -1, -1, -1, -1, -1},
  { 3, 11,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0, 11,  2,  8, 11,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  9,  0,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1, 11,  2,  1,  9, 11,  9,  8, 11, -1, -1, -1, -1, -1, -1, -1},
  { 3, 10,  1, 11, 10,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0, 10,  1,  0,  8, 10,  8, 11, 10, -1, -1, -1, -1, -1, -1, -1},
  { 3,  9,  0,  3, 11,  9, 11, 10,  9, -1, -1, -1, -1, -1, -1, -1},
  { 9,  8, 10, 10,  8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  3,  0,  7,  3,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  1,  9,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  1,  9,  4,  7,  1,  7,  3,  1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 10,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 3,  4,  7,  3,  0,  4,  1,  2, 10, -1, -1, -1, -1, -1, -1, -1},
  { 9,  2, 10,  9,  0,  2,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1},
  { 2, 10,  9,  2,  9,  7,  2,  7,  3,  7,  9,  4, -1, -1, -1, -1},
  { 8,  4,  7,  3, 11,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11,  4,  7, 11,  2,  4,  2,  0,  4, -1, -1, -1, -1, -1, -1, -1},
  { 9,  0,  1,  8,  4,  7,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1},
  { 4,  7, 11,  9,  4, 11,  9, 11,  2,  9,  2,  1, -1, -1, -1, -1},
  { 3, 10,  1,  3, 11, 10,  7,  8,  4, -1, -1, -1, -1, -1, -1, -1},
  { 1, 11, 10,  1,  4, 11,  1,  0,  4,  7, 11,  4, -1, -1, -1, -1},
  { 4,  7,  8,  9,  0, 11,  9, 11, 10, 11,  0,  3, -1, -1, -1, -1},
  { 4,  7, 11,  4, 11,  9,  9, 11, 10, -1, -1, -1, -1, -1, -1, -1},
  { 9,  5,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 9,  5,  4,  0,  8,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  5,  4,  1,  5,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 8,  5,  4,  8,  3,  5,  3,  1,  5, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 10,  9,  5,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 3,  0,  8,  1,  2, 10,  4,  9,  5, -1, -1, -1, -1, -1, -1, -1},
  { 5,  2, 10,  5,  4,  2,  4,  0,  2, -1, -1, -1, -1, -1, -1, -1},
  { 2, 10,  5,  3,  2,  5,  3,  5,  4,  3,  4,  8, -1, -1, -1, -1},
  { 9,  5,  4,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0, 11,  2,  0,  8, 11,  4,  9,  5, -1, -1, -1, -1, -1, -1, -1},
  { 0,  5,  4,  0,  1,  5,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1},
  { 2,  1,  5,  2,  5,  8,  2,  8, 11,  4,  8,  5, -1, -1, -1, -1},
  {10,  3, 11, 10,  1,  3,  9,  5,  4, -1, -1, -1, -1, -1, -1, -1},
  { 4,  9,  5,  0,  8,  1,  8, 10,  1,  8, 11, 10, -1, -1, -1, -1},
  { 5,  4,  0,  5,  0, 11,  5, 11, 10, 11,  0,  3, -1, -1, -1, -1},
  { 5,  4,  8,  5,  8, 10, 10,  8, 11, -1, -1, -1, -1, -1, -1, -1},
  { 9,  7,  8,  5,  7,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 9,  3,  0,  9,  5,  3,  5,  7,  3, -1, -1, -1, -1, -1, -1, -1},
  { 0,  7,  8,  0,  1,  7,  1,  5,  7, -1, -1, -1, -1, -1, -1, -1},
  { 1,  5,  3,  3,  5,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 9,  7,  8,  9,  5,  7, 10,  1,  2, -1, -1, -1, -1, -1, -1, -1},
  {10,  1,  2,  9,  5,  0,  5,  3,  0,  5,  7,  3, -1, -1, -1, -1},
  { 8,  0,  2,  8,  2,  5,  8,  5,  7, 10,  5,  2, -1, -1, -1, -1},
  { 2, 10,  5,  2,  5,  3,  3,  5,  7, -1, -1, -1, -1, -1, -1, -1},
  { 7,  9,  5,  7,  8,  9,  3, 11,  2, -1, -1, -1, -1, -1, -1, -1},
  { 9,  5,  7,  9,  7,  2,  9,  2,  0,  2,  7, 11, -1, -1, -1, -1},
  { 2,  3, 11,  0,  1,  8,  1,  7,  8,  1,  5,  7, -1, -1, -1, -1},
  {11,  2,  1, 11,  1,  7,  7,  1,  5, -1, -1, -1, -1, -1, -1, -1},
  { 9,  5,  8,  8,  5,  7, 10,  1,  3, 10,  3, 11, -1, -1, -1, -1},
  { 5,  7,  0,  5,  0,  9,  7, 11,  0,  1,  0, 10, 11, 10,  0, -1},
  {11, 10,  0, 11,  0,  3, 10,  5,  0,  8,  0,  7,  5,  7,  0, -1},
  {11, 10,  5,  7, 11,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {10,  6,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  3,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 9,  0,  1,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  8,  3,  1,  9,  8,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1},
  { 1,  6,  5,  2,  6,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  6,  5,  1,  2,  6,  3,  0,  8, -1, -1, -1, -1, -1, -1, -1},
  { 9,  6,  5,  9,  0,  6,  0,  2,  6, -1, -1, -1, -1, -1, -1, -1},
  { 5,  9,  8,  5,  8,  2,  5,  2,  6,  3,  2,  8, -1, -1, -1, -1},
  { 2,  3, 11, 10,  6,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11,  0,  8, 11,  2,  0, 10,  6,  5, -1, -1, -1, -1, -1, -1, -1},
  { 0,  1,  9,  2,  3, 11,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1},
  { 5, 10,  6,  1,  9,  2,  9, 11,  2,  9,  8, 11, -1, -1, -1, -1},
  { 6,  3, 11,  6,  5,  3,  5,  1,  3, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8, 11,  0, 11,  5,  0,  5,  1,  5, 11,  6, -1, -1, -1, -1},
  { 3, 11,  6,  0,  3,  6,  0,  6,  5,  0,  5,  9, -1, -1, -1, -1},
  { 6,  5,  9,  6,  9, 11, 11,  9,  8, -1, -1, -1, -1, -1, -1, -1},
  { 5, 10,  6,  4,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  3,  0,  4,  7,  3,  6,  5, 10, -1, -1, -1, -1, -1, -1, -1},
  { 1,  9,  0,  5, 10,  6,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1},
  {10,  6,  5,  1,  9,  7,  1,  7,  3,  7,  9,  4, -1, -1, -1, -1},
  { 6,  1,  2,  6,  5,  1,  4,  7,  8, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2,  5,  5,  2,  6,  3,  0,  4,  3,  4,  7, -1, -1, -1, -1},
  { 8,  4,  7,  9,  0,  5,  0,  6,  5,  0,  2,  6, -1, -1, -1, -1},
  { 7,  3,  9,  7,  9,  4,  3,  2,  9,  5,  9,  6,  2,  6,  9, -1},
  { 3, 11,  2,  7,  8,  4, 10,  6,  5, -1, -1, -1, -1, -1, -1, -1},
  { 5, 10,  6,  4,  7,  2,  4,  2,  0,  2,  7, 11, -1, -1, -1, -1},
  { 0,  1,  9,  4,  7,  8,  2,  3, 11,  5, 10,  6, -1, -1, -1, -1},
  { 9,  2,  1,  9, 11,  2,  9,  4, 11,  7, 11,  4,  5, 10,  6, -1},
  { 8,  4,  7,  3, 11,  5,  3,  5,  1,  5, 11,  6, -1, -1, -1, -1},
  { 5,  1, 11,  5, 11,  6,  1,  0, 11,  7, 11,  4,  0,  4, 11, -1},
  { 0,  5,  9,  0,  6,  5,  0,  3,  6, 11,  6,  3,  8,  4,  7, -1},
  { 6,  5,  9,  6,  9, 11,  4,  7,  9,  7, 11,  9, -1, -1, -1, -1},
  {10,  4,  9,  6,  4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4, 10,  6,  4,  9, 10,  0,  8,  3, -1, -1, -1, -1, -1, -1, -1},
  {10,  0,  1, 10,  6,  0,  6,  4,  0, -1, -1, -1, -1, -1, -1, -1},
  { 8,  3,  1,  8,  1,  6,  8,  6,  4,  6,  1, 10, -1, -1, -1, -1},
  { 1,  4,  9,  1,  2,  4,  2,  6,  4, -1, -1, -1, -1, -1, -1, -1},
  { 3,  0,  8,  1,  2,  9,  2,  4,  9,  2,  6,  4, -1, -1, -1, -1},
  { 0,  2,  4,  4,  2,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 8,  3,  2,  8,  2,  4,  4,  2,  6, -1, -1, -1, -1, -1, -1, -1},
  {10,  4,  9, 10,  6,  4, 11,  2,  3, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  2,  2,  8, 11,  4,  9, 10,  4, 10,  6, -1, -1, -1, -1},
  { 3, 11,  2,  0,  1,  6,  0,  6,  4,  6,  1, 10, -1, -1, -1, -1},
  { 6,  4,  1,  6,  1, 10,  4,  8,  1,  2,  1, 11,  8, 11,  1, -1},
  { 9,  6,  4,  9,  3,  6,  9,  1,  3, 11,  6,  3, -1, -1, -1, -1},
  { 8, 11,  1,  8,  1,  0, 11,  6,  1,  9,  1,  4,  6,  4,  1, -1},
  { 3, 11,  6,  3,  6,  0,  0,  6,  4, -1, -1, -1, -1, -1, -1, -1},
  { 6,  4,  8, 11,  6,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 7, 10,  6,  7,  8, 10,  8,  9, 10, -1, -1, -1, -1, -1, -1, -1},
  { 0,  7,  3,  0, 10,  7,  0,  9, 10,  6,  7, 10, -1, -1, -1, -1},
  {10,  6,  7,  1, 10,  7,  1,  7,  8,  1,  8,  0, -1, -1, -1, -1},
  {10,  6,  7, 10,  7,  1,  1,  7,  3, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2,  6,  1,  6,  8,  1,  8,  9,  8,  6,  7, -1, -1, -1, -1},
  { 2,  6,  9,  2,  9,  1,  6,  7,  9,  0,  9,  3,  7,  3,  9, -1},
  { 7,  8,  0,  7,  0,  6,  6,  0,  2, -1, -1, -1, -1, -1, -1, -1},
  { 7,  3,  2,  6,  7,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 2,  3, 11, 10,  6,  8, 10,  8,  9,  8,  6,  7, -1, -1, -1, -1},
  { 2,  0,  7,  2,  7, 11,  0,  9,  7,  6,  7, 10,  9, 10,  7, -1},
  { 1,  8,  0,  1,  7,  8,  1, 10,  7,  6,  7, 10,  2,  3, 11, -1},
  {11,  2,  1, 11,  1,  7, 10,  6,  1,  6,  7,  1, -1, -1, -1, -1},
  { 8,  9,  6,  8,  6,  7,  9,  1,  6, 11,  6,  3,  1,  3,  6, -1},
  { 0,  9,  1, 11,  6,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 7,  8,  0,  7,  0,  6,  3, 11,  0, 11,  6,  0, -1, -1, -1, -1},
  { 7, 11,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 7,  6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 3,  0,  8, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  1,  9, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 8,  1,  9,  8,  3,  1, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1},
  {10,  1,  2,  6, 11,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 10,  3,  0,  8,  6, 11,  7, -1, -1, -1, -1, -1, -1, -1},
  { 2,  9,  0,  2, 10,  9,  6, 11,  7, -1, -1, -1, -1, -1, -1, -1},
  { 6, 11,  7,  2, 10,  3, 10,  8,  3, 10,  9,  8, -1, -1, -1, -1},
  { 7,  2,  3,  6,  2,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 7,  0,  8,  7,  6,  0,  6,  2,  0, -1, -1, -1, -1, -1, -1, -1},
  { 2,  7,  6,  2,  3,  7,  0,  1,  9, -1, -1, -1, -1, -1, -1, -1},
  { 1,  6,  2,  1,  8,  6,  1,  9,  8,  8,  7,  6, -1, -1, -1, -1},
  {10,  7,  6, 10,  1,  7,  1,  3,  7, -1, -1, -1, -1, -1, -1, -1},
  {10,  7,  6,  1,  7, 10,  1,  8,  7,  1,  0,  8, -1, -1, -1, -1},
  { 0,  3,  7,  0,  7, 10,  0, 10,  9,  6, 10,  7, -1, -1, -1, -1},
  { 7,  6, 10,  7, 10,  8,  8, 10,  9, -1, -1, -1, -1, -1, -1, -1},
  { 6,  8,  4, 11,  8,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 3,  6, 11,  3,  0,  6,  0,  4,  6, -1, -1, -1, -1, -1, -1, -1},
  { 8,  6, 11,  8,  4,  6,  9,  0,  1, -1, -1, -1, -1, -1, -1, -1},
  { 9,  4,  6,  9,  6,  3,  9,  3,  1, 11,  3,  6, -1, -1, -1, -1},
  { 6,  8,  4,  6, 11,  8,  2, 10,  1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 10,  3,  0, 11,  0,  6, 11,  0,  4,  6, -1, -1, -1, -1},
  { 4, 11,  8,  4,  6, 11,  0,  2,  9,  2, 10,  9, -1, -1, -1, -1},
  {10,  9,  3, 10,  3,  2,  9,  4,  3, 11,  3,  6,  4,  6,  3, -1},
  { 8,  2,  3,  8,  4,  2,  4,  6,  2, -1, -1, -1, -1, -1, -1, -1},
  { 0,  4,  2,  4,  6,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  9,  0,  2,  3,  4,  2,  4,  6,  4,  3,  8, -1, -1, -1, -1},
  { 1,  9,  4,  1,  4,  2,  2,  4,  6, -1, -1, -1, -1, -1, -1, -1},
  { 8,  1,  3,  8,  6,  1,  8,  4,  6,  6, 10,  1, -1, -1, -1, -1},
  {10,  1,  0, 10,  0,  6,  6,  0,  4, -1, -1, -1, -1, -1, -1, -1},
  { 4,  6,  3,  4,  3,  8,  6, 10,  3,  0,  3,  9, 10,  9,  3, -1},
  {10,  9,  4,  6, 10,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  9,  5,  7,  6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  3,  4,  9,  5, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1},
  { 5,  0,  1,  5,  4,  0,  7,  6, 11, -1, -1, -1, -1, -1, -1, -1},
  {11,  7,  6,  8,  3,  4,  3,  5,  4,  3,  1,  5, -1, -1, -1, -1},
  { 9,  5,  4, 10,  1,  2,  7,  6, 11, -1, -1, -1, -1, -1, -1, -1},
  { 6, 11,  7,  1,  2, 10,  0,  8,  3,  4,  9,  5, -1, -1, -1, -1},
  { 7,  6, 11,  5,  4, 10,  4,  2, 10,  4,  0,  2, -1, -1, -1, -1},
  { 3,  4,  8,  3,  5,  4,  3,  2,  5, 10,  5,  2, 11,  7,  6, -1},
  { 7,  2,  3,  7,  6,  2,  5,  4,  9, -1, -1, -1, -1, -1, -1, -1},
  { 9,  5,  4,  0,  8,  6,  0,  6,  2,  6,  8,  7, -1, -1, -1, -1},
  { 3,  6,  2,  3,  7,  6,  1,  5,  0,  5,  4,  0, -1, -1, -1, -1},
  { 6,  2,  8,  6,  8,  7,  2,  1,  8,  4,  8,  5,  1,  5,  8, -1},
  { 9,  5,  4, 10,  1,  6,  1,  7,  6,  1,  3,  7, -1, -1, -1, -1},
  { 1,  6, 10,  1,  7,  6,  1,  0,  7,  8,  7,  0,  9,  5,  4, -1},
  { 4,  0, 10,  4, 10,  5,  0,  3, 10,  6, 10,  7,  3,  7, 10, -1},
  { 7,  6, 10,  7, 10,  8,  5,  4, 10,  4,  8, 10, -1, -1, -1, -1},
  { 6,  9,  5,  6, 11,  9, 11,  8,  9, -1, -1, -1, -1, -1, -1, -1},
  { 3,  6, 11,  0,  6,  3,  0,  5,  6,  0,  9,  5, -1, -1, -1, -1},
  { 0, 11,  8,  0,  5, 11,  0,  1,  5,  5,  6, 11, -1, -1, -1, -1},
  { 6, 11,  3,  6,  3,  5,  5,  3,  1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 10,  9,  5, 11,  9, 11,  8, 11,  5,  6, -1, -1, -1, -1},
  { 0, 11,  3,  0,  6, 11,  0,  9,  6,  5,  6,  9,  1,  2, 10, -1},
  {11,  8,  5, 11,  5,  6,  8,  0,  5, 10,  5,  2,  0,  2,  5, -1},
  { 6, 11,  3,  6,  3,  5,  2, 10,  3, 10,  5,  3, -1, -1, -1, -1},
  { 5,  8,  9,  5,  2,  8,  5,  6,  2,  3,  8,  2, -1, -1, -1, -1},
  { 9,  5,  6,  9,  6,  0,  0,  6,  2, -1, -1, -1, -1, -1, -1, -1},
  { 1,  5,  8,  1,  8,  0,  5,  6,  8,  3,  8,  2,  6,  2,  8, -1},
  { 1,  5,  6,  2,  1,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  3,  6,  1,  6, 10,  3,  8,  6,  5,  6,  9,  8,  9,  6, -1},
  {10,  1,  0, 10,  0,  6,  9,  5,  0,  5,  6,  0, -1, -1, -1, -1},
  { 0,  3,  8,  5,  6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {10,  5,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11,  5, 10,  7,  5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11,  5, 10, 11,  7,  5,  8,  3,  0, -1, -1, -1, -1, -1, -1, -1},
  { 5, 11,  7,  5, 10, 11,  1,  9,  0, -1, -1, -1, -1, -1, -1, -1},
  {10,  7,  5, 10, 11,  7,  9,  8,  1,  8,  3,  1, -1, -1, -1, -1},
  {11,  1,  2, 11,  7,  1,  7,  5,  1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  3,  1,  2,  7,  1,  7,  5,  7,  2, 11, -1, -1, -1, -1},
  { 9,  7,  5,  9,  2,  7,  9,  0,  2,  2, 11,  7, -1, -1, -1, -1},
  { 7,  5,  2,  7,  2, 11,  5,  9,  2,  3,  2,  8,  9,  8,  2, -1},
  { 2,  5, 10,  2,  3,  5,  3,  7,  5, -1, -1, -1, -1, -1, -1, -1},
  { 8,  2,  0,  8,  5,  2,  8,  7,  5, 10,  2,  5, -1, -1, -1, -1},
  { 9,  0,  1,  5, 10,  3,  5,  3,  7,  3, 10,  2, -1, -1, -1, -1},
  { 9,  8,  2,  9,  2,  1,  8,  7,  2, 10,  2,  5,  7,  5,  2, -1},
  { 1,  3,  5,  3,  7,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  7,  0,  7,  1,  1,  7,  5, -1, -1, -1, -1, -1, -1, -1},
  { 9,  0,  3,  9,  3,  5,  5,  3,  7, -1, -1, -1, -1, -1, -1, -1},
  { 9,  8,  7,  5,  9,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 5,  8,  4,  5, 10,  8, 10, 11,  8, -1, -1, -1, -1, -1, -1, -1},
  { 5,  0,  4,  5, 11,  0,  5, 10, 11, 11,  3,  0, -1, -1, -1, -1},
  { 0,  1,  9,  8,  4, 10,  8, 10, 11, 10,  4,  5, -1, -1, -1, -1},
  {10, 11,  4, 10,  4,  5, 11,  3,  4,  9,  4,  1,  3,  1,  4, -1},
  { 2,  5,  1,  2,  8,  5,  2, 11,  8,  4,  5,  8, -1, -1, -1, -1},
  { 0,  4, 11,  0, 11,  3,  4,  5, 11,  2, 11,  1,  5,  1, 11, -1},
  { 0,  2,  5,  0,  5,  9,  2, 11,  5,  4,  5,  8, 11,  8,  5, -1},
  { 9,  4,  5,  2, 11,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 2,  5, 10,  3,  5,  2,  3,  4,  5,  3,  8,  4, -1, -1, -1, -1},
  { 5, 10,  2,  5,  2,  4,  4,  2,  0, -1, -1, -1, -1, -1, -1, -1},
  { 3, 10,  2,  3,  5, 10,  3,  8,  5,  4,  5,  8,  0,  1,  9, -1},
  { 5, 10,  2,  5,  2,  4,  1,  9,  2,  9,  4,  2, -1, -1, -1, -1},
  { 8,  4,  5,  8,  5,  3,  3,  5,  1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  4,  5,  1,  0,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 8,  4,  5,  8,  5,  3,  9,  0,  5,  0,  3,  5, -1, -1, -1, -1},
  { 9,  4,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4, 11,  7,  4,  9, 11,  9, 10, 11, -1, -1, -1, -1, -1, -1, -1},
  { 0,  8,  3,  4,  9,  7,  9, 11,  7,  9, 10, 11, -1, -1, -1, -1},
  { 1, 10, 11,  1, 11,  4,  1,  4,  0,  7,  4, 11, -1, -1, -1, -1},
  { 3,  1,  4,  3,  4,  8,  1, 10,  4,  7,  4, 11, 10, 11,  4, -1},
  { 4, 11,  7,  9, 11,  4,  9,  2, 11,  9,  1,  2, -1, -1, -1, -1},
  { 9,  7,  4,  9, 11,  7,  9,  1, 11,  2, 11,  1,  0,  8,  3, -1},
  {11,  7,  4, 11,  4,  2,  2,  4,  0, -1, -1, -1, -1, -1, -1, -1},
  {11,  7,  4, 11,  4,  2,  8,  3,  4,  3,  2,  4, -1, -1, -1, -1},
  { 2,  9, 10,  2,  7,  9,  2,  3,  7,  7,  4,  9, -1, -1, -1, -1},
  { 9, 10,  7,  9,  7,  4, 10,  2,  7,  8,  7,  0,  2,  0,  7, -1},
  { 3,  7, 10,  3, 10,  2,  7,  4, 10,  1, 10,  0,  4,  0, 10, -1},
  { 1, 10,  2,  8,  7,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  9,  1,  4,  1,  7,  7,  1,  3, -1, -1, -1, -1, -1, -1, -1},
  { 4,  9,  1,  4,  1,  7,  0,  8,  1,  8,  7,  1, -1, -1, -1, -1},
  { 4,  0,  3,  7,  4,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 4,  8,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 9, 10,  8, 10, 11,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 3,  0,  9,  3,  9, 11, 11,  9, 10, -1, -1, -1, -1, -1, -1, -1},
  { 0,  1, 10,  0, 10,  8,  8, 10, 11, -1, -1, -1, -1, -1, -1, -1},
  { 3,  1, 10, 11,  3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  2, 11,  1, 11,  9,  9, 11,  8, -1, -1, -1, -1, -1, -1, -1},
  { 3,  0,  9,  3,  9, 11,  1,  2,  9,  2, 11,  9, -1, -1, -1, -1},
  { 0,  2, 11,  8,  0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 3,  2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 2,  3,  8,  2,  8, 10, 10,  8,  9, -1, -1, -1, -1, -1, -1, -1},
  { 9, 10,  2,  0,  9,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 2,  3,  8,  2,  8, 10,  0,  1,  8,  1, 10,  8, -1, -1, -1, -1},
  { 1, 10,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 1,  3,  8,  9,  1,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  9,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  { 0,  3,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}
};

#endif // _LOOKUPTABLE_H_

