#include <torch/csrc/jit/mobile/import.h>
#include <ATen/core/ivalue.h>
#include <caffe2/serialize/inline_container.h>
#include <torch/csrc/jit/api/compilation_unit.h>
#include <torch/csrc/jit/mobile/observer.h>
#include <torch/csrc/jit/runtime/instruction.h>
#include <torch/csrc/jit/serialization/import_export_constants.h>
#include <torch/csrc/jit/serialization/unpickler.h>
#include <torch/custom_class.h>

#include <exception>
#include <fstream>
#include <string>
#include <vector>

// The import process to serialize the bytecode package.
// An example for bytecode.pkl of a small mobile_module looks like:
// (3,
//   ('__torch__.m.forward',
//     (('instructions',
//       (('STOREN', 1, 2),
//        ('DROPR', 1, 0),
//        ('MOVE', 2, 0),
//        ('OP', 0, 0),
//        ('RET', 0, 0))),
//      ('operators', (('aten::Int', 'Tensor'),)),
//      ('constants', ()),
//      ('types', ()),
//      ('register_size', 2))))

// In addition, the module debugging information can be saved
// in mobile_debug.pkl. An example for it looks like:
// (3,
//   ('__torch__.m.forward',
//     (('module_debug_info', (top(A).foo(B).forward)))))

// Note that currently the backward compatibility is not supported by bytecode.
// This format and process need to be revisted and redesigned if we want to
// support backward compatibility in future.

namespace c10 {
// std::string serializeType(const Type &t);
TypePtr parseType(const std::string& pythonStr);
} // namespace c10

namespace torch {
namespace jit {
using caffe2::serialize::IStreamAdapter;
using caffe2::serialize::PyTorchStreamReader;
using caffe2::serialize::ReadAdapterInterface;

OpCode parseOpCode(const char* str);

IValue expect_field(
    IValue tup,
    const std::string& expected_name,
    size_t entry) {
  auto row = tup.toTuple()->elements().at(entry).toTuple();
  TORCH_INTERNAL_ASSERT(
      row->elements().at(0).toStringRef() == expected_name,
      "Expected ",
      expected_name,
      " found ",
      row->elements().at(0).toStringRef());
  return row->elements().at(1);
}

std::string operator_str(
    const std::string& name,
    const std::string& overloadname) {
  std::string result = name;
  if (!overloadname.empty()) {
    result += "." + overloadname;
  }
  return result;
}

namespace {
void print_unsupported_ops_and_throw(
    const std::unordered_set<std::string>& unsupported_ops) {
  std::string error_message("{");
  for (const auto& op_name : unsupported_ops) {
    error_message += op_name + ", ";
  }
  error_message += "}";
  TORCH_CHECK(
      false,
      "Following ops cannot be found. ",
      "May need to add them explicitly to the selective build operator whitelist, ",
      "or re-run the export_opnames to update the whitelist:",
      error_message);
}

void parseMethods(
    const std::vector<IValue>& vals,
    const c10::optional<std::vector<IValue>>& debug_info_vals,
    mobile::CompilationUnit& mcu) {
  TORCH_CHECK(vals.size() > 0, "Bytecode has no elements. ");
  // Initialized with the version number when kProducedBytecodeVersion was
  // introduced. The old models (some of them already in production) without
  // version number don't have to be re-generated.
  int64_t model_version = 0x3L;
  size_t method_i_start = 0;
  if (vals[0].isInt()) {
    model_version = vals[0].toInt();
    method_i_start = 1;
  }
  TORCH_CHECK(
      caffe2::serialize::kMinSupportedBytecodeVersion <= model_version &&
          model_version <= caffe2::serialize::kProducedBytecodeVersion,
      "Lite Interpreter verson number does not match. ",
      "The model version must be between ",
      caffe2::serialize::kMinSupportedBytecodeVersion,
      " and ",
      caffe2::serialize::kProducedBytecodeVersion,
      "But the model version is ",
      model_version);

  bool has_debug_info = debug_info_vals.has_value();
  if (has_debug_info) {
    TORCH_CHECK(
        debug_info_vals->size() == vals.size(),
        "The numbers of bytecode values and debug info values do not match.");
  }

  for (size_t i = method_i_start; i < vals.size(); ++i) {
    const auto& element = vals[i];
    const auto& m_tuple = element.toTuple()->elements();
    const std::string& function_name = m_tuple[0].toStringRef();
    IValue table = m_tuple[1];

    auto function = std::unique_ptr<mobile::Function>(
        new mobile::Function(c10::QualifiedName(function_name)));

    const auto& ins_list =
        expect_field(table, "instructions", BYTECODE_INDEX_INSTRUCTION)
            .toTuple()
            ->elements();
    const auto& ops_list =
        expect_field(table, "operators", BYTECODE_INDEX_OPERATOR)
            .toTuple()
            ->elements();
    const auto& consts_list =
        expect_field(table, "constants", BYTECODE_INDEX_CONSTANT)
            .toTuple()
            ->elements();
    const auto& types_list =
        expect_field(table, "types", BYTECODE_INDEX_TYPE).toTuple()->elements();
    const auto& register_size =
        expect_field(table, "register_size", BYTECODE_INDEX_REGISTER_SIZE)
            .toInt();

    std::vector<IValue> module_debug_info_list;
    if (has_debug_info) {
      const auto& debug_info_element = (*debug_info_vals)[i];
      const auto& debug_info_m_tuple = debug_info_element.toTuple()->elements();
      const std::string& debug_info_function_name =
          debug_info_m_tuple[0].toStringRef();
      TORCH_CHECK(
          debug_info_function_name == function_name,
          "The function names in the bytecode table and the debug info table do not match.");
      IValue debug_info_table = debug_info_m_tuple[1];
      module_debug_info_list = expect_field(
                                   debug_info_table,
                                   "module_debug_info",
                                   BYTECODE_INDEX_MODULE_DEBUG_INFO)
                                   .toTuple()
                                   ->elements();
      TORCH_CHECK(
          module_debug_info_list.size() == ops_list.size(),
          "The numbers of operators and module info strings do not match.");
    }

    function->set_module_debug_info_list_size(ins_list.size());
    for (size_t i = 0; i < ins_list.size(); ++i) {
      auto ins_item = ins_list[i].toTuple()->elements();
      TORCH_CHECK(
          ins_item.size() == 3,
          "There should be three parts in an instruction. The function name is ",
          function_name);
      OpCode op_code = parseOpCode(ins_item[0].toString()->string().c_str());
      int X = ins_item[1].toInt();
      int N = ins_item[2].toInt();
      function->append_instruction(op_code, X, N);
      if (op_code == OP) {
        std::string module_debug_info = (has_debug_info)
            ? module_debug_info_list[X].toString()->string()
            : "";
        function->set_module_info(module_debug_info, i);
      }
    }

    std::unordered_set<std::string> unsupported_op_names;
    for (const auto& op : ops_list) {
      auto op_item = op.toTuple()->elements();
      TORCH_CHECK(
          op_item.size() == 2,
          "There should be two parts in an operator name.");
      auto op_found = function->append_operator(
          op_item[0].toString()->string(),
          op_item[1].toString()->string(),
          model_version);
      if (!op_found) {
        unsupported_op_names.emplace(operator_str(
            op_item[0].toString()->string(), op_item[1].toString()->string()));
      }
    }
    if (!unsupported_op_names.empty()) {
      print_unsupported_ops_and_throw(unsupported_op_names);
    };

    for (const auto& constant : consts_list) {
      function->append_constant(constant);
    }

    for (const auto& t : types_list) {
      function->append_type(c10::parseType(t.toStringRef()));
    }

    function->set_register_size(register_size);

    mcu.register_function(std::move(function));
  }
}

// The deserializer class which loads the bytecode package from bc files.
class BytecodeDeserializer final {
 public:
  explicit BytecodeDeserializer(std::unique_ptr<PyTorchStreamReader> reader);
  mobile::Module deserialize(c10::optional<at::Device> device);
  std::unordered_map<std::string, std::string> deserializeMetadata(
      c10::optional<at::Device> device);

 private:
  c10::IValue readArchive(
      const std::string& archive_name,
      std::shared_ptr<mobile::CompilationUnit> mcu);
  std::unordered_map<std::string, std::string> readMobileMetadata(
      std::shared_ptr<mobile::CompilationUnit> mcu);
  std::shared_ptr<CompilationUnit> compilation_unit_;
  std::unordered_set<std::string> imported_libs_;
  std::unique_ptr<PyTorchStreamReader> reader_;
  c10::optional<at::Device> device_;
};

BytecodeDeserializer::BytecodeDeserializer(
    std::unique_ptr<PyTorchStreamReader> reader)
    : compilation_unit_(std::make_shared<CompilationUnit>()),
      reader_(std::move(reader)) {}

std::unordered_map<std::string, std::string> BytecodeDeserializer::
    deserializeMetadata(c10::optional<at::Device> device) {
  device_ = device;
  auto mcu = std::make_shared<mobile::CompilationUnit>();
  return readMobileMetadata(mcu);
}

mobile::Module BytecodeDeserializer::deserialize(
    c10::optional<at::Device> device) {
  device_ = device;
  auto mcu = std::make_shared<mobile::CompilationUnit>();
  auto bvals = readArchive("bytecode", mcu).toTuple()->elements();

  c10::optional<std::vector<IValue>> debug_info_bvals;
  if (reader_->hasRecord("mobile_debug.pkl")) {
    debug_info_bvals = readArchive("mobile_debug", mcu).toTuple()->elements();
  }
  parseMethods(bvals, debug_info_bvals, *mcu);
  auto meta_dict = readMobileMetadata(mcu);
  return mobile::Module(readArchive("data", mcu).toObject(), meta_dict, mcu);
}

std::unordered_map<std::string, std::string> BytecodeDeserializer::
    readMobileMetadata(std::shared_ptr<mobile::CompilationUnit> mcu) {
  std::unordered_map<std::string, std::string> res;
  if (!reader_->hasRecord("metadata.pkl")) {
    return res;
  }
  auto ivalue_dict = readArchive("metadata", mcu).toGenericDict();
  for (auto it = ivalue_dict.begin(); it != ivalue_dict.end(); ++it) {
    auto key = it->key().toString()->string();
    auto value = it->value().toString()->string();
    res[key] = value;
  }
  return res;
}

c10::IValue BytecodeDeserializer::readArchive(
    const std::string& archive_name,
    std::shared_ptr<mobile::CompilationUnit> mcu) {
  std::stringstream picklename;
  picklename << archive_name << ".pkl";
  at::DataPtr pickle_ptr;
  size_t pickle_size;
  std::tie(pickle_ptr, pickle_size) = reader_->getRecord(picklename.str());

  size_t bytes_read = 0;
  auto data = reinterpret_cast<const char*>(pickle_ptr.get());
  auto reader = [&](char* buffer, size_t len) -> size_t {
    if (bytes_read >= pickle_size) {
      return 0;
    }
    len = std::min(pickle_size - bytes_read, len);
    // Copy len bytes into buffer
    const char* start = data + bytes_read;
    std::memcpy(buffer, start, len);
    bytes_read += len;
    return len;
  };

  static const c10::QualifiedName torchPrefix = "__torch__";
  auto type_resolver = [&](const c10::QualifiedName& qn) {
    TypePtr type;
    // HACK: first we check whether the name starts with `__torch__` to tell if
    // it's "supposed" to be a class type. This is a reliable check today, but
    // there is no guarantee that this is the case. The real solution is to
    // merge type parsers so we can share class resolution logic.
    if (torchPrefix.isPrefixOf(qn)) {
      if (compilation_unit_->get_class(qn) == nullptr) {
        auto typeptr = ClassType::create(qn, compilation_unit_, true);
        compilation_unit_->register_type(typeptr);
      }
      type = compilation_unit_->get_class(qn);
    } else {
      type = c10::parseType(qn.qualifiedName());
    }
    return c10::StrongTypePtr(compilation_unit_, type);
  };

  auto obj_loader = [&](at::StrongTypePtr type, IValue input) {
    auto cls = type.type_->expect<at::ClassType>();
    auto qn = cls->name();
    c10::QualifiedName method_name(qn.value(), "__setstate__");
    auto setstate = mcu->find_function(method_name);
    auto find_custom_class_with_setstate = [&qn]() -> c10::ClassTypePtr {
      auto custom_class_type = torch::jit::getCustomClass(qn->qualifiedName());
      if (custom_class_type && custom_class_type->findMethod("__setstate__")) {
        return custom_class_type;
      }
      return nullptr;
    };
    if (setstate) {
      auto obj = c10::ivalue::Object::create(type, 0);
      Stack stack({obj, input});
      setstate->run(stack);
      return obj;
    } else if (auto custom_class_type = find_custom_class_with_setstate()) {
      auto obj = c10::ivalue::Object::create(
          c10::StrongTypePtr(nullptr, custom_class_type), 1);
      Stack stack({obj, input});
      custom_class_type->getMethod("__setstate__").run(stack);
      return obj;
    } else {
      auto dict = std::move(input).toGenericDict();
      size_t ndict = dict.size();
      auto obj = c10::ivalue::Object::create(type, ndict);
      auto it = dict.begin();
      for (size_t i = 0; i < ndict; ++i) {
        std::stringstream name;
        name << it->key();
        cls->addOrCheckAttribute(name.str(), it->key().type());
        obj->setSlot(i, it->value());
        ++it;
      }
      return obj;
    }
  };

  auto read_record = [&](const std::string& name) {
    std::stringstream ss;
    ss << archive_name << "/" << name;
    return std::get<0>(reader_->getRecord(ss.str()));
  };

  Unpickler unpickler(
      reader,
      std::move(type_resolver),
      std::move(obj_loader),
      std::move(read_record),
      device_);
  return unpickler.parse_ivalue();
}

} // namespace

mobile::Module _load_for_mobile(
    std::istream& in,
    c10::optional<at::Device> device) {
  std::unique_ptr<IStreamAdapter> rai = std::make_unique<IStreamAdapter>(&in);
  auto module = _load_for_mobile(std::move(rai), device);
  return module;
}

mobile::Module _load_for_mobile(
    const std::string& filename,
    c10::optional<at::Device> device) {
  std::unique_ptr<FileAdapter> rai = std::make_unique<FileAdapter>(filename);
  auto module = _load_for_mobile(std::move(rai), device);
  return module;
}

mobile::Module _load_for_mobile(
    std::unique_ptr<ReadAdapterInterface> rai,
    c10::optional<c10::Device> device) {
  auto observer = torch::observerConfig().getModuleObserver();
  if (observer) {
    observer->onEnterLoadModel();
  }
  auto reader = torch::make_unique<PyTorchStreamReader>(std::move(rai));
  BytecodeDeserializer deserializer(std::move(reader));
  try {
    mobile::Module result = deserializer.deserialize(std::move(device));
    std::unordered_map<std::string, std::string> copied_metadata =
        result.metadata();
    if (result.metadata().find("model_name") == result.metadata().end()) {
      copied_metadata["model_name"] = result.name();
    }
    if (observer) {
      observer->onExitLoadModel(copied_metadata);
    }
    return result;
  } catch (c10::Error& error) {
    if (observer) {
      observer->onFailLoadModel(
          error.what(), deserializer.deserializeMetadata(std::move(device)));
    }
    TORCH_RETHROW(error);
  } catch (...) {
    auto currentException = std::current_exception();
    try {
      if (!currentException) {
        TORCH_CHECK(false, "Unknown exception");
      } else {
        try {
          std::rethrow_exception(currentException);
        } catch (const std::exception& e) {
          TORCH_CHECK(false, e.what());
        }
      }
    } catch (c10::Error& error) {
      if (observer) {
        observer->onFailLoadModel(
            error.what(), deserializer.deserializeMetadata(std::move(device)));
      }
      TORCH_RETHROW(error);
    }
  }
}

} // namespace jit
} // namespace torch
