/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

#ifndef __DRONE
#define __DRONE

#include "fader.h"
#include "beat2value.h"
#include "solver.h"
#include "play.h"

struct mod_params {

  float depth;
  float bpm;
	
  float b, db; // beat, delta on beat2value curve -> see calc (..)

  float result;

  mod_params (beat2value& bv) {
    int reset = 1;
    clear (bv, reset);
  }

  void set_bpm (beat2value& bv, float value, int reset = 0) {

    bv.set_bpm (value);
    bpm = bv.bpm;

    extern audio_out aout;
    db = bv.delta * aout.samples_per_channel; // modulate once every audio buffer

    if (reset) b = bv.sol.firstx;

  }

  void calc (beat2value& bv) {
    result = depth * bv.sol (b, db);
  }

  void clear (beat2value& bv, int reset = 0) {
    depth = 0;
    set_bpm (bv, 30, reset);
  }

};

struct drone_modulation {
	
	// am, fm modulation curves - shared by all drones
	// x - beat, y - modulation level
  static beat2value am_crv, fm_crv; 
  
  enum {FM = 1, AM};
  mod_params am, fm;
  
  int active;

  drone_modulation () : am (am_crv), fm (fm_crv) {
    active = 0;
  }

  void clear () {
    am.clear (am_crv);
    fm.clear (fm_crv);
    active = 0;
  }

  void calc () {
    am.calc (am_crv);
    fm.calc (fm_crv);
  }

  void calc_active () {
    active = (((am.depth == 0) || (am.bpm == 0)) && ((fm.depth == 0) || (fm.bpm == 0))) ? 0:1;
  }

};

struct din;
struct drone {

  play player; // player for this drone
  solver sol; // solver for this drone

  float step; // determines pitch (see note.h)
  float vol; // actual volume
	
  int range; // range number on microtonal-keyboard
  float pos; // position in range (0 - left, 1 - right, can go < 0 & > 1 too)

  int sel; // selected?

  // visual
  //
  int x, y; // current position
  int dy; // height from microtonal-keyboard bottom
  box<int> handle; // for picking
  float r, g, b; // color

  int cx, cy; // position on creation / unmodulated position
	
	// drone trails
  std::list< point<int> > trailq;
  std::list< point<int> >::size_type num_trail_points;
  
  int handle_size;

  enum {DEAD, ACTIVE, RISING, FALLING}; // states
  int state; // state
  fader fdr; // for rising/falling -> fade in/out color

  drone_modulation mod; // modulation parameters

  drone () {
    step = vol = 0;
    range = pos = 0;
    r = g = b = 0.5;
    x = y = dy = 0;
    sel = 0;
    state = RISING;
    
    extern int DRONE_HANDLE_SIZE, TRAIL_LENGTH;
    handle_size = DRONE_HANDLE_SIZE;
    num_trail_points = TRAIL_LENGTH;

  }

  void calc_handle () {
    extern int BOTTOM;
    y = BOTTOM + dy;
    handle (x - handle_size, y - handle_size, x + handle_size, y + handle_size);
  }
	
  void set_xy (din& board, const int& x, const int& y);

  void change_bpm (mod_params& mp, beat2value& bv, float delta) {
    float bpm = mp.bpm + delta;
    mp.set_bpm (bv, bpm);
  }

  void change_bpm (int i, int what, float delta) {
    if (what == drone_modulation::FM) {
      change_bpm (mod.fm, drone_modulation::fm_crv, delta);
      cons << "drone: " << i << ", FM bpm = " << mod.fm.bpm << eol;
    } else {
      change_bpm (mod.am, drone_modulation::am_crv, delta);
      cons << "drone: " << i << ", AM bpm = " << mod.am.bpm << eol;
    }
    mod.calc_active ();
  }

  void change_depth (int i, int what, float delta) {
    if (what == drone_modulation::FM) {
      mod.fm.depth += delta;
      cons << "drone: " << i << ", FM depth = " << mod.fm.depth << eol;
    } else {
      mod.am.depth += delta;
      cons << "drone: " << i << ", AM depth = " << mod.am.depth << eol;
    }
    mod.calc_active ();
  }

};

#endif
